% function T = generate_T(TO, FROM, ROBOT)
%
% generates the transformation matrix that transforms coordinates
% in referance frame TO given coordinates in reference frame FROM.
%
% TO and FROM should be integers between 0 and 6 for a 6DoF arm.
%
% This function is for generic use. In specific cases, a more
% specific function will probably be faster.
%
% ROBOT is a matrix such that 
% ROBOT = [al_im1, a_im1, d_i, theta_i]
%
% Where
%
% col vector al_im1 contains the alpha-values, indexed from 0 (the
%    notation 'im1' meaning 'i minus 1') 
%
% col vector a_im1 contains the a_values, indexed from 0
%
% col vector d_i contains the d values, indexed from 1
%
% col vector theta_i contains the theta values, indexed from 1
%
%
% NB: the Denavit-Hartenberg notation is done according to
% J.J. Craig


function T = generate_T(to, from, robot)

al_im1 = robot(:,1);
a_im1 = robot(:,2);
d_i = robot(:,3);
theta_i = robot(:,4);

if ~(sum(to==(0:6))) | ~(sum(from==(0:6)))
  error(['TO and FROM should be integers between 0 and 6 for a 6DoF' ...
	 ' arm!'])
end

% If both frames are the same, generate identity transform
T = eye(4);

% For two different frames, generate transform
if (to~=from)
  
  % the following variables are cos(alpha_i-1) and sin(alpha_i-1)
  ca=cos(al_im1);
  sa=sin(al_im1);
  
  % generate transform matrices for all relevant steps:
  % (transform matrix generated according to p.84 in "Introduction to
  % robotics" by J.J. Craig) 
  for k = min([to from])+1:max([to from])
    c = cos(theta_i(k));
    s = sin(theta_i(k));
     T_i = [     c       -s       0        a_im1(k);
	    s*ca(k)  c*ca(k)  -sa(k)  -sa(k)*d_i(k);
	    s*sa(k)  c*sa(k)   ca(k)   ca(k)*d_i(k);
	         0        0       0              1];
    T=T*T_i;
  end
  
  % in the more common case, a coordinate in a higher number frame
  % is represented in a lower number frame. This has already been
  % calculated. In the reverse case, the inverse is needed:
  if (to>from)
      T=inv(T);
  end
end