//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// this is a -*- C++ -*- file
//----------------------------------------------------------------------
/*!\file    Dynamics.h
 *
 * \author   Christian Smith
 * \date    02.05.05
 *
 * $Revision: 1.7 $
 * $Id: NullModule.h,v 1.7 2003/02/24 14:36:21 gassmann Exp $
 *
 * \brief   Contains mDynamics
 *
 * \b mDynamics
 *
 * A module that calculates the dynamic forces ond torques for the Powerarm
 *
 */
//----------------------------------------------------------------------

#ifndef _Dynamics_h_
#define _Dynamics_h_
//----------------------------------------------------------------------
// System Includes - include with <>
//----------------------------------------------------------------------
// z.B. #include <stdio.h>
#include <Module.h>
#include <fstream>
//----------------------------------------------------------------------
// Project Includes - include with ""
//----------------------------------------------------------------------
#include "vectorMath.h"
#include "robot_arm_params.h"
//----------------------------------------------------------------------
// Extern methods and global vars
//----------------------------------------------------------------------

//----------------------------------------------------------------------
// Forward class declarations
//----------------------------------------------------------------------

//----------------------------------------------------------------------
// Implementation
//----------------------------------------------------------------------

//! Short description of mDynamics
/*! 
  Thid module is used to calculate the dynamic forces and torques for
  the powerarm. It takes angles, angular velocities and accelerations 
  as input, and gives torques as output. It logs forces.
*/
class mDynamics:public tModule
{
 public:
  /*!
    Anonymous enumeration typ which contains the indices of the
    controller inputs.
  */
  _DESCR_(static,mDynamics::ci_description,4,Natural,cDATA_VECTOR_END_MARKER);
  enum {
    eCI_JOINT_ACC_WAIST, /*!< desired angular acc of waist joint */
    eCI_JOINT_ACC_SHOULDER, /*!< desired angular acc of shoulder joint */
    eCI_JOINT_ACC_ELBOW, /*!< desired angular acc of elbow joint */
    eCI_JOINT_ACC_WRIST_ROT, /*!< desired angular acc of wrist rot joint */
    eCI_JOINT_ACC_WRIST_BEND, /*!< desired angular acc of wrist bend joint */
    eCI_JOINT_ACC_FLANGE, /*!< desired angular acc of flange joint */
    eCI_DIMENSION /*!< Endmarker and Dimension */
  };
  
  /*!
    Anonymous enumeration typ which contains the indices of the
    controller outputs.
  */
  _DESCR_(static,mDynamics::co_description,4,Natural,cDATA_VECTOR_END_MARKER);
  enum {
    eCO_JOINT_ACC_WAIST, /*!< desired angular acc of waist joint */
    eCO_JOINT_ACC_SHOULDER, /*!< desired angular acc of shoulder joint */
    eCO_JOINT_ACC_ELBOW, /*!< desired angular acc of elbow joint */
    eCO_JOINT_ACC_WRIST_ROT, /*!< desired angular acc of wrist rot joint */
    eCO_JOINT_ACC_WRIST_BEND, /*!< desired angular acc of wrist bend joint */
    eCO_JOINT_ACC_FLANGE, /*!< desired angular acc of flange joint */
    eCO_DIMENSION /*!< Endmarker and Dimension */
  };
  
  /*!
    Anonymous enumeration typ which contains the indices of the
    sensor inputs.
  */
  _DESCR_(static,mDynamics::si_description,4,Natural,cDATA_VECTOR_END_MARKER);
  enum {
    
    eSI_WAIST,       /*! Angle of the waist-joint [rad]          */
    eSI_SHOULDER,    /*! Angle of the shoulder-joint [rad]       */
    eSI_ELBOW,       /*! Angle of the elbow-joint [rad]          */
    eSI_WRIST_ROT,   /*! Angle of the wrist_rotation-joint [rad] */
    eSI_WRIST_BEND,  /*! Angle of the wrist_bend-joint [rad]     */
    eSI_FLANGE,      /*! Angle of the flange-joint [rad]         */
    
    eSI_WAIST_VEL,     /*! Angular vel. of the waist-joint [rad/s]         */
    eSI_SHOULDER_VEL,  /*! Angular vel. of the shoulder-joint [rad/s]      */
    eSI_ELBOW_VEL,     /*! Angular vel. of the elbow-joint [rad/s]         */
    eSI_WRIST_ROT_VEL, /*! Angular vel. of the wrist_rotation-joint [rad/s]*/
    eSI_WRIST_BEND_VEL,/*! Angular vel. of the wrist_bend-joint [rad/s]    */
    eSI_FLANGE_VEL,    /*! Angular vel. of the flange-joint [rad/s]        */
    
    eSI_WAIST_ACC,     /*! Angular acc. of the waist-joint [rad/s^2]         */
    eSI_SHOULDER_ACC,  /*! Angular acc. of the shoulder-joint [rad/s^2]      */
    eSI_ELBOW_ACC,     /*! Angular acc. of the elbow-joint [rad/s^2]         */
    eSI_WRIST_ROT_ACC, /*! Angular acc. of the wrist_rotation-joint [rad/s^2]*/
    eSI_WRIST_BEND_ACC,/*! Angular acc. of the wrist_bend-joint [rad/s^2]    */
    eSI_FLANGE_ACC,    /*! Angular acc. of the flange-joint [rad/s^2]        */
    eSI_DIMENSION /*!< Endmarker and Dimension */
  };
  
  /*!
    Anonymous enumeration typ which contains the indices of the
    sensor outputs.
  */
  _DESCR_(static,mDynamics::so_description,4,Natural,cDATA_VECTOR_END_MARKER);
  enum {
    eSO_WAIST_TRQ,       /*! Torque of the waist-joint [Nm]          */
    eSO_SHOULDER_TRQ,    /*! Torque of the shoulder-joint [Nm]       */
    eSO_ELBOW_TRQ,       /*! Torque of the elbow-joint [Nm]          */
    eSO_WRIST_ROT_TRQ,   /*! Torque of the wrist_rotation-joint [Nm] */
    eSO_WRIST_BEND_TRQ,  /*! Torque of the wrist_bend-joint [Nm]     */
    eSO_FLANGE_TRQ,      /*! Torque of the flange-joint [Nm]         */
    eSO_DIMENSION /*!< Endmarker and Dimension */
  };
  
  /*!
    Anonymous enumeration typ which contains the indices of the
    parameters.
  */
  enum {
    ePAR_MAX_TORQUE_WAIST, /*!< maximum torque of waist joint */
    ePAR_MAX_TORQUE_SHOULDER, /*!< maximum torque of shoulder joint */
    ePAR_MAX_TORQUE_ELBOW, /*!< maximum torque of elbow joint */
    ePAR_MAX_TORQUE_WRIST_ROT, /*!< maximum torque of wrist rot joint */
    ePAR_MAX_TORQUE_WRIST_BEND, /*!< maximum torque of wrist bend joint */
    ePAR_MAX_TORQUE_FLANGE, /*!< maximum torque of flange joint */
    ePAR_DIMENSION /*!< Endmarker and Dimension */
  };
  
  /*! \param parent the parent
   *  \param name The module name
   *  \param fixit whether to use FixIt() or not
   */
  mDynamics(tParent *parent,
	    tDescription name="Dynamics",bool fixit=true);
  /*!
   */
  ~mDynamics();
  /*!
   */
  void Control();
  /*!
   */
  void Sense();
  /*!
   */
  void Exception(tExceptionType type);
  
  VM_3matrix RotationMatrix(int from, int to);
  VM_3matrix RotationMatrix2(int from, int to);
  VM_3vector TranslationVector(int from, int to);
  VM_3vector TranslationVector2(int from, int to);
  VM_3vector MassCenter(int frame);
  double Mass(int frame);
  VM_3matrix InertiaTensor(int frame);
  void InitializePositions(void);
 
  // The following use 7-element arrays in order to acieve
  // consistent indexing. The links are indexed 1-6 (with 0
  // being the stationary, "world", frame)
  VM_7vector InverseDynamics(VM_7vector _theta, VM_7vector _theta_p, VM_7vector _theta_pp, VM_3vector _grav, int _position);
  // Calculates torques needed for given angular values,
  // vel. and acc. The vector gives the acceleration that equals 
  // the gravity used. This means that normal gravity would be {0, 0, g}.
  // Note that the direction is up! If this is set to a zero vector,
  // the effects of gravity on dynamics are ignored. The _position 
  // parameter tells the function what links we are interested in.
  // Only the torques on link _position and outwards are calculated.
  // Links inward from position are _reported as zero torque. Note 
  // that the effects of the inward links on the outward ones are not
  // ignored, however. Setting _position to -1 tells the function to
  // write to the log file, if this is enabled.

  

  VM_7vector ForwardDynamics(VM_7vector _theta, VM_7vector _theta_p, VM_7vector _torque7);
  // Calculates accelerations acheived with given angular
  // values, vels, and torques:

  
 private:  
  // Variables to store sense input for more convenient access.
  // note that joints are numbered 1-6 (index 0 is reserved for
  // the first, stationary, link)
  VM_7vector theta;
  VM_7vector theta_p;
  VM_7vector theta_pp;
  VM_7vector theta_ctrl;
  VM_7vector theta_p_ctrl;
  VM_7vector theta_pp_ctrl;
  
  // variables used for calculations:
  VM_3vector v[7];
  VM_3vector w[7];
  VM_3vector vp[7];
  VM_3vector wp[7];
  VM_3vector vcp[7];
  double m[7];
  VM_3vector F[7];
  VM_3vector N[7];
  VM_3vector n[7];
  VM_3vector f[7];

  double PA_rb_mass_a[7];
  double PA_rb_mass_b[7];
  double PA_rb_size[7];

  double PA_dh_alpha[7];
  double PA_dh_a[7];
  double PA_dh_d[7];

  FILE *dyn_logfile;
  FILE *kin_logfile;
  int dyn_logfile_opened;
  int kin_logfile_opened;

  int DYN_sensor_changed;
  double max_torque[6];

  tTime D_sense_timer;

  /* positions of (center of) cubes in frame that cube is stationary in [mm] */
  VM_3vector PA_rb_pos1[8];
  VM_3vector PA_rb_pos2[7];
};

#endif
