// this is a -*- C++ -*- file
//----------------------------------------------------------------------
/*!\file    Kinematic.C
 *
 * written by Christian Smith Apr 29 2005
 * updated by Christian Smith May 02 2005
 *
 * This code is basically a rewrite of code by the same name,
 * with the original credits as follows:
 *
 * \author   Kay-Ulrich Scholl
 * \date    06.11.02
 *
 * $Revision: 1.1 $
 * $Id: Kinematic.C,v 1.1 2002/11/06 16:08:54 scholl Exp $
 *
 */
//----------------------------------------------------------------------
// Includes
//----------------------------------------------------------------------

#include "Kinematic.h"
#include "robot_arm_params.h"

//----------------------------------------------------------------------
// Module Debug
//----------------------------------------------------------------------
//#undef MODULE_DEBUG
#define MODULE_DEBUG

// This automatically chooses the 'flip' parameter for minimum movement
#define K_AUTOFLIP   

//#define KIN_TIMING_TEST
//#define Christian_debug1
//#define Christian_debug2
//#define Christian_debug3
#include <module_debug.h>

//----------------------------------------------------------------------
// defines and consts
//----------------------------------------------------------------------

//----------------------------------------------------------------------
// class Kinematic constructor
//----------------------------------------------------------------------
tKinematic::tKinematic(tParent *parent,
			   tDescription name,bool fixit)
  :tModule(parent,name,
	       eSI_DIMENSION,eSO_DIMENSION,eCI_DIMENSION,eCO_DIMENSION,
	       si_description,so_description,ci_description,co_description,
	       fixit)
{
  InitParameter(ePAR_DIMENSION,
		new tParameterFloat("Parameter A2[mm]", -500.0, 500.0, PA_DH_A2),
		new tParameterFloat("Parameter D2[mm]", -300.0, 300.0, PA_DH_D2),
		new tParameterFloat("Parameter A3[mm]", -50.0, 50.0,  PA_DH_A3),
		new tParameterFloat("Parameter D4[mm]", -500.0, 500.0, PA_DH_D4),
		new tParameterFloat("Parameter D6[mm]", -150.0, 150.0, PA_DH_TOOL),
		new tParameterFloat("Parameter D3[mm]", -100.0, 100.0, PA_DH_D3),

		new tParameterFloat("Waist Min [rad]", -2*PI, 2*PI, -2*PI),
		new tParameterFloat("Waist Max [rad]", -2*PI, 2*PI, 2*PI),
		new tParameterFloat("Shoulder Min [rad]", -2*PI, 2*PI, -2*PI),
		new tParameterFloat("Shoulder Max [rad]", -2*PI, 2*PI, 2*PI),
		new tParameterFloat("Elbow Min [rad]", -2*PI, 2*PI, -2*PI),
		new tParameterFloat("Elbow Max [rad]", -2*PI, 2*PI, 2*PI),
		new tParameterFloat("Wrist Rotation Min [rad]", -2*PI, 2*PI, -2*PI),
		new tParameterFloat("Wrist Rotation Max [rad]", -2*PI, 2*PI, 2*PI),
		new tParameterFloat("Wrist Bend Min [rad]", -2*PI, 2*PI, -2*PI),
		new tParameterFloat("Wrist Bend Max [rad]", -2*PI, 2*PI, 2*PI),
		new tParameterFloat("Flange Min [rad]", -2*PI, 2*PI, -2*PI),
		new tParameterFloat("Flange Max [rad]", -2*PI, 2*PI, 2*PI));


  for (int i=0; i<6; d_temp[i++]=0.0);
}

//----------------------------------------------------------------------
// class Kinematic destructor
//----------------------------------------------------------------------
tKinematic::~tKinematic()
{
}

//----------------------------------------------------------------------
// class Kinematic Control
//----------------------------------------------------------------------
void  tKinematic::Control()
{
  // temps for the ControllerInput
  long double tool_x;
  long double tool_y;
  long double tool_z;
  long double tool_o;
  long double tool_a;
  long double tool_t;
  long double arm;
  long double elbow;
  long double wrist;
  long double flip;

  // temps for ControllerOutput (the six angles)
  long double d[6];
  // if some error occurs we can fall back

#ifdef KIN_TIMING_TEST
    K_sense_timer.FromNow();
#endif

  if (ControllerInputChanged())
    {
      long double old_overflow=overflow;

      overflow=0.0;
      reflect=false;
      tool_x=ControllerInput(eCI_TOOL_X);
      tool_y=ControllerInput(eCI_TOOL_Y);
      tool_z=ControllerInput(eCI_TOOL_Z);
      tool_o=ControllerInput(eCI_TOOL_O);
      tool_a=ControllerInput(eCI_TOOL_A);
      tool_t=ControllerInput(eCI_TOOL_T);
      arm=Sgn(ControllerInput(eCI_ARM));
      elbow=Sgn(ControllerInput(eCI_ELBOW));
      wrist=Sgn(ControllerInput(eCI_WRIST));
      flip=Sgn(ControllerInput(eCI_FLIP));

      // calculate Values often needed:
      long double so=sin(tool_o);
      long double sa=sin(tool_a);
      long double st=sin(tool_t);
      long double co=cos(tool_o);
      long double ca=cos(tool_a);
      long double ct=cos(tool_t);

      //fprintf(stdout,"tool_o = %f, tool_a = %f, tool_t = %f\n",(double)tool_o, (double)tool_a, (double)tool_t); 


      //create Matrix:
      long double r13=so*ca;
      long double r23=-co*ca;
      long double r33=-sa;
      long double r12=so*sa*st+co*ct;
      long double r22=so*ct-co*sa*st;
      long double r32=ca*st;

      // n calculates aout of the crossproduct s x a
      long double r11 = r22*r33 - r23*r32;
      long double r21 = r32*r13 - r12*r33;
      long double r31 = r12*r23 - r13*r22;


      long double pxt=tool_x;
      long double pyt=tool_y;
      long double pzt=tool_z;

      M_DEBUG("the Controller Matrix\n");
      M_DEBUG("%Lf %Lf %Lf %Lf\n",r11, r12, r13, pxt);
      M_DEBUG("%Lf %Lf %Lf %Lf\n",r21, r22, r23, pyt);
      M_DEBUG("%Lf %Lf %Lf %Lf\n",r31, r32, r33, pzt);
      M_DEBUG("0 0 0 1\n");

#ifdef Christian_debug1
      fprintf(stdout,"the Controller Matrix\n");
      fprintf(stdout,"%Lf %Lf %Lf %Lf\n",r11, r12, r13, pxt);
      fprintf(stdout,"%Lf %Lf %Lf %Lf\n",r21, r22, r23, pyt);
      fprintf(stdout,"%Lf %Lf %Lf %Lf\n",r31, r32, r33, pzt);
      fprintf(stdout,"0 0 0 1\n");
#endif

      // the position - vector of 0T4
      long double px=pxt - (d6*r13);
      long double py=pyt - (d6*r23);
      long double pz=pzt - (d6*r33);
      
      // get first angle
      long double a_1 = atan2(py,px);

      long double tempc0 = arm*sqrt(px*px + py*py - d3*d3);
      long double temps0 = -d3;
      Small(&tempc0);
      Small(&temps0);
      d[0]= a_1 - atan2(temps0,tempc0);
      long double c1 = cos(d[0]);
      long double s1 = sin(d[0]);

      if(d[0]<(waist_min)){
	d[0]+=2*PI;
      }
      
      if(d[0]>(waist_max)){
	d[0]-=2*PI;
      }


      // check if in range and possible
      if (isnan(d[0]) || !InRange(&d[0], &waist_min, &waist_max)) {
	M_DEBUG("WAIST ERROR!!!\n");
#ifdef Christian_debug1
	fprintf(stdout,"WAIST ERROR!!!\n");
#endif
	overflow = 1.0;
      }

      M_DEBUG("d[0]=%Lf\n",d[0]);

      // get the third angle
      long double s3 = (a2*a2 + d3*d3 + d4*d4 - (px*px + py*py + pz*pz))/(2*d4*a2);
      long double c3 = elbow*sqrt(1-(s3*s3));

#ifdef Christian_debug1
      fprintf(stdout,"a2*a2 + d3*d3 + d4*d4=%f\n",(double)(a2*a2 + d3*d3 + d4*d4));
      fprintf(stdout,"(px*px + py*py * pz*pz)=%f\n",(double)(px*px + py*py * pz*pz));
      fprintf(stdout,"(2*d4*a2)=%f\n",(double)(2*d4*a2));

      fprintf(stdout,"s3=%f, c3=%f\n",(double)s3,(double)c3);
#endif

      Small(&s3);
      Small(&c3);
      d[2] = atan2(s3,c3);

      if(d[2]<(elbow_min)){
	d[2]+=2*PI;
      }
      
      if(d[2]>(elbow_max)){
	d[2]-=2*PI;
      }



      // check if in range and possible
      if (isnan(d[2])|| !InRange(&d[2],&elbow_min,&elbow_max)) {
	M_DEBUG("ELBOW ERROR!!!\n");
#ifdef Christian_debug1
	fprintf(stdout,"ELBOW ERROR!!!\n");
#endif
	overflow=3.0;
      }

      M_DEBUG("d[2]=%Lf\n",d[2]);

      // get the second angle
      long double s23 = ((-a2*c3)*pz - (c1*px + s1*py) * (d4-a2*s3))/
                	(pz*pz + (c1*px +s1*py)*(c1*px +s1*py));

      long double c23 = ((a2*s3 - d4)*pz + (a2*c3*(c1*px + s1*py)))/
                	(pz*pz + (c1*px + s1*py)*(c1*px + s1*py));
      Small(&s23);
      Small(&c23);
      long double t23 = atan2(s23,c23);
      
      d[1]=t23-d[2];
      
      if(d[1]<(shoulder_min)){
	d[1]+=2*PI;
      }
      
      if(d[1]>(shoulder_max)){
	d[1]-=2*PI;
      }

      // check if in range and possible
      if (isnan(d[1])|| !InRange(&d[1],&shoulder_min,&shoulder_max)) {
	M_DEBUG("SHOULDER ERROR!!!\n");
#ifdef Christian_debug1
	fprintf(stdout,"SHOULDER ERROR!!!\n");
#endif
	overflow=2.0;
      }

      M_DEBUG("d[1]=%Lf\n",d[1]);

      // get the fourth angle
      
      long double alpha4 = -r13*s1 + r23*c1;
      long double beta4 = -r13*c1*c23 - r23*s1*c23 + r33*s23; 

      // if both alpha4 and beta4 are too small, we are in a singular configuration for
      // the fourth joint. This means that any value will do, so we reuse the old
      // value in order to minimize movement.
#define ANGLE_4_SINGULARITY_LIMIT 0.0000001

      if ( ((alpha4*Sgn(alpha4)) < ANGLE_4_SINGULARITY_LIMIT) && ((beta4*Sgn(beta4)) < ANGLE_4_SINGULARITY_LIMIT)){
	d[3] = SensorInput(3);
      }
      else{
	d[3] = atan2(alpha4,beta4);
      }
      long double s4 = sin(d[3]);
      long double c4 = cos(d[3]);

      if(d[3]<(wrist_rot_min)){
	d[3]+=2*PI;
      }
      
      if(d[3]>(wrist_rot_max)){
	d[3]-=2*PI;
      }
      
      // check if in range and possible
      if (isnan(d[3])|| !InRange(&d[3],&wrist_rot_min,&wrist_rot_max)) {
	M_DEBUG("WRIST_ROT ERROR!!!\n");
#ifdef Christian_debug1
	fprintf(stdout,"WRIST_ROT ERROR!!!\n");
#endif
	overflow=4.0;
      }

      M_DEBUG("d[3]=%Lf\n",d[3]);

      // get the fifth angle
      long double s5 = (c4*c23*c1 + s4*s1)*r13 + (c4*c23*s1 - s4*c1)*r23 - (c4*s23)*r33;
      long double c5 = (-s23*c1)*r13 + (-s23*s1)*r23 - (c23)*r33;
      Small(&s5);
      Small(&c5);
      d[4] = atan2(s5,c5);

      if(d[4]<(wrist_bend_min)){
	d[4]+=2*PI;
      }
      
      if(d[4]>(wrist_bend_max)){
	d[4]-=2*PI;
      }


      // check if in range and possible
      if (isnan(d[4])|| !InRange(&d[4],&wrist_bend_min,&wrist_bend_max)) {
	M_DEBUG("WRIST_BEND ERROR!!!\n");
#ifdef Christian_debug1
	fprintf(stdout,"WRIST_BEND ERROR!!!\n");
#endif
	overflow=5.0;
      }

      M_DEBUG("d[3]=%Lf\n",d[3]);

      // get the 6th angle
      long double s6 = (-s4*c23*c1 + c4*s1)*r11 + (-s4*c23*s1 - c4*c1)*r21 + (s4*s23)*r31;
      long double c6 = ((c5*c4*c23 + s5*s23)*c1 + c5*s4*s1)*r11 + 
                       ((c5*c4*c23 + s5*s23)*s1 - c5*s4*c1)*r21 + 
	               (-c5*c4*s23 + s5*c23)*r31;
      Small(&s6);
      Small(&c6);
      d[5] = atan2(s6,c6);

      if(d[5]<(flange_min)){
	d[5]+=2*PI;
      }
      
      if(d[5]>(flange_max)){
	d[5]-=2*PI;
      }

      // check if in range and possible
      if (isnan(d[5]) || !InRange(&d[5],&flange_min,&flange_max)) {
	M_DEBUG("FLANGE ERROR!!!\n");
#ifdef Christian_debug1
	fprintf(stdout,"FLANGE ERROR!!!\n");
#endif
	overflow=6.0;
      }
      M_DEBUG("d[5]=%Lf\n",d[5]);

      // Allow for alternative, "flipped" configuration
#ifdef K_AUTOFLIP
      flip=1;
      long double K_angle_dist1=0;
      long double angle_diff; 

      // See if angles are closer "other way around"
      for(int j=3; j<6; j+=2){
	if (fabs(angle_diff=K_input_d[j]-d[j])>PI){
	  d[j]+=Sgn(angle_diff)*2*PI;
	}
      }
      if(d[3]<(wrist_rot_min)){
	d[3]+=2*PI;
      }
      
      if(d[3]>(wrist_rot_max)){
	d[3]-=2*PI;
      }  
      
      if(d[5]<(flange_min)){
	d[5]+=2*PI;
      }
      
      if(d[5]>(flange_max)){
	d[5]-=2*PI;
      }  
      for(int j=3; j<6; j++){
	K_angle_dist1+=fabs(K_input_d[j]-d[j]);
      }
      long double K_angle_dist2=0;
      long double d_fliptest[6];

      d_fliptest[3]=d[3]+PI;
      d_fliptest[4]=d[4]*-1;
      d_fliptest[5]=d[5]+PI;
      for(int j=3; j<6; j+=2){
	if (fabs(angle_diff=K_input_d[j]-d_fliptest[j])>PI){
	  d_fliptest[j]+=Sgn(angle_diff)*2*PI;
	}
      }
      // fix range
      if(d_fliptest[3]<(wrist_rot_min)){
	d[3]+=2*PI;
      }
      
      if(d_fliptest[3]>(wrist_rot_max)){
	d[3]-=2*PI;
      }  
      
      if(d_fliptest[5]<(flange_min)){
	d[5]+=2*PI;
      }
      
      if(d_fliptest[5]>(flange_max)){
	d[5]-=2*PI;
      }  
      
      for(int j=3; j<6; j++){
	K_angle_dist2+=fabs(K_input_d[j]-d_fliptest[j]);
      }
      flip = 1;
      if(K_angle_dist1>K_angle_dist2){
	flip = -1;
      }
#endif	
      
      d[3]+=(flip<0)*PI;
      d[4]*=flip;
      d[5]+=(flip<0)*PI;


      // fix range
      if(d[3]<(wrist_rot_min)){
	d[3]+=2*PI;
      }
      
      if(d[3]>(wrist_rot_max)){
	d[3]-=2*PI;
      }  

      if(d[5]<(flange_min)){
	d[5]+=2*PI;
      }
      
      if(d[5]>(flange_max)){
	d[5]-=2*PI;
      }  
      
      // new angles are in range => we can save them
      if ( overflow == 0.0 )
	for ( int i=0; i<ControllerOutputDimension(); i++ )
	  d_temp[i]=d[i];
      else if (old_overflow != overflow) reflect=true;
      
      for ( int i=0; i<ControllerOutputDimension(); i++ )
	{
	  ControllerOutput(i)=d_temp[i];

#ifdef KIN_TIMING_TEST
    double K_elapsed_time =(tTime::Now()-K_sense_timer).ToUSec();
    fprintf(stdout,"Inverse kin. calc time (usec)= %f\n",K_elapsed_time);
#endif


	  
#ifdef Christian_debug2
	  fprintf(stdout,"\n angle %d=%f",i,(double)d[i]);
#endif
	}
      
      ClearControllerInput();
      SetControllerOutputChanged();
    }
  else
    ClearControllerOutput();
  
}


//----------------------------------------------------------------------
// class Kinematic Sense
//----------------------------------------------------------------------

void  tKinematic::Sense()
{
  // temporary keep Angles here:
  long double d[6];
  // temps for the switches
  long double arm,elbow,wrist,flip;

  if (SensorInputChanged()||reflect)
    {
      for (int i=0;i<SensorInputDimension();i++)
	K_input_d[i]=d[i]=SensorInput(i);
      reflect=false;
      M_DEBUG("%Lf %Lf %Lf %Lf %Lf %Lf\n",d[0],d[1],d[2],d[3],d[4],d[5]);
      // Calculate some Values needed:
      long double c1 = cos(d[0]);
      long double c2 = cos(d[1]);
      long double c3 = cos(d[2]);
      long double c4 = cos(d[3]);
      long double c5 = cos(d[4]);
      long double c6 = cos(d[5]);
      long double s1 = sin(d[0]);
      long double s2 = sin(d[1]);
      long double s3 = sin(d[2]);
      long double s4 = sin(d[3]);
      long double s5 = sin(d[4]);
      long double s6 = sin(d[5]);
      long double c23 = cos(d[1] + d[2]);
      long double s23 = sin(d[1] + d[2]);

      // First, we calculate the rotations:
      // Some common expressions:
      long double temp_A = c1*c23*c4 + s1*s4;
      long double temp_B = s1*c23*c4 - c1*s4;
      long double temp_C = c6*s23*s5 - s6*c23*s4;
      long double temp_D = s6*s23*s5 + c6*c23*s4;

      long double r11 = c6*c5*(temp_A) + c1*(temp_C) + s6*s1*c4;
      long double r21 = c6*c5*(temp_B) + s1*(temp_C) - s6*c1*c4;
      long double r31 = -c6*s23*c4*c5 + c6*c23*s5 + s23*s4*s6;

      long double r12 = -s6*c5*(temp_A) - c1*(temp_D) + c6*s1*c4;
      long double r22 = -s6*c5*(temp_B) - s1*(temp_D) - c6*c1*c4;
      long double r32 = s6*s23*c4*c5 - s6*c23*s5 + s23*s4*c6;

      long double r13 = s5*c1*c23*c4 + s5*s1*s4 - c1*s23*c5;
      long double r23 = s5*s1*c23*c4 - s5*c1*s4 - s1*s23*c5;
      long double r33 = -s23*c4*s5 - c23*c5;

      // Next we calculate the position of joint 4:
      long double px4 = -c1*s23*d4 + c1*c2*a2 + s1*d3;
      long double py4 = -s1*s23*d4 + s1*c2*a2 - c1*d3;
      long double pz4 = -c23*d4 - s2*a2;
      
      // And the position of the tool is...
      long double px = px4 + (d6*r13);
      long double py = py4 + (d6*r23);
      long double pz = pz4 + (d6*r33);


      Small(&px);
      Small(&py);
      Small(&pz);
      M_DEBUG("The Sensor Matrix:\n");
      M_DEBUG("%Lf %Lf %Lf %Lf\n",r11, r12, r13, px);
      M_DEBUG("%Lf %Lf %Lf %Lf\n",r21, r22, r23, py);
      M_DEBUG("%Lf %Lf %Lf %Lf\n",r31, r32, r33, pz);
      M_DEBUG("0 0 0 1\n");

      // get the Flags
      // arm and elbow is an easy one :)
      arm=Sgn(-px*c1 - py*s1);
      elbow=arm*Sgn(d4*c3 - a3*s3);

      // now get the wrist:
      long double z4x = -s4*c1*c23 - s1*c4;
      long double z4y = -s4*s1*c23 + c4*c1;
      long double z4z = -s23*s4;

      //calculate s*z4
      long double sxz4 = r12*z4x + r22*z4y +r32*z4z;
      if (sxz4==0.0)
	wrist= Sgn(r11*z4x + r21*z4y +r31*z4z);
      else
	wrist=Sgn(sxz4);


      M_DEBUG("ARM=%Lf ELBOW=%Lf\n WRIST=%Lf",arm,elbow,wrist);
      flip=wrist;

      // Calculate the orientations O T A
      Small(&r32);
      Small(&r31);
      long double tool_t = atan2(r32, -r31);

      long double st = sin(tool_t);
      long double ct = cos(tool_t);

      long double tas = -r33;
      long double tac = -r31*ct +r32*st;
      Small(&tas);
      Small(&tac);
      long double tool_a = atan2(tas,tac);
      long double tos = r21*st + r22*ct;
      long double toc = r11*st + r12*ct;
      Small(&tos);
      Small(&toc);
      long double tool_o = atan2(tos,toc);

      // Output the values:

      SensorOutput(eSO_TOOL_X)= px;
      SensorOutput(eSO_TOOL_Y)= py;
      SensorOutput(eSO_TOOL_Z)= pz;

      SensorOutput(eSO_TOOL_O)=tool_o;
      SensorOutput(eSO_TOOL_A)=tool_a;
      SensorOutput(eSO_TOOL_T)=tool_t;

      SensorOutput(eSO_ARM)=arm;
      SensorOutput(eSO_ELBOW)=elbow;
      SensorOutput(eSO_WRIST)=wrist;
      SensorOutput(eSO_FLIP)=flip;
      SensorOutput(eSO_OVERFLOW)=overflow;


      SetSensorOutputChanged();
      ClearSensorInput();
    }
  /* else{
      ClearSensorOutput();
      }*/
}

//----------------------------------------------------------------------
// class Kinematic Exception
//----------------------------------------------------------------------
void tKinematic::Exception(tExceptionType type)
{
  switch (type)
    {
    case eET_ParameterChanged:
      // save parameters in our private location (for better reading)
      a2=ParameterFloat(ePAR_A2);
      d2=ParameterFloat(ePAR_D2);
      a3=ParameterFloat(ePAR_A3);
      d4=ParameterFloat(ePAR_D4);
      d6=ParameterFloat(ePAR_D6);
      d3=ParameterFloat(ePAR_D3);
      waist_min=ParameterFloat(ePAR_WAIST_MIN);
      waist_max=ParameterFloat(ePAR_WAIST_MAX);
      Convert(&waist_min,&waist_max);
      shoulder_min=ParameterFloat(ePAR_SHOULDER_MIN);
      shoulder_max=ParameterFloat(ePAR_SHOULDER_MAX);
      Convert(&shoulder_min,&shoulder_max);
      elbow_min=ParameterFloat(ePAR_ELBOW_MIN);
      elbow_max=ParameterFloat(ePAR_ELBOW_MAX);
      Convert(&elbow_min,&elbow_max);
      wrist_rot_min=ParameterFloat(ePAR_WRIST_ROT_MIN);
      wrist_rot_max=ParameterFloat(ePAR_WRIST_ROT_MAX);
      Convert(&wrist_rot_min,&wrist_rot_max);
      wrist_bend_min=ParameterFloat(ePAR_WRIST_BEND_MIN);
      wrist_bend_max=ParameterFloat(ePAR_WRIST_BEND_MAX);
      Convert(&wrist_bend_min,&wrist_bend_max);
      flange_min=ParameterFloat(ePAR_FLANGE_MIN);
      flange_max=ParameterFloat(ePAR_FLANGE_MAX);
      Convert(&flange_min,&flange_max);
      break;
    case eET_PrepareForFirstSense:
      break;
    case eET_PrepareForFirstControl:
      break;
    case eET_PrepareForBreak:
      break;
    case eET_PrepareForRestart:
      break;
    }
}

void tKinematic::Small(long double *value)
{
  if (fabs(*value)< (long double)0.1e-10) *value=0.0;
}

bool tKinematic::InRange(long double *angle, long double  *par_min, long double
			 *par_max)
{
  if  ( ((*par_min<*par_max) && (*angle<*par_min || *angle>*par_max))
	|| ((*par_max<*par_min) && (*angle<*par_min && *angle>*par_max)) ){

    fprintf(stdout,"parmin=%f, parmax=%f, angle=%f\n",(double)(*par_min),(double)(*par_max),(double)(*angle));
    return false;
  }
  else
    return true;

}

void tKinematic::Convert(long double *par_min, long double *par_max)
{
  if (*par_max - *par_min >=2*M_PI) {
    *par_min=-M_PI;
    *par_max=M_PI;
  } else {
    *par_min=NormalizeAngleSigned(*par_min);
    *par_max=NormalizeAngleSigned(*par_max);
  }
}
