% 2013-08-19 Magnus Burenius

% Calibrate three or more affine cameras and transform a 3D reconstruction
% to a euclidean coordinate system. This function computes the
% rectification matrix that should be multiplied, in different ways, to the
% reconstructed points and cameras. The algorithm assumes the scaled
% orthographic camera model.

% C number of cameras. Assumes C >= 3.
% N number of reconstructed points.

% P 2Cx3 matrix. The affine camera matrices stacked on top of each other.
% X 3xN  matrix. The reconstructed 3D points.

% U 3x3 upper triangular rectification matrix such that:
% the rectified cameras are P1 = P*inv(U)
% the rectified 3D points are X1 = U*X

function [P1 X1] = Rectify( P, X )

v = [ 1 0 0 1 0]'; % initial guess corresponding to identity matrix

options = optimset( 'Display',      'off', ...
                    'Algorithm',    'levenberg-marquardt' ); % Default ? = 0.01.
%options = optimset('MaxIter',15);
v = lsqnonlin( @(x) RectifyFunction(P,x), v , [], [], options );

V = [ v(1) v(2) v(3) ;
      0    v(4) v(5) ;
      0    0    1    ];
  
P1 = P * V;
X1 = V \ X;

end


% The function that is minimized by Rectify. Imposes scaled orthographic
% camera constraints. The rows of a 2x3 camera matrix should be orthogonal
% and have the same norm.

% v vector of length 5 corresponding to the non zero elements of the upper
% triangular matrix V.

% C number of cameras

% P 2Cx3 matrix. The camera matrices stacked on top of each other.

function A = RectifyFunction( P, v )

V = [ v(1) v(2) v(3) ;
      0    v(4) v(5) ;
      0    0    1    ];

W = V*V';

% Camera constraints:
C = size(P,1) / 2;
A = zeros(2*C,1);

for c = 1:C
    a = P(2*c,:);
    b = P(2*c-1,:);
    
    A(2*c-1) = a*W*b';          % The rows should be orthogonal
    A(2*c)   = a*W*a' - b*W*b'; % and have the same norm.
end

end

