/***************************************************************************
                                databank.cpp
                             -------------------
                               Thu Nov 14 2002
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "databank.hpp"
#include "evexp.hpp"
#include "insrc.hpp"
#include "outxml.hpp"

Databank::Databank() {
}

void Databank::addFilter(const std::string &filterName, const std::string &fileName) {
     FilterMap::iterator iter = filters.find(filterName);
     if (iter == filters.end()) {
	  filters[filterName] = new EvFilter(fileName);
     }     
}

void Databank::addInputBuffer(const std::string &buffName, const std::string &fileName, const std::string &format) {
     BufferMap::iterator iter = buffers.find(buffName);
     
     if (iter == buffers.end()) {
	  if (format == "plain")
	       buffers[buffName] = new ISource(fileName);
	  else if (format == "xml") {
	       buffers[buffName] = new XSource(fileName);
	  }
	  else if (format == "stdin") {
	       buffers[buffName] = new StdinSource();
	  }
     }
     else {
	  
	  /* Make sure the old input type matches the new */
	  ISource *isplain = dynamic_cast<ISource*>(iter->second);
	  XSource *isxml   = dynamic_cast<XSource*>(iter->second);
	  StdinSource *isstdin = dynamic_cast<StdinSource*>(iter->second);
	  if ( (isplain && format != "plain" && format != "stdin") || 
	       (isxml   && format != "xml")   ||
	       (isstdin && format != "stdin" && format != "plain") ) {
	       EvExp e;
	       e.section = "Databank";
	       e.error = "Old definition of input source is not of type";
	       e.data = format;
	       throw e;
	  }
     }
}

void Databank::addOutputBuffer(const std::string &buffName, const std::string &fileName) {
     OutputBuffMap::iterator iter = outputs.find(buffName);
     
     if (iter == outputs.end()) {
	  outputs[buffName] = new XOutput(fileName);
     }
}

void Databank::delFilters() {
     for(FilterMap::const_iterator iter=filters.begin();iter!=filters.end();++iter)
	  delete iter->second;
}

void Databank::delInputBuffers() {
     for(BufferMap::const_iterator iter=buffers.begin();iter!=buffers.end();++iter)
	  delete iter->second;
}

void Databank::delOutputBuffers() {
     for(OutputBuffMap::const_iterator iter=outputs.begin();iter!=outputs.end();++iter)
	  delete iter->second;
}

void Databank::flushOutputBuffers() const {
     for(OutputBuffMap::const_iterator iter=outputs.begin();iter!=outputs.end();++iter)
	  iter->second->flush();
}

DSource& Databank::getBuffer(const std::string &buffName) {
     BufferMap::iterator iter = buffers.find(buffName);
     if (iter == buffers.end()) {
	  EvExp e;
	  e.section = "Databank";
	  e.error = "Inputbuffer not found";
	  e.data = buffName;
	  throw e;
     }
     DSource &dSrc = *(iter->second);
     return dSrc;
}


double Databank::getCnt(const std::string &cntName, const std::string &mapName) const {
     CntConMap::const_iterator cciter = counterCons.find(mapName);     
     if (cciter == counterCons.end()) {
	  EvExp e;
	  e.section = "Databank";
	  e.error = "Cannot find counter container";
	  e.data = mapName;
	  throw e;
     }
     CntMap::const_iterator citer = cciter->second.find(cntName);
     if (citer == cciter->second.end()) {
	  EvExp e;
	  e.section = "Databank";
	  e.error = "Cannot find counter variable";
	  e.data = cntName;
	  throw e;
     }
     return citer->second;
}


const CntMap& Databank::getCntMap(const std::string &mapName) const { 
     CntConMap::const_iterator cciter = counterCons.find(mapName);
     if (cciter == counterCons.end()) {
	  EvExp e;
	  e.section = "Databank";
	  e.error = "Cannot find counter container";
	  e.data = mapName;
	  throw e;
     }
     const CntMap& counters = cciter->second;
     return counters;
}


EvFilter& Databank::getFilter(const std::string &filterName) {
     FilterMap::iterator iter = filters.find(filterName);
     if (iter == filters.end()) {
	  EvExp e;
	  e.section = "Databank";
	  e.error = "Filter not found";
	  e.data = filterName;
	  throw e;
     }
     EvFilter &filter = *(iter->second);
     return filter;
}


XOutput& Databank::getOutBuffer(const std::string &buffName) {
     OutputBuffMap::iterator iter = outputs.find(buffName);
     if (iter == outputs.end()) {
	  EvExp e;
	  e.section = "Databank";
	  e.error = "Outbuffer not found";
	  e.data = buffName;
	  throw e;
     }
     XOutput &xOut = *(iter->second);
     return xOut;
}


std::string Databank::getVar(const std::string &varName, const std::string &mapName) const {     
     VarConMap::const_iterator vciter = varCons.find(mapName);
     if (vciter == varCons.end()) {
	  EvExp e;
	  e.section = "Databank";
	  e.error = "Cannot find string container";
	  e.data = mapName;
	  throw e;
     }
     DataMap::const_iterator viter = vciter->second.find(varName);
     if (viter == vciter->second.end()) {
	  EvExp e;
	  e.section = "Databank";
	  e.error = "Cannot find string variable";
	  e.data = varName;
	  throw e;
     }
     return viter->second;
}


const DataMap& Databank::getVarMap(const std::string &mapName) {
     VarConMap::const_iterator vciter=varCons.find(mapName);
     if (vciter == varCons.end()) {
	  EvExp e;
	  e.section = "Databank";
	  e.error = "Cannot find string container";
	  e.data = mapName;
	  throw e;
     }
     const DataMap& variables = vciter->second;
     return variables;
}

double& Databank::giveCntRef(const std::string &varName, const std::string &mapName) {
     return counterCons[mapName][varName];
}

std::string& Databank::giveVarRef(const std::string &varName, const std::string &mapName) {
     return varCons[mapName][varName];
}

Databank::~Databank() {
     delInputBuffers();
     delFilters();
     delOutputBuffers();
}


Databank* getDatabank() {
     static Databank db;
     return &db;
}
