/***************************************************************************
                                  functor.hpp
                             -------------------
                               Sat Oct 19 2002
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef FUNCTOR_H
#define FUNCTOR_H

#include <string>
#include <sstream>
#include <map>

#include "evexp.hpp"
#include "execute.hpp"

#include "parser/node.hpp"

class Functor_base;
typedef std::multimap<std::string, const Functor_base *> funcmap;
template<class R> class Functor_type;

/** 
 * Base class of the functor, which 
 * encapsulates external function pointers 
 */
class Functor_base {
public:
    Functor_base(int a) : args(a) {}
    virtual ~Functor_base() {}
public:
    int args;
public:
    template<class R>
    const Functor_type<R> &downcast(Type2type<R>) const { 
        return dynamic_cast<const Functor_type<R> &>(*this);
    }

#define CONVERT_FAILED(R) \
    virtual const Functor_type<R> &downcast(Type2type<R>) const  \
    {                                                           \
	throw std::bad_cast();                                  \
    }

    CONVERT_FAILED(bool)
        CONVERT_FAILED(std::string)
        CONVERT_FAILED(double)
        CONVERT_FAILED(std::string &)
        CONVERT_FAILED(double &)
        };

template<class R>
class Functor_type : public Functor_base {
public:
    Functor_type(int args) : Functor_base(args) {}
    virtual Execute<R> &process(NodeInfo::Node* oper, bool outmost) const = 0;
public:
    int args;
};

/* Functor class, representing external functions with one argument */
template<class R, class T1>
class Functorone : public Functor_type<R> {
    typedef R return_type;
    typedef R (*function_type)(T1);
public:
    Functorone(function_type f) : Functor_type<R>(1), fp(f) {}
    Execute<R> &process(NodeInfo::Node* oper, bool outmost) const;
private:
    const Functor_type<R> &downcast(Type2type<R>) const { return *this; }
    function_type fp;
};

/* Functor class, representing external functions with two arguments */
template<class R, class T1, class T2>
class Functortwo : public Functor_type<R> {
    typedef R return_type;
    typedef R (*function_type)(T1, T2);
public:
    Functortwo(function_type f) : Functor_type<R>(2), fp(f) {}
    Execute<R> &process(NodeInfo::Node* oper, bool outmost) const;
private:
    const Functor_type<R> &downcast(Type2type<R>) const { return *this; }
    function_type fp;
};

/* Functor class, representing external functions with three arguments */
template<class R, class T1, class T2, class T3>
class Functorthree : public Functor_type<R> {
    typedef R return_type;
    typedef R (*function_type)(T1, T2, T3);
public:
    Functorthree(function_type f) : Functor_type<R>(3), fp(f) {}
    Execute<R> &process(NodeInfo::Node* oper, bool outmost) const;
private:
    const Functor_type<R> &downcast(Type2type<R>) const { return *this; }
    function_type fp;
};

/* Functor class, representing external functions with four arguments */
template<class R, class T1, class T2, class T3, class T4>
class Functorfour : public Functor_type<R> {
    typedef R return_type;
    typedef R (*function_type)(T1, T2, T3, T4);
public:
    Functorfour(function_type f) : Functor_type<R>(4), fp(f) {}
    Execute<R> &process(NodeInfo::Node* oper, bool outmost) const;
private:
    const Functor_type<R> &downcast(Type2type<R>) const { return *this; }
    function_type fp;
};

/* Functor class, representing external functions with five arguments */
template<class R, class T1, class T2, class T3, class T4, class T5>
class Functorfive : public Functor_type<R> {
    typedef R return_type;
    typedef R (*function_type)(T1, T2, T3, T4, T5);
public:
    Functorfive(function_type f) : Functor_type<R>(5), fp(f) {}
    Execute<R> &process(NodeInfo::Node* oper, bool outmost) const;
private:
    const Functor_type<R> &downcast(Type2type<R>) const { return *this; }
    function_type fp;
};

/* Functor class, representing external functions with six arguments */
template<class R, class T1, class T2, class T3, class T4, class T5, class T6>
class Functorsix : public Functor_type<R> {
    typedef R return_type;
    typedef R (*function_type)(T1, T2, T3, T4, T5, T6);
public:
    Functorsix(function_type f) : Functor_type<R>(6), fp(f) {}
    Execute<R> &process(NodeInfo::Node* oper, bool outmost) const;
private:
    const Functor_type<R> &downcast(Type2type<R>) const { return *this; }
    function_type fp;
};

/* Functor class, representing external functions with seven arguments */
template<class R, class T1, class T2, class T3, class T4, class T5, class T6, class T7>
class Functorseven : public Functor_type<R> {
    typedef R return_type;
    typedef R (*function_type)(T1, T2, T3, T4, T5, T6, T7);
public:
    Functorseven(function_type f) : Functor_type<R>(7), fp(f) {}
    Execute<R> &process(NodeInfo::Node* oper, bool outmost) const;
private:
    const Functor_type<R> &downcast(Type2type<R>) const { return *this; }
    function_type fp;
};

/* Functor class, representing external functions with eight arguments */
template<class R, class T1, class T2, class T3, class T4, class T5, class T6, class T7, class T8>
class Functoreight : public Functor_type<R> {
    typedef R return_type;
    typedef R (*function_type)(T1, T2, T3, T4, T5, T6, T7, T8);
public:
    Functoreight(function_type f) : Functor_type<R>(8), fp(f) {}
    Execute<R> &process(NodeInfo::Node* oper, bool outmost) const;
private:
    const Functor_type<R> &downcast(Type2type<R>) const { return *this; }
    function_type fp;
};

/* Functor class, representing external functions with nine arguments */
template<class R, class T1, class T2, class T3, class T4, class T5, class T6, class T7, class T8, class T9>
class Functornine : public Functor_type<R> {
    typedef R return_type;
    typedef R (*function_type)(T1, T2, T3, T4, T5, T6, T7, T8, T9);
public:
    Functornine(function_type f) : Functor_type<R>(9), fp(f) {}
    Execute<R> &process(NodeInfo::Node* oper, bool outmost) const;
private:
    const Functor_type<R> &downcast(Type2type<R>) const { return *this; }
    function_type fp;
};

template<class R> Execute<R> &process_arg(NodeInfo::Node *argdata);

/********************* Implementation *********************/

template <class R, class T1>
Execute<R> &Functorone<R, T1>::process(NodeInfo::Node* oper, bool outmost) const {
    NodeInfo::NodeVector *args = oper->args;
    Execute<T1> &t1 = process_arg<T1>(args->at(0));
    Execute<R>  *rt = 
        new Execute_func1<function_type, R, Execute<T1> >(fp, t1, oper);
    queue(rt, outmost);
    return *rt;
}

template <class R, class T1, class T2>
Execute<R> &Functortwo<R, T1, T2>::process(NodeInfo::Node* oper, bool outmost) const {
    NodeInfo::NodeVector *args = oper->args;
    Execute<T1> &t1 = process_arg<T1>(args->at(0));
    Execute<T2> &t2 = process_arg<T2>(args->at(1));
    Execute<R>  *rt = 
        new Execute_func2<function_type, R, Execute<T1>, Execute<T2> >
        (fp, t1, t2, oper);
    queue(rt, outmost);
    return *rt;
}

template <class R, class T1, class T2, class T3>
Execute<R> &Functorthree<R, T1, T2, T3>::process(NodeInfo::Node* oper, bool outmost) const {
    NodeInfo::NodeVector *args = oper->args;
    Execute<T1> &t1 = process_arg<T1>(args->at(0));
    Execute<T2> &t2 = process_arg<T2>(args->at(1));
    Execute<T3> &t3 = process_arg<T3>(args->at(2));
    Execute<R>  *rt =
        new Execute_func3<function_type, R, Execute<T1>, Execute<T2>, Execute<T3> >
        (fp, t1, t2, t3, oper);
    queue(rt, outmost);
    return *rt;
}

template <class R, class T1, class T2, class T3, class T4>
Execute<R> &Functorfour<R, T1, T2, T3, T4>::process(NodeInfo::Node* oper, bool outmost) const {
    NodeInfo::NodeVector *args = oper->args;
    Execute<T1> &t1 = process_arg<T1>(args->at(0));
    Execute<T2> &t2 = process_arg<T2>(args->at(1));
    Execute<T3> &t3 = process_arg<T3>(args->at(2));
    Execute<T4> &t4 = process_arg<T4>(args->at(3));
    Execute<R>  *rt =
        new Execute_func4<function_type, R, Execute<T1>, Execute<T2>, 
        Execute<T3>, Execute<T4> >
        (fp, t1, t2, t3, t4, oper);
    queue(rt, outmost);
    return *rt;
}

template <class R, class T1, class T2, class T3, class T4, class T5>
Execute<R> &Functorfive<R, T1, T2, T3, T4, T5>::process(NodeInfo::Node* oper, bool outmost) const {
    NodeInfo::NodeVector *args = oper->args;
    Execute<T1> &t1 = process_arg<T1>(args->at(0));
    Execute<T2> &t2 = process_arg<T2>(args->at(1));
    Execute<T3> &t3 = process_arg<T3>(args->at(2));
    Execute<T4> &t4 = process_arg<T4>(args->at(3));
    Execute<T5> &t5 = process_arg<T5>(args->at(4));
    Execute<R>  *rt =
        new Execute_func5<function_type, R, Execute<T1>, Execute<T2>, 
        Execute<T3>, Execute<T4>, Execute<T5> >
        (fp, t1, t2, t3, t4, t5, oper);
    queue(rt, outmost);
    return *rt;
}
  
template <class R, class T1, class T2, class T3, class T4, class T5, class T6>
Execute<R> &Functorsix<R, T1, T2, T3, T4, T5, T6>::process(NodeInfo::Node* oper, bool outmost) const {
    NodeInfo::NodeVector *args = oper->args;
    Execute<T1> &t1 = process_arg<T1>(args->at(0));
    Execute<T2> &t2 = process_arg<T2>(args->at(1));
    Execute<T3> &t3 = process_arg<T3>(args->at(2));
    Execute<T4> &t4 = process_arg<T4>(args->at(3));
    Execute<T5> &t5 = process_arg<T5>(args->at(4));
    Execute<T6> &t6 = process_arg<T6>(args->at(5));
    Execute<R>  *rt =
        new Execute_func6<function_type, R, Execute<T1>, Execute<T2>, 
        Execute<T3>, Execute<T4>, Execute<T5>, Execute<T6> >
        (fp, t1, t2, t3, t4, t5, t6, oper);
    queue(rt, outmost);
    return *rt;
}
  
template <class R, class T1, class T2, class T3, class T4, class T5, class T6, class T7>
Execute<R> &Functorseven<R, T1, T2, T3, T4, T5, T6, T7>::process(NodeInfo::Node* oper, bool outmost) const {
    NodeInfo::NodeVector *args = oper->args;
    Execute<T1> &t1 = process_arg<T1>(args->at(0));
    Execute<T2> &t2 = process_arg<T2>(args->at(1));
    Execute<T3> &t3 = process_arg<T3>(args->at(2));
    Execute<T4> &t4 = process_arg<T4>(args->at(3));
    Execute<T5> &t5 = process_arg<T5>(args->at(4));
    Execute<T6> &t6 = process_arg<T6>(args->at(5));
    Execute<T7> &t7 = process_arg<T7>(args->at(6));
    Execute<R>  *rt =
        new Execute_func7<function_type, R, Execute<T1>, Execute<T2>, 
        Execute<T3>, Execute<T4>, Execute<T5>, 
        Execute<T6>, Execute<T7> >
        (fp, t1, t2, t3, t4, t5, t6, t7, oper);
    queue(rt, outmost);
    return *rt;
}
  
template <class R, class T1 , class T2, class T3, class T4, class T5, class T6, class T7, class T8>
Execute<R> &Functoreight<R, T1, T2, T3, T4, T5, T6, T7, T8>::process(NodeInfo::Node* oper, bool outmost) const {
    NodeInfo::NodeVector *args = oper->args;
    Execute<T1> &t1 = process_arg<T1>(args->at(0));
    Execute<T2> &t2 = process_arg<T2>(args->at(1));
    Execute<T3> &t3 = process_arg<T3>(args->at(2));
    Execute<T4> &t4 = process_arg<T4>(args->at(3));
    Execute<T5> &t5 = process_arg<T5>(args->at(4));
    Execute<T6> &t6 = process_arg<T6>(args->at(5));
    Execute<T7> &t7 = process_arg<T7>(args->at(6));
    Execute<T8> &t8 = process_arg<T8>(args->at(7));
    Execute<R>  *rt =
        new Execute_func8<function_type, R, Execute<T1>, Execute<T2>, 
        Execute<T3>, Execute<T4>, Execute<T5>, 
        Execute<T6>, Execute<T7>, Execute<T8> >
        (fp, t1, t2, t3, t4, t5, t6, t7, t8, oper);
    queue(rt, outmost);
    return *rt;
}

template <class R, class T1 , class T2, class T3, class T4, class T5, class T6, class T7, class T8, class T9>
Execute<R> &Functornine<R, T1, T2, T3, T4, T5, T6, T7, T8, T9>::process(NodeInfo::Node* oper, bool outmost) const {
    NodeInfo::NodeVector *args = oper->args;
    Execute<T1> &t1 = process_arg<T1>(args->at(0));
    Execute<T2> &t2 = process_arg<T2>(args->at(1));
    Execute<T3> &t3 = process_arg<T3>(args->at(2));
    Execute<T4> &t4 = process_arg<T4>(args->at(3));
    Execute<T5> &t5 = process_arg<T5>(args->at(4));
    Execute<T6> &t6 = process_arg<T6>(args->at(5));
    Execute<T7> &t7 = process_arg<T7>(args->at(6));
    Execute<T8> &t8 = process_arg<T8>(args->at(7));
    Execute<T9> &t9 = process_arg<T9>(args->at(8));
    Execute<R>  *rt =
        new Execute_func9<function_type, R, Execute<T1>, Execute<T2>, 
        Execute<T3>, Execute<T4>, Execute<T5>, 
        Execute<T6>, Execute<T7>, Execute<T8>, Execute<T9> >
        (fp, t1, t2, t3, t4, t5, t6, t7, t8, t9, oper);
    queue(rt, outmost);
    return *rt;
}

/*** Supportive functions ***/

template<class T> bool isTokenConst(NodeInfo::Node* arg);
template<class T>
bool isTokenConst(NodeInfo::Node* arg)
{
    return false;
}

template<> bool isTokenConst<bool>(NodeInfo::Node* arg);
template<> bool isTokenConst<double>(NodeInfo::Node* arg);
template<> bool isTokenConst<std::string>(NodeInfo::Node* arg);

template<class T> T getTokenConst(NodeInfo::Node* arg);
template<class T>
T getTokenConst(NodeInfo::Node* arg) {
    EvExp e;
    e.section = "Varprocessor";
    e.error = "Expected token constant, got";
    e.data = "something else";
    throw e;
}

template<> bool getTokenConst(NodeInfo::Node* arg);
template<> double getTokenConst(NodeInfo::Node* arg);
template<> std::string getTokenConst(NodeInfo::Node* arg);

#include "process.hpp"
namespace dumdidum {
    /*this silly namespace solves a bug in gcc 3.4.4 (it can't handle the
      ::getVarprocessor() call that should be used
      (we can't just call getVarprocessor() since the function is templated).*/
    Varprocessor* getVarprocessor();
}

template<class R>
Execute<R> &process_arg(NodeInfo::Node *argdata) {     
    if(argdata->type == NodeInfo::T_CALL) {
        Execute<R> &ret = dumdidum::getVarprocessor()->compile(argdata, Type2type<R>());
        return ret;
    }
    else if(isTokenConst<R>(argdata)) {
        Execute<R> *e = new Execute_const<R>(getTokenConst<R>(argdata), argdata);
        queue(e, false, true);
        return *e;
    }
    else {
        std::ostringstream strm;
        strm << "Token is not of type '" << type_name(Type2type<R>()) << "'";
        throw Parse::Error(strm.str(), argdata);
    }
}
#endif
