/***************************************************************************
                                  insrc.cpp
                             -------------------
                               Mon Sep 2 2002
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "insrc.hpp"

#include <iostream>
#include <sstream>

#include <boost/regex.hpp>

#include "evexp.hpp"

EvFilter::EvFilter(const std::string &filename) : configurated_(false) {
     filename_ = filename;
     std::ifstream test(filename.c_str());
     if (!test) {
	  EvExp e;
	  e.section = "Filter file reader:";
	  e.error = "Cannot open file";
	  e.data = filename;
	  throw e;
     }
     test.close();
}


void EvFilter::configurate(const std::string &idelim, const std::string &edelim) {
     if (configurated_) {
	  EvExp e;
	  e.section = "Filter";
	  e.error = "Internal program error, already configured (called configure() twice?)";
	  e.data = "";
	  throw e;
     }
     ISource file(filename_);     
     std::string buffer;
     boost::smatch wordMatch;
     std::ostringstream sts;
     sts << "([^" << idelim << "\\n\\r]+)" << "(" << idelim << ")" 
	 << "([^" << edelim << "\\n\\r]+)" << "(" << edelim << ")";
     boost::regex regExpr(sts.str());
     while (!file.isExhausted()) {
	  buffer = file.getData();
	  if (boost::regex_search(buffer, wordMatch, regExpr)) {
	       filterdata_[wordMatch[1]] = wordMatch[3];
	       file.dataSeen(wordMatch[0].length());
	  }
	  else {
	       file.dataSeen(0);
	  }
     }
     configurated_ = true;
}

bool EvFilter::isConfigurated() const {
     return configurated_;
}

const DataMap& EvFilter::getFiltdata() const {
     const DataMap& fltdata = filterdata_;
     return fltdata;
}

EvFilter::~EvFilter() {
}

/***** ISource *****/

ISource::ISource(const std::string &filename) : eofreached_(false), exhausted_(false) {
     filename_ = filename;
     inputFile_.open(filename.c_str());
     if (!inputFile_.is_open()) {
	  EvExp e;
	  e.section = "Plain Text Datasource";
	  e.error = "Cannot open file";
	  e.data = filename;
	  throw e;
     }
     
     searchBuffer_.erase();
     chunkSize_ = INI_CHUNK;
}

void ISource::configurate(const DataMap &meta) {
     EvExp e;
     e.section = "Plain Text Datasource";
     e.error = "Plain Text Datasource needs no meta configuration (ran xfield on Plain Data Source ?)";
     e.data = "";
     throw e;
}

ISource::ISource() {
}

void ISource::dataSeen(const int size) {
     if (size == 0) {
	  if (eofreached_) 
	       exhausted_ = true;
	  else 
	       chunkSize_ = chunkSize_ + ADD_CHUNK;
     }
     else if (size < MAX_CHUNK) {
	  if (chunkSize_ != INI_CHUNK)
	       chunkSize_ = INI_CHUNK;
	  searchBuffer_ = std::string(searchBuffer_, size);
     }
     else {
	  EvExp e;
	  e.section = "Plain Text Datasource";
	  e.error = "Maximum buffer size reached, but data could not be recognized by reader (invalid format declaration or buffer too small) in";
	  e.data = getAssociatedFile();
	  throw e;
     }
}


std::string ISource::getAssociatedFile() const {
     return filename_;
}


std::string ISource::getData() {
     eofreached_ = !readChunkFromFile(); 
     return searchBuffer_;
}


bool ISource::isConfigurated() const {
     return true;
}


bool ISource::isExhausted() const {
     return exhausted_;
}


bool ISource::readChunkFromFile() {// Tricky...
     char charBuffer;
     int i;
     int length;
     bool success;
     success = !inputFile_.eof();
     i = 0;
     length = chunkSize_ - searchBuffer_.size();
     if (length > 0) {
	  while(inputFile_.read(&charBuffer, 1)) {
	       if (inputFile_.eof())
		    break;
	       searchBuffer_.append(1, charBuffer);
	       ++i;
	       if (i == length)
		    break;
	  }
     }
     return success;
}

ISource::~ISource() {
     if (inputFile_.is_open()) {
	  inputFile_.close();
     }
}

/***** StdinSource *****/

StdinSource::StdinSource() {
     eofreached_ = false;
     exhausted_ = false;
     filename_ = "standard input";
     searchBuffer_.erase();
     chunkSize_ = INI_CHUNK;
}

bool StdinSource::readChunkFromFile() {// Tricky...
     char charBuffer;
     int i;
     int length;
     bool success;
     success = !!std::cin;
     i = 0;
     length = chunkSize_ - searchBuffer_.size();
     if (length > 0) {
	  while(std::cin.get(charBuffer)) {
	       if (!std::cin)
		    break;
	       searchBuffer_.append(1, charBuffer);
	       ++i;
	       if (i == length)
		    break;
	  }
     }
     return success;
}

StdinSource::~StdinSource() {
}

/***** XsHandler *****/

XsHandler::XsHandler(XSource *xsrc) : xsrc_(xsrc) {
}


void XsHandler::startElement (const XMLCh *const uri, const XMLCh *const localname, const XMLCh *const qname, const Attributes &attrs) {
     xsrc_->sawElementStart(localname, attrs);
}


void XsHandler::endElement (const XMLCh *const uri, const XMLCh *const localname, const XMLCh *const qname) {
     xsrc_->sawElementEnd(localname);
}


void XsHandler::characters(const XMLCh* const chars, const unsigned int length) {
     xsrc_->sawChars(chars);
}


void XsHandler::error(const SAXParseException &exception) {
     fatalError(exception);
}


void XsHandler::warning(const SAXParseException &exception) {
     fatalError(exception);
}


XsHandler::~XsHandler() {
}

/***** XSource *****/

XSource::XSource(const std::string &filename) : busy_(false), configurated_(false), 
eofreached_(false), exhausted_(false), readnexttnode_(false), state_(-1) {
     filename_ = filename;
     std::ifstream test(filename.c_str());
     if (!test) {
	  EvExp e;
	  e.section = "XML input file reader:";
	  e.error = "Cannot open file";
	  e.data = filename;
	  throw e;
     }
     test.close();
}

void XSource::addBufferData(const std::string &data) {
     searchBuffer_ += data;
}


void XSource::configurate(const DataMap &meta) {
     if (configurated_) {
	  EvExp e;
	  e.section = "XML Datasource";
	  e.error = "Internal program error, already configured (called configure() twice?)";
	  e.data = "";
	  throw e;
     }
     
     nodes_.clear();
     
     EvNode enode;
     enode.reledata.clear();
     enode.name.clear();

     bool allfound = true;     
     allfound &= meta.find("root") != meta.end();
     allfound &= meta.find("datanode") != meta.end();
     allfound &= meta.find("data1") != meta.end();
     allfound &= meta.find("data2") != meta.end();

     if (!allfound) {
	  EvExp e;
	  e.section = "XML Datasource";
	  e.error = "Internal program error, meta data is missing in configuration, data required is";
	  e.data = "root, datanode, data1, data2";
	  throw e;
     }

     enode.name = meta.find("root")->second;
     nodes_.push_back(enode);
     
     enode.name = meta.find("datanode")->second;
     
     DataVector rnodes;
     rnodes.push_back(meta.find("data1")->second);
     rnodes.push_back(meta.find("data2")->second);
     enode.reledata = rnodes;
     nodes_.push_back(enode);
     
     parser_ = XMLReaderFactory::createXMLReader();
     handler_ = new XsHandler(this);
     parser_->setContentHandler(handler_);
     parser_->setErrorHandler(handler_);
     
     try {
	  parser_->parseFirst(filename_.c_str(), scantoken_);
     }     
     catch (const XMLException& toCatch) {
	  EvExp e;
	  e.section = "XML DataSource";
	  e.error = "Xerces Error during first parse";
	  e.data = XML2Ch(toCatch.getMessage()).toStr();
	  throw e;
     }     
     catch (const SAXParseException& toCatch) {
	  char *filename = XMLString::transcode(toCatch.getSystemId());
	  EvExp e;
	  e.section = "XML config file reader";
	  e.error = "Xerces Error during first parse";
	  std::ostringstream sts;
	  sts << XML2Ch(toCatch.getMessage()).toStr();
	  if (filename != 0) {
	       sts << " in file: " << filename;
	       XMLSSize_t line = toCatch.getLineNumber();
	       if (line != -1) {
		    sts << " at line: " << line;
		    XMLSSize_t column = toCatch.getColumnNumber();
		    if (column != -1) {
			 sts << ", column: " << column;
		    }
	       }
	  }
	  e.data = sts.str();
	  delete [] filename;
	  throw e;
     }
     configurated_ = true;
}


void XSource::dataSeen(const int size) {
     if (size == 0) {
	  if (!eofreached_) {
	       EvExp e;
	       e.section = "XML DataSource";
	       e.error = "A set of variables are in the buffer, but they could not be recognized by reader in buffer for";
	       e.data = searchBuffer_;
	       throw e;
	  }
	  exhausted_ = true;
     }
     searchBuffer_ = std::string(searchBuffer_, size);
}


std::string XSource::getAssociatedFile() const {
     return filename_;
}


std::string XSource::getData() {
     readChunkFromFile(); 
     return searchBuffer_;
}


bool XSource::isConfigurated() const {
     return configurated_;
}


bool XSource::isExhausted() const {
     return exhausted_;
}

bool XSource::readChunkFromFile() {
     busy_ = true;
     try {
	  while(busy_ && !eofreached_)
	       eofreached_ = !parser_->parseNext(scantoken_);
     }
     catch (const XMLException& toCatch) {
	  EvExp e;
	  e.section = "XML DataSource";
	  e.error = "Xerces Error during parsing";
	  e.data = XML2Ch(toCatch.getMessage()).toStr();
	  throw e;
     }  
     catch (const SAXParseException& toCatch) {
	  char *filename = XMLString::transcode(toCatch.getSystemId());
	  EvExp e;
	  e.section = "XML DataSource";
	  e.error = "Xerces Error during parsing";
	  std::ostringstream sts;
	  sts << XML2Ch(toCatch.getMessage()).toStr();
	  if (filename != 0) {
	       sts << " in file: " << filename;
	       XMLSSize_t line = toCatch.getLineNumber();
	       if (line != -1) {
		    sts << " at line: " << line;
		    XMLSSize_t column = toCatch.getColumnNumber();
		    if (column != -1) {
			 sts << ", column: " << column;
		    }
	       }
	  }
	  e.data = sts.str();
	  delete [] filename;
	  throw e;
     }
     
     return eofreached_;
}


void XSource::sawChars(const XMLCh *const chars) {
     if (readnexttnode_) {
	  std::string val = XML2Ch(chars).toStr();
	  val += "\t";
	  addBufferData(val);
	  readnexttnode_ = false;
     }
}


void XSource::sawElementStart(const XMLCh *const name,const Attributes &attrs) {
     if (nodes_[state_ + 1].name == XML2Ch(name).toStr()) {
	  ++state_;
	  for (DataVector::const_iterator iter = nodes_[state_].reledata.begin(); iter != nodes_[state_].reledata.end(); ++iter) {
	       if (*iter == "#txtnode")
		    readnexttnode_ = true;
	       else {
		    const XMLCh* attrval = attrs.getValue(Ch2XML(*iter).toXML());
		    if (attrval != 0) {
			 std::string val = XML2Ch(attrval).toStr();
			 val += "\t";
			 addBufferData(val);
		    }
	       }
	  }
     }
}


void XSource::sawElementEnd(const XMLCh *const name) {
     if (state_ > -1)
	  if (nodes_[state_].name == XML2Ch(name).toStr()) {
	       if (state_ == (int)nodes_.size() - 1)
		    busy_ = false;
	       --state_;
	  }
}

XSource::~XSource() {
     if (configurated_) {
	  parser_->parseReset(scantoken_);
	  delete parser_;
	  delete handler_;
     }
}
