%{
#include <stdio.h>
#include <iostream>
#include "parse.hpp"
#include "error.hpp"
#include "node.hpp"
#include "control.hpp"

extern void yyrestart(FILE *);
%}

%union {
     const char *str;
     int integer;
     double dbl;
     bool boolean;
     NodeInfo::Node *node;
}

%{
#include "scan_parse.hpp"
using namespace Parse;
static void error(const char *m, const char *id, YYLTYPE pos);
%}

%pure_parser

/* Expect one shift/reduce confilict. The conflict happens 
   when an error is found inside an 'if' or 'else' clause */
%expect 1

%token <str>     STR ID
%token <integer> INT TMPLREF
%token <dbl>     DBL
%token <boolean> BOOL

%token <operator> OR    "||"
%token <operator> AND   "&&"
%token <operator> EQ    "=="
%token <operator> NEQ   "!="
%token <operator> LEQ   "<="
%token <operator> GEQ   ">="
%token <operator> PL_PL "++"
%token <operator> MI_MI "--"
%token <operator> PL_EQ "+="
%token <operator> MI_EQ "-="

%token IF ELSE

%type <node> expr_l expr list plistp in_if ifexpr var cont name braces parens

%left "||" "&&"
%left "==" "!=" "<=" ">=" '<' '>'
%nonassoc '=' "+=" "-="
%left '.' '%'
%left '+' '-'
%left '*' '/'
%nonassoc "++" "--"
%left UMINUS

%%

start : /* empty */		{ result(0);  }
      | expr_l			{ result($1); }
      ;

expr_l : expr ';'		{ $$ = Line($1, 0);  }
       | expr ';' expr_l	{ $$ = Line($1, $3); }
       | ifexpr			{ $$ = Line($1, 0);  }
       | ifexpr expr_l		{ $$ = Line($1, $2); }
       | expr error		{ error("Missing semi-colon",     "E001", @2); }
       | expr ')'		{ error("Unbalanced parentheses", "E002", @2); }
       | ';'			{ error("Empty expression",       "E017", @1); }
       ;

ifexpr : IF parens braces	{ $$ = Call("if_", List($2, List($3, 0)), @1, @3); }
       | IF parens braces ELSE braces { $$ = Call("ifelse", List($2, List($3, List($5, 0))), @1, @5); }
       ;

expr : ID plistp		{ $$ = Call($1, $2, @1, @2); }
     | STR			{ $$ = Str( $1,     @1, @1); }
     | INT			{ $$ = Int( $1,     @1, @1); }
     | '-' INT %prec UMINUS     { $$ = Int(-$2,     @1, @2); }
     | DBL			{ $$ = Dbl( $1,     @1, @1); }
     | '-' DBL %prec UMINUS     { $$ = Dbl(-$2,     @1, @2); }
     | BOOL			{ $$ = Bool($1,     @1, @1); }
     | TMPLREF			{ $$ = TmplRef($1,  @1, @1); }
     | var                      { $$ = $1;                   }
     | expr '.'  expr           { $$ = Call("concat", List($1, List($3, 0)), @1, @3); }
     | expr '%'  expr           { $$ = Call("mod",    List($1, List($3, 0)), @1, @3); }
     | expr '+'  expr           { $$ = Call("add",    List($1, List($3, 0)), @1, @3); }
     | expr '-'  expr           { $$ = Call("sub",    List($1, List($3, 0)), @1, @3); }
     | expr '*'  expr           { $$ = Call("mul",    List($1, List($3, 0)), @1, @3); }
     | expr '/'  expr           { $$ = Call("div",    List($1, List($3, 0)), @1, @3); }
     | expr "||" expr		{ $$ = Call("or",     List($1, List($3, 0)), @1, @3); }
     | expr "&&" expr		{ $$ = Call("and",    List($1, List($3, 0)), @1, @3); }
     | expr "==" expr           { $$ = Call("_eq",    List($1, List($3, 0)), @1, @3); }
     | expr "!=" expr           { $$ = Call("not",    List(Call("_eq", 
				         List($1, List($3, 0)), @1, @3), 0), @1, @3); }
     | expr "<=" expr           { $$ = Call("leq",    List($1, List($3, 0)), @1, @3); }
     | expr ">=" expr           { $$ = Call("leq",    List($3, List($1, 0)), @1, @3); }
     | expr '>'  expr           { $$ = Call("not",    List(Call("leq", 
                                         List($1, List($3, 0)), @1, @3), 0), @1, @3); }
     | expr '<'  expr           { $$ = Call("not",    List(Call("leq", 
                                         List($3, List($1, 0)), @1, @3), 0), @1, @3); }
     | expr '='  expr		{ $$ = Call("_set",   List($1, List($3, 0)), @1, @3); }
     | expr "+=" expr		{ $$ = Call("inc",    List($1, List($3, 0)), @1, @3); }
     | expr "-=" expr		{ $$ = Call("dec",    List($1, List($3, 0)), @1, @3); }
     | "++" expr                { $$ = Call("inc",    List($2, 0),           @1, @2); }
     | expr "++"                { $$ = Call("inc",    List($1, 0),           @1, @2); }
     | "--" expr                { $$ = Call("dec",    List($2, 0),           @1, @2); }
     | expr "--"                { $$ = Call("dec",    List($1, 0),           @1, @2); }
     | error			{ error("Syntax error", "E003", @1);                  }
     ;

plistp : '(' list ')'		{ $$ = $2; }
       | error			{ error("Unknown identifier",   "E004", @0); } // @0
       ;

list : expr			{ $$ = List($1, 0);  }
     | expr ',' list		{ $$ = List($1, $3); }
     | expr error		{ error("Comma sign expected", "E011", @2); }
     | ','			{ error("Missing expression",  "E012", @1); }
     ;

parens : '(' expr ')'		{ $$ = $2; }
       | '(' expr error		{ error("Expecting closing parenthesis", "E006", @3); }
       | error			{ error("Missing opening parenthesis",   "E004", @1); }
       ;

braces : '{' in_if '}'		{ $$ = $2; }
       | '{' in_if error	{ error("Missing closing brace", "E008", @3); }
       | error			{ error("Missing opening brace", "E007", @1); }
       ;

in_if : expr ';'		{ $$ = $1; }
      | expr ';' in_if		{ $$ = Call("and", List($1, List($3, 0)), @1, @3); }
      | expr error		{ error("Missing semi-colon",     "E001", @2);     }
      | expr ')'		{ error("Unbalanced parentheses", "E002", @2);     }
      | IF                      { error("Nested if clauses is not allowed", "E019", @1); }
      ;

var: cont '$' name              { $$ = Call("lookupcnt", List($3, List($1, 0)), @1, @3); }
   | cont ':' name              { $$ = Call("lookup",    List($3, List($1, 0)), @1, @3); }
   | cont '$' error             { error("Missing counter variable identifier", "E015", @3); }
   | cont ':' error             { error("Missing string variable identifier", "E016", @3); }
   | cont ':' ':'               { error("Nested variable references must be enclosed in parentheses", "E018", @3); }
   | cont ':' '$'               { error("Nested variable references must be enclosed in parentheses", "E018", @3); }
   | cont '$' ':'               { error("Nested variable references must be enclosed in parentheses", "E018", @3); }
   | cont '$' '$'               { error("Nested variable references must be enclosed in parentheses", "E018", @3); }
   ;

cont:                           { $$ = Str("global", 0, 0, 0, 0); }
    | ID                        { $$ = Str($1, @1, @1); }
    | STR                       { $$ = Str($1, @1, @1); }
    | '(' expr ')'              { $$ = $2;              }
    | '(' expr error		{ error("Expecting closing parenthesis", "E006", @3); }
    ;

name: ID                        { $$ = Str($1, @1, @1); }
    | STR                       { $$ = Str($1, @1, @1); }
    | '(' expr ')'              { $$ = $2;              }
    | '(' expr error		{ error("Expecting closing parenthesis", "E006", @3); }
    ;

%%

void yyerror(char *s) {
     //fprintf(stderr, "Parser: %s\n", s);
}

void error(const char *m, const char *id, YYLTYPE pos) {
     throw Parse::Error(m, pos.first_line, pos.first_column, getControl()->filename, id);
}

NodeInfo::Node* Parse::parse(const char *s, NodeInfo::Alloc *alloc, StrPool &pool, int line, int column, const char *filename) {
     Control ctrl(s, alloc, pool, line, column, filename);
     setControl(&ctrl);
     yyrestart(0);
     yyparse((void *) &ctrl);
     return getControl()->root;
}

int do_input(char *buf, int max) {
     int len = std::strlen(Parse::getControl()->toParse);
     int res = (max < len - Parse::getControl()->scan_pos ?
	        max : len - Parse::getControl()->scan_pos);
     if (res > 0) {
	memcpy(buf, &Parse::getControl()->toParse[Parse::getControl()->scan_pos], res);
	Parse::getControl()->scan_pos += res;
     }
     return res;
}
