/***************************************************************************
                                  execute.hpp
                             -------------------
                               Tue Feb 4 2003
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef EXECUTE_H
#define EXECUTE_H

#include <string>
#include <sstream>

#include "types.hpp"
#include "evexp.hpp"
#include "parser/node.hpp"
#include "parser/error.hpp"

/*
 *  queue() will queue the object for execution and, 
 *    at program halt, frees the memory.
 *  execute() executes the queue
 *
 */
class Execute_base;
void queue(Execute_base *new_exec, bool outmost, bool constant = false);
bool execute();
void set_skip(bool b);
void set_long_skip(bool b);
void reset_execute();


class Execute_base {
public:
    virtual ~Execute_base() {};
    virtual bool execute() = 0;
    virtual std::string desc() const = 0;
public:
    NodeInfo::Node* node;
};


template<class R>
class Execute : public Execute_base
{
public:/*fixes bug in gcc 3.4.4, no public: scope should be used*/
    typedef R return_type;
    typedef R *ptr_type;

public:
    return_type  value() const       { return r; }
    bool         execute() = 0;

public:
    return_type *get()               { return &r; }

protected:
    void         set(return_type t)  { r = t; }
    bool         ret() const;
    std::string  descr(std::string s) const { return type_name(Type2type<R>()) + " " + s; }

private:
    return_type  r;
};

template<class R>
class Execute<R &> : public Execute_base
{
public:/*fixes bug in gcc 3.4.4, no public: scope should be used*/
    typedef R return_type;
    typedef R * &ptr_type;

public:
    return_type   &value() const       { return *r; }
    bool           execute() = 0;

public:
    return_type * &get()               { return r; }

protected:
    void           set(return_type &t) { r = &t; }
    bool           ret() const;
    std::string    descr(std::string s) const { return type_name(Type2type<R &>()) + " " + s; }

private:
    return_type   *r;
};

template<class R> 
inline bool Execute<R>::ret() const { return true; }
template<class R> 
inline bool Execute<R &>::ret() const { return true; }
template<> 
inline bool Execute<bool>::ret() const { return r; }



template<class R>
class Execute_const : public Execute<R>
{
public:/*fixes bug in gcc 3.4.4, no public: scope should be used*/
    typedef R return_type;

public:
    Execute_const(return_type t, NodeInfo::Node *oper) { set(t); this->node = oper; }

    bool execute() {
        try {
            return this->ret();
        }
        catch (const EvExp &e) {
            std::ostringstream strm;
            strm << e.error << ": " << e.data;
            throw Parse::Error(strm.str(), this->node);
        }
    }
    std::string desc() const { return this->descr("const"); }
};


template<class F, class R, class T1>
class Execute_func1 : public Execute<R>
{
    typedef F function_type;

public:
    Execute_func1(F g, T1 &i1, NodeInfo::Node *oper)
        : f(g), t1(i1.get())
    { this->node = oper; }

    bool execute() { 
        try {
            set(f(*t1)); 
            return this->ret(); 
        }
        catch (const EvExp &e) {
            std::ostringstream strm;
            strm << e.error << ": " << e.data;
            throw Parse::Error(strm.str(), this->node);
        }
    }
    std::string desc() const { return this->descr("func1"); }

private:
    function_type f; 
    typename T1::ptr_type t1;
};


template<class F, class R, class T1, class T2>
class Execute_func2 : public Execute<R>
{
    typedef F function_type;

public:
    Execute_func2(F g, T1 &i1, T2 &i2, NodeInfo::Node *oper)
        : f(g), t1(i1.get()), t2(i2.get())
    { this->node = oper; }

    bool execute() { 
        try {
            set(f(*t1, *t2)); 
            return this->ret(); 
        }
        catch (const EvExp &e) {
            std::ostringstream strm;
            strm << e.error << ": " << e.data;
            throw Parse::Error(strm.str(), this->node);
        }
    }
    std::string desc() const { return this->descr("func2"); }

private:
    function_type f;
    typename T1::ptr_type t1;
    typename T2::ptr_type t2;
};


template<class F, class R, class T1, class T2, class T3>
class Execute_func3 : public Execute<R>
{
    typedef F function_type;

public:
    Execute_func3(F g, T1 &i1, T2 &i2, T3 &i3, NodeInfo::Node *oper)
        : f(g), t1(i1.get()), t2(i2.get()), t3(i3.get())
    { this->node = oper; }

    bool execute() { 
        try {
            set(f(*t1, *t2, *t3)); 
            return this->ret(); 
        }
        catch (const EvExp &e) {
            std::ostringstream strm;
            strm << e.error << ": " << e.data;
            throw Parse::Error(strm.str(), this->node);
        }
    }
    std::string desc() const { return this->descr("func3"); }

private:
    function_type f;
    typename T1::ptr_type t1;
    typename T2::ptr_type t2;
    typename T3::ptr_type t3;
};


template<class F, class R, class T1, class T2, class T3, class T4>
class Execute_func4 : public Execute<R>
{
    typedef F function_type;

public:
    Execute_func4(F g, T1 &i1, T2 &i2, T3 &i3, T4 &i4, NodeInfo::Node *oper)
        : f(g), t1(i1.get()), t2(i2.get()), t3(i3.get()), t4(i4.get())
    { this->node = oper; }

    bool execute() { 
        try {
            set(f(*t1, *t2, *t3, *t4)); 
            return this->ret(); 
        }
        catch (const EvExp &e) {
            std::ostringstream strm;
            strm << e.error << ": " << e.data;
            throw Parse::Error(strm.str(), this->node);
        }
    }
    std::string desc() const { return this->descr("func4"); }

private:
    function_type f;
    typename T1::ptr_type t1;
    typename T2::ptr_type t2;
    typename T3::ptr_type t3;
    typename T4::ptr_type t4;
};


template<class F, class R, class T1, class T2, class T3, class T4, class T5>
class Execute_func5 : public Execute<R>
{
    typedef F function_type;

public:
    Execute_func5(F g, T1 &i1, T2 &i2, T3 &i3, T4 &i4, T5 &i5, NodeInfo::Node *oper)
        : f(g), t1(i1.get()), t2(i2.get()), t3(i3.get()), 
          t4(i4.get()), t5(i5.get())
    { this->node = oper; }

    bool execute() { 
        try {
            set(f(*t1, *t2, *t3, *t4, *t5)); 
            return this->ret(); 
        }
        catch (const EvExp &e) {
            std::ostringstream strm;
            strm << e.error << ": " << e.data;
            throw Parse::Error(strm.str(), this->node);
        }
    }
    std::string desc() const { return this->descr("func5"); }

private:
    function_type f;
    typename T1::ptr_type t1;
    typename T2::ptr_type t2;
    typename T3::ptr_type t3;
    typename T4::ptr_type t4;
    typename T5::ptr_type t5;
};


template<class F, class R, class T1, class T2, class T3, class T4, class T5, class T6>
class Execute_func6 : public Execute<R>
{
    typedef F function_type;

public:
    Execute_func6(F g, T1 &i1, T2 &i2, T3 &i3, T4 &i4, T5 &i5, T6 &i6, NodeInfo::Node *oper)
        : f(g), t1(i1.get()), t2(i2.get()), t3(i3.get()), 
          t4(i4.get()), t5(i5.get()), t6(i6.get())
    { this->node = oper; }

    bool execute() { 
        try {
            set(f(*t1, *t2, *t3, *t4, *t5, *t6)); 
            return this->ret(); 
        }
        catch (const EvExp &e) {
            std::ostringstream strm;
            strm << e.error << ": " << e.data;
            throw Parse::Error(strm.str(), this->node);
        }
    }
    std::string desc() const { return this->descr("func6"); }

private:
    function_type f;
    typename T1::ptr_type t1;
    typename T2::ptr_type t2;
    typename T3::ptr_type t3;
    typename T4::ptr_type t4;
    typename T5::ptr_type t5;
    typename T6::ptr_type t6;
};


template<class F, class R, class T1, class T2, class T3, class T4, class T5, class T6, class T7>
class Execute_func7 : public Execute<R>
{
    typedef F function_type;

public:
    Execute_func7(F g, T1 &i1, T2 &i2, T3 &i3, T4 &i4, T5 &i5, T6 &i6, T7 &i7, NodeInfo::Node *oper)
        : f(g), t1(i1.get()), t2(i2.get()), t3(i3.get()), 
          t4(i4.get()), t5(i5.get()), t6(i6.get()), t7(i7.get())
    { this->node = oper; }

    bool execute() { 
        try {
            set(f(*t1, *t2, *t3, *t4, *t5, *t6, *t7)); 
            return this->ret(); 
        }
        catch (const EvExp &e) {
            std::ostringstream strm;
            strm << e.error << ": " << e.data;
            throw Parse::Error(strm.str(), this->node);
        }
    }
    std::string desc() const { return this->descr("func7"); }

private:
    function_type f;
    typename T1::ptr_type t1;
    typename T2::ptr_type t2;
    typename T3::ptr_type t3;
    typename T4::ptr_type t4;
    typename T5::ptr_type t5;
    typename T6::ptr_type t6;
    typename T7::ptr_type t7;
};


template<class F, class R, class T1, class T2, class T3, class T4, class T5, class T6, class T7, class T8>
class Execute_func8 : public Execute<R>
{
    typedef F function_type;

public:
    Execute_func8(F g, T1 &i1, T2 &i2, T3 &i3, T4 &i4, T5 &i5, T6 &i6, T7 &i7, T8 &i8, NodeInfo::Node *oper)
        : f(g), t1(i1.get()), t2(i2.get()), t3(i3.get()), 
          t4(i4.get()), t5(i5.get()), t6(i6.get()), 
          t7(i7.get()), t8(i8.get())
    { this->node = oper; }

    bool execute() { 
        try {
            set(f(*t1, *t2, *t3, *t4, *t5, *t6, *t7, *t8)); 
            return this->ret(); 
        }
        catch (const EvExp &e) {
            std::ostringstream strm;
            strm << e.error << ": " << e.data;
            throw Parse::Error(strm.str(), this->node);
        }
    }
    std::string desc() const { return this->descr("func8"); }

private:
    function_type f;
    typename T1::ptr_type t1;
    typename T2::ptr_type t2;
    typename T3::ptr_type t3;
    typename T4::ptr_type t4;
    typename T5::ptr_type t5;
    typename T6::ptr_type t6;
    typename T7::ptr_type t7;
    typename T8::ptr_type t8;
};


template<class F, class R, class T1, class T2, class T3, class T4, class T5, class T6, class T7, class T8, class T9>
class Execute_func9 : public Execute<R>
{
    typedef F function_type;

public:
    Execute_func9(F g, T1 &i1, T2 &i2, T3 &i3, T4 &i4, T5 &i5, T6 &i6, T7 &i7, T8 &i8, T9 &i9, NodeInfo::Node *oper)
        : f(g), t1(i1.get()), t2(i2.get()), t3(i3.get()), 
          t4(i4.get()), t5(i5.get()), t6(i6.get()), 
          t7(i7.get()), t8(i8.get()), t9(i9.get())
    { this->node = oper; }

    bool execute() { 
        try {
            set(f(*t1, *t2, *t3, *t4, *t5, *t6, *t7, *t8, *t9)); 
            return this->ret(); 
        }
        catch (const EvExp &e) {
            std::ostringstream strm;
            strm << e.error << ": " << e.data;
            throw Parse::Error(strm.str(), this->node);
        }
    }
    std::string desc() const { return this->descr("func9"); }

private:
    function_type f;
    typename T1::ptr_type t1;
    typename T2::ptr_type t2;
    typename T3::ptr_type t3;
    typename T4::ptr_type t4;
    typename T5::ptr_type t5;
    typename T6::ptr_type t6;
    typename T7::ptr_type t7;
    typename T8::ptr_type t8;
    typename T9::ptr_type t9;
};




#endif // EXECUTE_H
