/***************************************************************************
                                  insrc.hpp
                             -------------------
                               Mon Sep 2 2002
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef INSRC_H
#define INSRC_H

#include <string>
#include <vector>
#include <fstream>

#include <xercesc/sax2/Attributes.hpp>
#include <xercesc/sax2/DefaultHandler.hpp>
#include <xercesc/sax2/SAX2XMLReader.hpp>
#include <xercesc/sax2/XMLReaderFactory.hpp>

#include "evutil.hpp"
#include "types.hpp"

XERCES_CPP_NAMESPACE_USE

#define INI_CHUNK 100 /* Initial size in bytes of the 
			 plain text data source buffer */
#define ADD_CHUNK 50  /* Increment step size in bytes of the 
			 plain text data source buffer */
#define MAX_CHUNK 250 /* Maximal size in bytes of the 
			 plain text data source buffer */

/**
 * Structure that specfies relevant data nodes in a xml input document 
 */
struct EvNode {  
     /* Name of node*/
     std::string name;
     
     /* Name of nodes containing data set pieces, child nodes to the current node. 
	Reledata is empty if it has no relevant child nodes */
     DataVector reledata;  
};

typedef std::vector<EvNode> EvNodeVector;

/**
 * A filter object, used for string substitution
 */
class EvFilter {
public:
     EvFilter(const std::string &filename);
     
     /* Configures the filter, specify the data format, this
	function also load the filter data into memory */
     void configurate(const std::string &idelim, const std::string &edelim);
     
     /* Returns the configuration status of the filter */
     bool isConfigurated() const;
     
     /* Returns a reference to the DataMap which contains substitution data */ 
     const DataMap& getFiltdata() const;

     ~EvFilter();
private:
     /* Configuration status */
     bool configurated_;

     /* The file associated with the filter */
     std::string filename_;

     /* Datamap which contains the string mapped to the substitution string */
     DataMap filterdata_;
};

/**
 * The abstract interface for input data sources 
 */
class DSource {
  public:
     DSource() { }

     /* Configuration function for data sources which 
	require	addtional configuration parameters */
     virtual void configurate(const DataMap &meta) = 0;

     /* Used by a reader to report how much of the data
	buffer it has read and recognized as relevant */
     virtual void dataSeen(const int size) = 0;

     /* Returns the filename associated with the input data source */     
     virtual std::string getAssociatedFile() const = 0;

     /* Returns a piece of data from the databuffer */
     virtual std::string getData() = 0;

     /* Returns the configuration state of the buffer */
     virtual bool isConfigurated() const = 0;

     /* Returns exhaustion status, the buffer is exhausted if
	the associated file has reached eof and the buffer is
	empty ofrelevant data */
     virtual bool isExhausted() const = 0;
     
     virtual ~DSource() { }
};

/**
 * Plain text input data source 
 */
class ISource : public DSource {
  public:
    ISource(const std::string &filename);

     /* This input sources is always configurated, 
	calling this function is not necessary. */
    void configurate(const DataMap &meta);
    void dataSeen(const int size);
    std::string getAssociatedFile() const;
    std::string getData();
    bool isConfigurated() const;
    bool isExhausted() const;

    virtual ~ISource();
  protected:
     ISource();

    /* Read data from input file */ 
    virtual bool readChunkFromFile();

    /*  Data size of the chunk that will be read from the file */
    int chunkSize_;

    /* Set when end of file is reached */
    bool eofreached_;

    /* Set when end of file is reached and there is no relevant data in the buffer */
    bool exhausted_;

    /* File associated with the buffer */
    std::string filename_;

    /* input file stream used to read the file */
    std::ifstream inputFile_;

    /* Data buffer which contains a piece of the input file */
    std::string searchBuffer_;
};

/**
 * STDIN input source
 */
class StdinSource : public ISource {
public:
     StdinSource();
     ~StdinSource();
protected:
     bool readChunkFromFile();
};

class XSource;

/**
 * Implementation of a SAX2 parse event handler 
 */
class XsHandler : public DefaultHandler {
public:
     XsHandler(XSource *src);
     
     /* Called by the parser when it sees the start of a element. */
     void startElement (const XMLCh *const uri,const XMLCh *const localname,const XMLCh *const qname,const Attributes &attrs);
     
     /* Called by the parser when it sees the end of a element. */
     void endElement (const XMLCh *const uri,const XMLCh *const localname,const XMLCh *const qname);
     
     /* Called by the parser when it sees text data (text node). */
     void characters(const XMLCh* const chars, const unsigned int length);
     
     /* Called by the parser when a error occurs. */
     void error(const SAXParseException &exception);
     
     /* Called by the parser when a warning occurs. */
     void warning (const SAXParseException &exception);
     
     ~XsHandler();
private:
     /* A pointer to its associated xml input source, used for callback purposes */
     XSource *xsrc_;
};

/**
 * XML input data source 
 */
class XSource : public DSource {
public:
     XSource(const std::string &filename);
     
     /* Takes data, which contains information on which node house relevant data */
     void configurate(const DataMap &meta);

     void dataSeen(const int size);
     bool isConfigurated() const;
     bool isExhausted() const;
     std::string getAssociatedFile() const;
     std::string getData();
     
     /* Callback function which is called by the parse event handler 
	it gets a its characters function gets called */
     void sawChars(const XMLCh *const chars);
     
     /* Callback function which is called by the parse event handler 
	it gets a its startElement function gets called */
     void sawElementStart(const XMLCh *const name,const Attributes &attrs);
     
     /* Callback function which is called by the parse event handler 
	it gets a its endElement function gets called */
     void sawElementEnd(const XMLCh *const name);
     
     
     ~XSource();
private:
     /* Adds data to the internal buffer */
     void addBufferData(const std::string &data);

     /* Read a piece of relevant data from the input file */ 
     bool readChunkFromFile();

     bool busy_;
     bool configurated_;
     bool eofreached_;
     bool exhausted_;
     std::string filename_;
     XsHandler *handler_;

     /* Structure that specfies relevant data nodes in a xml input document*/
     EvNodeVector nodes_;

     SAX2XMLReader *parser_;

     /* Scantoken used by the parser to perform progressive parsing */
     XMLPScanToken scantoken_;

     std::string searchBuffer_;
     bool readnexttnode_;
     int state_;
};

#endif
