/***************************************************************************
                                  main.cpp
                             -------------------
                               Mon Sep 2 2002
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "main.hpp"

#include <iostream>
#include <fstream>
#include <cstdlib>  // rand
#include <ctime>    // time

#include "evexp.hpp"
#include "evinit.hpp"
#include "parseargs.hpp"
#include "internal.hpp" // outfilexml, infileplain, infilexml, filterplain

#include "parser/error.hpp"

void print_help(const char *name) {
    std::cout 
        << "\nUsage: " << name << " -c <filename> [ options... ]\n"
        << "Options:\n"
        << "  -c <configfile>\tSpecifies which configuration file to use\n"
        << "  -o <id>:<outfile>\tLet output identifier <id> write to <outfile>\n"
        << "  -i <id>:<infile>\tLet input identifier <id> read from text file <infile>\n"
        << "  -x <id>:<infile>\tLet input identifier <id> read from xml-file <infile>\n"
        << "  -s <id>\t\tLet input identifier <id> read from standard input\n"
        << "  -t <id>:<filterfile>\tLet filter identifier <id> use file <filterfile>\n"
        << "  -h\t\t\tDisplay this help message\n"
        << "  -v\t\t\tPrint version information\n";
}

void print_version() {
    std::cout
        << "\nEval: A Generic Evaluation Tool for Natural Language Applications\n"
        << "Version 0.2\n"
        << "2003 (C) licensed under GNU GPL v2\n\n"
        << "This program uses components from Xerces 2.1 and Boost Regex\n\n";
}

void locate_error(const std::string& filename, int row, int column) {
    try {
        std::ifstream file(filename.c_str());
        if (!file) {
            return;
        }
        std::string line;
        int i;
        for (i = 1; i < row - 2; ++i) {
            getline(file, line);
        }
        for ( ; i <= row; ++i) {
            getline(file, line);
            std::cerr << line << std::endl;
        }
        for (int j = 0; j < column; ++j) {
            if (line[j] == '\t') std::cerr << "\t";
            else std::cerr << " ";
        }
        std::cerr << "^\n\n";
    }
    catch (...) {
        //std::cerr << "Can't locate\n";
    }
}


int main(int argc, char* argv[]) {
    srand(time(0));
    try {
        ParseArgs args(argc, argv);
        if (args.exists("-h")) {
            print_help(argv[0]);
            return 0;
        }
        if (args.exists("-v")) {
            print_version();
            return 0;
        }
        if (!args.exists("-c")) {
            std::cout << "No configuration file specified\n";
            print_help(argv[0]);
            return 1;
        }
        try {
            XMLEngineStart();
            DoConfiguration(args.value("-c"));
            for (ParseArgs::CommandMap::const_iterator iter = args.commandMap.begin();
                 iter != args.commandMap.end(); ++iter) {
                if (iter->first == "-o") {
                    ParseArgs::IdFile idfile = args.split("-o");
                    outfilexml(idfile.first, idfile.second);
                }
                if (iter->first == "-i") {
                    ParseArgs::IdFile idfile = args.split("-i");
                    infileplain(idfile.first, idfile.second);
                }
                if (iter->first == "-x") {
                    ParseArgs::IdFile idfile = args.split("-x");
                    infilexml(idfile.first, idfile.second);			 
                }
                if (iter->first == "-s") {
                    stdinplain(iter->second);
                }
                if (iter->first == "-t") {
                    ParseArgs::IdFile idfile = args.split("-t");
                    filterplain(idfile.first, idfile.second);
                }
            }
            DoEvaluation();
            XMLEngineStop();
        }
        catch (EvExp &e) {
            std::cerr << "\n"
                      << e.section << "\n" 
                      << e.error   << "\n" 
                      << e.data << std::endl << std::endl;
            return 1;
        }
        catch (const Parse::Error &e) {
            std::cerr << "\n";
            if (e.filename.length() > 0) {
                std::cerr << "In file " << e.filename << ":\n";
            }
            std::cerr << "At row " << e.row << ", column " << e.column << ":\n";
            std::cerr << e.message << "\n\n";
            locate_error(e.filename, e.row, e.column);
            return 1;
        }	  
    }
    catch (const char *e) {	  
        std::cerr << "\nCommand line parse error: " << e << "\n";
        print_help(argv[0]);
        return 1;
    }
    catch (std::exception &e) {
        std::cerr << std::endl;
        std::cerr << "An unexpected error occurred. Exception description:\n";
        std::cerr << e.what() << std::endl << std::endl;
        return 1;
    }
    catch (...) {
        std::cerr << "\nAn Unexpected Error Occurred\n";
        return 1;
    }   

    return 0;
}
