/***************************************************************************
                                  parseargs.hpp
                             -------------------
                               Sat Oct 19 2002
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef PARSEARGS_H
#define PARSEARGS_H

#include <string>
#include <map>

/**
 * ParseArgs parses command line arguments and
 * places them in an easy access container
 */
class ParseArgs {
public:
     typedef std::multimap<std::string, std::string> CommandMap;
     typedef std::pair<std::string, std::string> IdFile;

     /* The constructor does all the parsing */
     ParseArgs(int argc, char *argv[]) {
	  std::string param = "";
	  std::string value = "";
	  if (argc > 1) {
	       if (*argv[1] != '-') {
		    throw "First command line parameter must be an option.";
	       }
	  }	  
	  for (int i = 0; argv[i]; ++i) {
	       char *arg = argv[i];
	       if (*arg == '-') {
		    if (param != "") {
			 commandMap.insert(std::make_pair(param, value));
		    }
		    switch (*++arg) {
		    case 'i':
			 param = "-i";
			 value = ++arg;
			 break;
		    case 'x':
			 param = "-x";
			 value = ++arg;
			 break;
		    case 's':
			 param = "-s";
			 value = ++arg;
			 break;
		    case 'o':
			 param = "-o";
			 value = ++arg;
			 break;
		    case 'c':
		    case 'f':
			 param = "-c";
			 value = ++arg;
			 break;
		    case 't':
			 param = "-t";
			 value = ++arg;
			 break;
		    case 'h':
			 param = "-h";
			 value = ++arg;
			 break;
		    case 'v':
			 param = "-v";
			 value = ++arg;
			 break;
		    case '\0':
			 throw "No option specified after '-' sign.";
			 break;
		    default:
			 throw std::string(std::string("Unknown parameter '-") + *arg + "'.").c_str();
			 break;
		    }
	       }
	       else {
		    value += arg;
	       }
	  }
	  if (param != "") {
	       commandMap.insert(std::make_pair(param, value));
	  }
     }

     /* Checks if an command line argument exists */
     bool exists(std::string key) const {
	  return commandMap.find(key) != commandMap.end();
     }

     /* Returns the value of a specified option */
     std::string value(std::string key) {
	  CommandMap::iterator iter = commandMap.find(key);
	  if (iter != commandMap.end()) {
	       return iter->second;
	  }
	  return "";
     }

     /* Splits the "id:value" value into two separate values */
     IdFile split(std::string line) {
	  line = value(line);
	  std::string::size_type idx = line.find(':');
	  if (idx == std::string::npos) {
	       throw "No ':' found separating identifier and file name";
	  }
	  if (idx == 0) {
	       throw "No identifier was specified";
	  }
	  if (idx == line.length()) {
	       throw "No filename was specified";
	  }
	  return std::make_pair(line.substr(0, idx), line.substr(idx+1, line.length()-idx));
     }
public:
     CommandMap commandMap;
};

#endif
