/******************************************************************************

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

******************************************************************************/

#include "alloc.hpp"

namespace NodeInfo {

     Alloc::~Alloc() {
	  for (unsigned int i = 0; i < mem.size(); ++i) {
	       switch (mem[i].type) {
	       case NodeInfo::T_CALL: delete mem[i].args;
	       default:
		    break;
	       }
	  }
     }

     Node* Alloc::alloc() {
	  Node node;
	  node.type = NodeInfo::T_STR;
	  node.cbegin = 0;
	  node.cend = 0;
	  node.lbegin = 0;
	  node.lend = 0;
	  node.args = 0;
	  node.name = 0;
	  node.filename = 0;
	  mem.push_back(node);
	  return &(*mem.rbegin());
     }

     Node* Alloc::alloc_str(const char *s, const char *name) {
	  Node *n = alloc();
	  n->type = NodeInfo::T_STR;
	  n->data.str = pool[s];
	  n->filename = pool[name];
	  return n;
     }

     Node* Alloc::alloc_int(int i, const char *name) {
	  Node *n = alloc();	  
	  n->type = NodeInfo::T_INT;
	  n->data.integer = i;
	  n->filename = pool[name];
	  return n;
     }

     Node* Alloc::alloc_dbl(double d, const char *name) {
	  Node *n = alloc();	  
	  n->type = NodeInfo::T_DBL;
	  n->data.dbl = d;
	  n->filename = pool[name];
	  return n;
     }

     Node* Alloc::alloc_tmplref(int i, const char *name) {
	  Node *n = alloc();	  
	  n->type = NodeInfo::T_TMPLREF;
	  n->data.integer = i;
	  n->filename = pool[name];
	  return n;
     }

     Node* Alloc::alloc_bool(bool b, const char *name) {
	  Node *n = alloc();	  
	  n->type = NodeInfo::T_BOOL;
	  n->data.boolean = b;
	  n->filename = pool[name];
	  return n;
     }

     Node* Alloc::alloc_call(const char *s, Node* l, const char *name) {
	  Node *n = alloc();
	  n->type = NodeInfo::T_CALL;
	  n->name = pool[s];
	  n->filename = pool[name];

	  n->args = l->args;
	  return n;
     }

     Node* Alloc::alloc_list(Node* h) {
	  Node *n =  alloc();
	  n->type = NodeInfo::T_LIST;
	  n->args = new std::vector<Node*>();
	  n->args->push_back(h);
	  return n;
     }

     Node* Alloc::alloc_line(Node* h, const char *name) {
	  Node *n = alloc();	  
	  n->type = NodeInfo::T_LINE;
	  n->filename = pool[name];
	  n->args = new std::vector<Node*>();
	  n->args->push_back(h);
	  return n;
     }

     Node* Alloc::clone_subtree(Node* node) {
	  if (!node) {
               return 0;
	  }
	  Node *ret;
	  switch (node->type) {
	  case NodeInfo::T_STR:
	  case NodeInfo::T_INT:
	  case NodeInfo::T_DBL:
	  case NodeInfo::T_BOOL:
	       ret = node;
	       break;
	  case NodeInfo::T_CALL:
	  case NodeInfo::T_LINE:
	       ret = alloc();
	       break;
	  case NodeInfo::T_TMPLREF:
	       ret = alloc();
	       ret->data.integer = node->data.integer;
	       break;
	  case NodeInfo::T_LIST:
	       std::cerr << "LIST found while cloning!\n";
	       break;
	  default:
	       std::cerr << "Error while cloning!\n";
	       break;
	  }
	  ret->type   = node->type;
	  ret->cbegin = node->cbegin;
	  ret->cend   = node->cend;
	  ret->lbegin = node->lbegin;
	  ret->lend   = node->lend;
	  ret->name   = node->name;
	  ret->filename = node->filename;

	  if (node->args) {
	       ret->args = new std::vector<Node*>();
	       for (NodeVector::iterator iter = node->args->begin(); iter != node->args->end(); ++iter) {
		    ret->args->push_back(clone_subtree(*iter));
	       }
	  }
	  return ret;
     }


}
