/******************************************************************************

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

******************************************************************************/

#include "control.hpp"

#include <algorithm>

namespace Parse {

     static Control *control = 0;

     Control* getControl() {
	  return control;
     }

     void setControl(Control *c) {
	  control = c;
     }
     
     using namespace NodeInfo;

     Node* Str(const char *s, int lbegin, int lend, int cbegin, int cend) {
	  const char* name = control->filename;
	  Node *n = control->alloc->alloc_str(s, name);
	  n->lbegin = lbegin;
	  n->lend = lend;
	  n->cbegin = cbegin;
	  n->cend = cend;
	  return n;
     }

     Node* Str(const char *s, YYLTYPE begin, YYLTYPE end) {
	  const char* name = control->filename;
	  Node *n   = control->alloc->alloc_str(s, name);
	  n->lbegin = begin.first_line;
	  n->lend   = end.last_line;
	  n->cbegin = begin.first_column;
	  n->cend   = end.last_column;
	  return n;
     }
     
     Node* Int(int i, YYLTYPE begin, YYLTYPE end) {
	  const char* name = control->filename;
	  Node *n   = control->alloc->alloc_int(i, name);
	  n->lbegin = begin.first_line;
	  n->lend   = end.last_line;
	  n->cbegin = begin.first_column;
	  n->cend   = end.last_column;
	  return n;
     }

     Node* Dbl(double d, YYLTYPE begin, YYLTYPE end) {
	  const char* name = control->filename;
	  Node *n   = control->alloc->alloc_dbl(d, name);
	  n->lbegin = begin.first_line;
	  n->lend   = end.last_line;
	  n->cbegin = begin.first_column;
	  n->cend   = end.last_column;
	  return n;
     }

     Node* TmplRef(int i, YYLTYPE begin, YYLTYPE end) {
	  const char* name = control->filename;
	  Node *n   = control->alloc->alloc_tmplref(i, name);
	  n->lbegin = begin.first_line;
	  n->lend   = end.last_line;
	  n->cbegin = begin.first_column;
	  n->cend   = end.last_column;
	  return n;
     }

     Node* Bool(bool b, YYLTYPE begin, YYLTYPE end) {
	  const char* name = control->filename;
	  Node *n   = control->alloc->alloc_bool(b, name);
	  n->lbegin = begin.first_line;
	  n->lend   = end.last_line;
	  n->cbegin = begin.first_column;
	  n->cend   = end.last_column;
	  return n;
     }

     Node* Call(const char *s, Node* l, int lbegin, int lend, int cbegin, int cend) {
	  const char* name = control->filename;
	  Node *n = control->alloc->alloc_call(s, l, name);
	  n->lbegin = lbegin;
	  n->lend = lend;
	  n->cbegin = cbegin;
	  n->cend = cend;
	  return n;
     }

     Node* Call(const char *s, Node* l, YYLTYPE begin, YYLTYPE end) {
	  const char* name = control->filename;
	  Node *n   = control->alloc->alloc_call(s, l, name);
	  n->lbegin = begin.first_line;
	  n->lend   = end.last_line;
	  n->cbegin = begin.first_column;
	  n->cend   = end.last_column;
	  return n;
     }

     Node* List(Node* h, Node* t) {
	  Node *n = 0;
	  if (!t) {
	       n = control->alloc->alloc_list(h);
	       n->lend = h->lend;
	       n->cend = h->cend;
	  }
	  else {
	       n = t;
	       n->args->push_back(h);
	       n->lend = t->lend;
	       n->cend = t->cend;
	  }
	  n->lbegin = h->lbegin;
	  n->cbegin = h->cbegin;
	  return n;
     }

     Node* Line(Node* h, Node* t) {
	  const char* name = control->filename;
	  Node *n = 0;
	  if (!t) {
	       n = control->alloc->alloc_line(h, name);
	       n->lend = h->lend;
	       n->cend = h->cend;
	  }
	  else {
	       n = t;
	       n->args->push_back(h);
	       n->lend = t->lend;
	       n->cend = t->cend;
	  }
	  n->lbegin = h->lbegin;
	  n->cbegin = h->cbegin;
	  return n;
     }

     void result(Node* n) {
	  reverse_lists(n);
	  control->root = n;
     }

     void reverse_lists(Node *node) {
	  if (!node) {
	       return;
	  }
	  switch (node->type) {
	  case T_CALL:
	  case T_LINE:
	       for (std::vector<Node*>::iterator iter = node->args->begin();
		    iter != node->args->end(); ++iter) {
		    reverse_lists(*iter);
	       }
	       std::reverse(node->args->begin(), node->args->end());
	       break;
	  case T_STR:
	  case T_INT:
	  case T_DBL:
	  case T_TMPLREF:
	  case T_BOOL:
	       break;
	  default:
	       fprintf(stderr, "Error in reverse!\n");
	       break;
	  }
     }

     void print_tree(Node* node) {
	  if (!node) {
	       fprintf(stderr, "No node here!\n");
	       return;
	  }
	  switch (node->type) {
	  case T_STR:
	       fprintf(stderr, "\"%s\"", node->data.str);
	       break;
	  case T_INT:
	       fprintf(stderr, "%d", node->data.integer);
	       break;
	  case T_DBL:
	       fprintf(stderr, "%f", node->data.dbl);
	       break;
	  case T_TMPLREF:
	       fprintf(stderr, "#%d", node->data.integer);
	       break;
	  case T_BOOL:
	       node->data.boolean ? fprintf(stderr, "true") : fprintf(stderr, "false");
	       break;
	  case T_CALL:
	       fprintf(stderr, "%s(", node->name);
	       for (unsigned int i = 0; i < node->args->size(); ++i) {
		    print_tree(node->args->at(i));
		    if (i < node->args->size() - 1) fprintf(stderr, ",");
	       }
	       fprintf(stderr, ")");
	       break;
	  case T_LINE:
	       for (unsigned int i = 0; i < node->args->size(); ++i) {
		    print_tree(node->args->at(i));
		    fprintf(stderr, ";\n");
	       }
	       break;
	  default:
	       fprintf(stderr, "Error in tree: '%d'\n", node->type);
	       return;
	  }
     }

     void print_parse(Node* node) {
	  if (!node) {
	       fprintf(stderr, "NULL");
	       return;
	  }
	  switch (node->type) {
	  case T_STR:
	       fprintf(stderr, "STR(l%d,b%d,e%d,%s)", node->lbegin, node->cbegin, node->cend, node->data.str);
	       break;
	  case T_INT:
	       fprintf(stderr, "INT(l%d,b%d,e%d,%d)", node->lbegin, node->cbegin, node->cend, node->data.integer);
	       break;
	  case T_DBL:
	       fprintf(stderr, "DBL(l%d,b%d,e%d,%f)", node->lbegin, node->cbegin, node->cend, node->data.dbl);
	       break;
	  case T_TMPLREF:
	       fprintf(stderr, "TMPLREF(l%d,b%d,e%d,#%d)", node->lbegin, node->cbegin, node->cend, node->data.integer);
	       break;
	  case T_BOOL:
	       fprintf(stderr, "BOOL(l%d,b%d,e%d,", node->lbegin, node->cbegin, node->cend);
	       node->data.boolean ? fprintf(stderr, "true") : fprintf(stderr, "false");
	       fprintf(stderr, ")");
	       break;
	  case T_CALL:
	       if (strlen(node->name) == 0) {
		    fprintf(stderr, "WARNING: adding 1 to pointer!!!");
		    ++(node->name);
		    fprintf(stderr, " (%s)\n", node->name);
	       }
	       fprintf(stderr, "CALL(l%d,b%d,e%d,%s[%d,%p],", node->lbegin, node->cbegin, node->cend, node->name, strlen(node->name), node->name);
	       for (unsigned int i = 0; i < node->args->size(); ++i) {
		    print_parse(node->args->at(i));
		    if (i < node->args->size() - 1) fprintf(stderr, ",");
	       }	       
	       fprintf(stderr, ")");
	       break;
	  case T_LINE:
	       fprintf(stderr, "LINE(l%d,b%d,e%d,", node->lbegin, node->cbegin, node->cend);
	       for (unsigned int i = 0; i < node->args->size(); ++i) {
		    print_parse(node->args->at(i));
		    fprintf(stderr, ";\n");
	       }
	       fprintf(stderr, ")\n");
	       break;
	  case T_LIST:
	       fprintf(stderr, "LIST(l%d,b%d,e%d,", node->lbegin, node->cbegin, node->cend);
	       for (unsigned int i = 0; i < node->args->size(); ++i) {
		    print_parse(node->args->at(i));
		    fprintf(stderr, ";\n");
	       }
	       fprintf(stderr, ")\n");
	       break;
	  default:
	       fprintf(stderr, "Error in tree: '%d'\n", node->type);
	       return;
	  }
     }
}

