/***************************************************************************
                                  process.cpp
                             -------------------
                               Sat Oct 19 2002
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "process.hpp"

#include <iostream>
#include <sstream>

#include "evexp.hpp"

void Varprocessor::addOper(NodeInfo::Node* varop, const unsigned int funtype) {
     static std::set<std::string> recursive_check;
     std::string inserted_item = "";

     // Resolve "outmost" templates
     if (isTmpl(varop)) {
	  inserted_item = varop->name;
	  if (recursive_check.find(inserted_item) != recursive_check.end()) {
	       std::ostringstream strm;
	       strm << "Recursive template found while resolving template"
		    << " function " << varop->name;
	       throw Parse::Error(strm.str(), varop);
	  }
	  recursive_check.insert(inserted_item);

	  NodeInfo::NodeVector funcs = reslvTmpl(varop);
	  for(NodeInfo::NodeVector::iterator iter = funcs.begin(); iter != funcs.end(); ++iter) {
	       addOper(*iter, funtype);
	  }
     }     
     else {
	  // Resolve non-outmost templates
	  varop = reslvInlineTmpl(varop, true);

	  switch(funtype) {
	  case 0:
	       preopers_.push_back(varop);
	       break;
	  case 1:
	       opers_.push_back(varop);
	       break;
	  case 2:
	       postopers_.push_back(varop);
	       break;
	  }
     }
     if (inserted_item != "") {
	  recursive_check.erase(inserted_item);
     }
}

void Varprocessor::addTmpl(std::string tmplName, NodeInfo::Node* node, const unsigned int args) {
     if (topers_.find(tmplName) != topers_.end()) {
	  std::cerr << "WARNING: Redefinition of template '" << tmplName << "'\n";
     }
     FunTmpl ftmpl;
     ftmpl.args = args;
     ftmpl.tmplOpers = node;
     topers_[tmplName] = ftmpl;
}

bool Varprocessor::doEntries(NodeInfo::NodeVector &opers) const {
     resolveAmbiguousOperators(opers);
     bool ret = true;
     for (NodeInfo::NodeVector::iterator funs = opers.begin(); funs != opers.end(); ++funs) {
	  ret &= compile(*funs, Type2type<bool>(), true).value();
     }
     return ret;
}

void Varprocessor::resolveAmbiguousOperators(NodeInfo::NodeVector &opers) const {
     for (NodeInfo::NodeVector::iterator iter = opers.begin(); iter != opers.end(); ++iter) {	  
	  NodeInfo::Node *oper = *iter;
	  if (oper->type == NodeInfo::T_CALL) {
	       std::string opname = oper->name;
	       if (opname == "_eq") {
		    if (oper->args->size() != 2) {
			 throw Parse::Error("Internal error: Comparison operator should take two arguments", oper);
		    }
		    if (argHasType<double>(oper->args->at(0)) &&
			argHasType<double>(oper->args->at(1))) {
			 getParseResult()->rename_function(oper, "eqdbl");
		    }
		    else if (argHasType<std::string>(oper->args->at(0)) &&
			     argHasType<std::string>(oper->args->at(1))) {
			 getParseResult()->rename_function(oper, "eqstr");
		    }
		    else {
			 throw Parse::Error("Cannot resolve operands of comparison operator to any appropriate types", oper);
		    }
	       }
	       else if (opname == "_set") {
		    if (oper->args->size() != 2) {
			 throw Parse::Error("Internal error: Assignment operator should take two arguments", oper);
		    }
		    if (argHasType<double&>(oper->args->at(0)) &&
			argHasType<double>(oper->args->at(1))) {
			 getParseResult()->rename_function(oper, "setdbl");
		    }
		    else if (argHasType<std::string&>(oper->args->at(0)) &&
			     argHasType<std::string>(oper->args->at(1))) {
			 getParseResult()->rename_function(oper, "setstr");
		    }
		    else {
			 throw Parse::Error("Cannot resolve operands of assignment operator to any appropriate types", oper);
		    }
	       }
	       if (oper->args) {
		    resolveAmbiguousOperators(*(oper->args));
	       }
	  }
     }
}


const Functor_base &Varprocessor::get(std::string funName, const int args) const {
     if (functors_.find(funName) == functors_.end()) {
	  EvExp e;
	  e.section = "Function execution";
	  std::ostringstream strm;
	  strm << "Function " << funName << " not found";
	  e.error = strm.str();
	  throw e;
     }
     
     funcmap::const_iterator enditer = functors_.upper_bound(funName);
     if (enditer != functors_.end())
	  ++enditer;
     for (funcmap::const_iterator iter = functors_.lower_bound(funName);
	 iter != enditer && iter != functors_.end();
	 ++iter) {
	  if(iter->second->args == args)
	       return *iter->second;
     }
     
     EvExp e;
     std::ostringstream sts;
     e.section = "Function execution";
     sts << "Function " << funName << " with " << args << " arguments not found";
     e.error = sts.str();
     throw e;
}

bool Varprocessor::isTmpl(NodeInfo::Node *node) const {
     ParseResult::check_type(node, NodeInfo::T_CALL);
     return topers_.find(node->name) != topers_.end();
}

void Varprocessor::processEntries() {
     doEntries(preopers_);    // compile
     ::execute();             // execute
     ::reset_execute();       // reset
     doEntries(opers_);       // compile
     while(::execute());      // execute
     ::reset_execute();       // reset
     doEntries(postopers_);   // compile
     ::execute();             // execute
}

NodeInfo::Node* Varprocessor::reslvTmpl_helper(NodeInfo::Node* node, NodeInfo::NodeVector *args) const {
     if (!node) {
	  throw Parse::Error("NULL found while resolving template");
     }
     switch (node->type) {
     case NodeInfo::T_STR: 
     case NodeInfo::T_INT:
     case NodeInfo::T_DBL:
     case NodeInfo::T_BOOL: break;
     case NodeInfo::T_CALL:
     case NodeInfo::T_LINE: 
	  // resolve recursively
	  for (NodeInfo::NodeVector::iterator iter = node->args->begin(); iter != node->args->end(); ++iter) {
	       *iter = reslvTmpl_helper(*iter, args);
	  }
	  break;
     case NodeInfo::T_TMPLREF: 
	  // resolve template
	  if (node->data.integer < 1 || node->data.integer > (int)args->size()) {
	       std::ostringstream strm;
	       strm << "Template has parameters named #1-#" << args->size() 
		    << ", but a reference to parameter #" << node->data.integer
		    << " was found";
	       throw Parse::Error(strm.str(), node);
	  }
	  return args->at(node->data.integer - 1);
	  break;
     default:
	  throw Parse::Error("Unknown node type found while resolving template");
	  break;
     }
     return node;
}

NodeInfo::NodeVector Varprocessor::reslvTmpl(NodeInfo::Node *oper) {
     ParseResult::check_type(oper, NodeInfo::T_CALL);
     unsigned int tmplargs = topers_.find(oper->name)->second.args;
     if (oper->args->size() != tmplargs) {
	  std::ostringstream strm;
	  strm << oper->args->size() << " argument(s) supplied, " << tmplargs
	       << " argument(s) expected in function " << oper->name;
	  throw Parse::Error(strm.str(), oper);
     }
     NodeInfo::NodeVector varops;
     NodeInfo::Node* tmplop = topers_.find(oper->name)->second.tmplOpers;
     tmplop = getParseResult()->clone(tmplop);
     ParseResult::check_type(tmplop, NodeInfo::T_LINE);
     tmplop = reslvTmpl_helper(tmplop, oper->args);
     for (NodeInfo::NodeVector::iterator iter = tmplop->args->begin(); iter != tmplop->args->end(); ++iter) {
	  varops.push_back(*iter);
     }
     return varops;
}

NodeInfo::Node* Varprocessor::reslvInlineTmpl(NodeInfo::Node* node, bool outmost) {
     static std::set<std::string> recursive_check;
     std::string inserted_item = "";
     if (node->type == NodeInfo::T_CALL) {
	  std::string opname = node->name;
	  if (!outmost && (opname == "if_" || opname == "ifelse")) {
	       std::ostringstream strm;
	       strm << "If clauses are not allowed in nested context";
	       throw Parse::Error(strm.str(), node);
	  }
	  if (isTmpl(node)) {
	       inserted_item = node->name;
	       if (recursive_check.find(inserted_item) != recursive_check.end()) {
		    std::ostringstream strm;
		    strm << "Recursive template found while resolving template"
			 << " function " << node->name;
		    throw Parse::Error(strm.str(), node);
	       }
	       recursive_check.insert(inserted_item);
	       NodeInfo::NodeVector opers = reslvTmpl(node);
	       if (opers.size() == 1) {
		    node = reslvInlineTmpl(opers.at(0), outmost);
	       }
	       else {
		    NodeInfo::Node* newnode = opers.at(opers.size() - 1);
		    for (int i = opers.size() - 2; i >= 0; --i) {
			 newnode = getParseResult()->andCall(opers.at(i), newnode);
		    }
		    node = newnode;
	       }
	  }
     }
     if (node->args) {
	  for (NodeInfo::NodeVector::iterator iter = node->args->begin(); iter != node->args->end(); ++iter) {
	       *iter = reslvInlineTmpl(*iter, false);
	  }
     }
     if (inserted_item != "") {
	  recursive_check.erase(inserted_item);
     }
     return node;
}

Varprocessor::~Varprocessor() {
}

Varprocessor* dumdidum::getVarprocessor() {
  static Varprocessor varprc;
  return &varprc;
}
