/***************************************************************************
                                  process.hpp
                             -------------------
                               Sat Oct 19 2002
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef PROCESS_H
#define PROCESS_H

#include <map>
#include <string>
#include <typeinfo>     // typeid

#include "execute.hpp"    // Execute<T>
#include "types.hpp"      // Type2type

#include "parser/parse.hpp"

class Functor_base;
typedef std::multimap<std::string, const Functor_base *> funcmap;
template<class R> class Functor_type;

/* Structure which contains a function template */
struct FunTmpl
{
     unsigned int args;
     NodeInfo::Node* tmplOpers;
};

typedef std::map<std::string,FunTmpl> TmplOpers;

class Varprocessor {
public:
    /* Associate a functor with a external function. 
       For external functions with 1-9 arguments */
     template<class R, class T1>
     bool addFunc(std::string funName, R (*fp)(T1));
     template<class R, class T1, class T2>
     bool addFunc(std::string funName, R (*fp)(T1, T2));
     template<class R, class T1, class T2, class T3>
     bool addFunc(std::string funName, R (*fp)(T1, T2, T3));
     template<class R, class T1, class T2, class T3, class T4>
     bool addFunc(std::string funName, R (*fp)(T1, T2, T3, T4));
     template<class R, class T1, class T2, class T3, class T4, 
	  class T5>
     bool addFunc(std::string funName, R (*fp)(T1, T2, T3, T4, T5));
     template<class R, class T1, class T2, class T3, class T4, 
	  class T5, class T6>
     bool addFunc(std::string funName, R (*fp)(T1, T2, T3, T4, T5, T6));
     template<class R, class T1, class T2, class T3, class T4, 
	  class T5, class T6, class T7>
     bool addFunc(std::string funName, R (*fp)(T1, T2, T3, T4, T5, T6, T7));
     template<class R, class T1, class T2, class T3, class T4, 
	  class T5, class T6, class T7, class T8>
     bool addFunc(std::string funName, R (*fp)(T1, T2, T3, T4, T5, T6, T7, T8));
     template<class R, class T1, class T2, class T3, class T4, 
	  class T5, class T6, class T7, class T8, class T9>
     bool addFunc(std::string funName, R (*fp)(T1, T2, T3, T4, T5, T6, T7, T8, T9));

     /* Adds an operation to the appropriate operator containter */
     void addOper(NodeInfo::Node *varop, const unsigned int funtype);

     /* Adds a template */
     void addTmpl(std::string tmplName, NodeInfo::Node* node, const unsigned int args);

     /* Compiles an operation to an Execute<R> object */
     template<class R>
     Execute<R> &compile(NodeInfo::Node *oper, Type2type<R>, bool outmost = false) const;

     /* Gets the function with the specified name and number of parameters */
     const Functor_base &get(std::string funName, const int args) const;

     /* Process all entries in the preopers, opers and postopers structures */
     void processEntries();

     ~Varprocessor();
private:
     /* Compiles the specified vector of operations */
     bool doEntries(NodeInfo::NodeVector &opers) const;

     /* Checks if a function call is a call to a template */
     bool isTmpl(NodeInfo::Node* node) const;

     /* Help function for resolving templates */
     NodeInfo::Node* reslvTmpl_helper(NodeInfo::Node* node, NodeInfo::NodeVector *args) const;

     /* Resolves a template */
     NodeInfo::NodeVector reslvTmpl(NodeInfo::Node* node);

     /* Resolve all templates recursively in a node tree. */
     NodeInfo::Node* reslvInlineTmpl(NodeInfo::Node* node, bool outmost);

     /* Operators that handle multiple types (=, ==) uses this function
	to resolve the types of the arguments and thus what kind of
	assignment or comparison that will be used. */
     void resolveAmbiguousOperators(NodeInfo::NodeVector &opers) const;

     /* Checks if a node can be converted to class R */
     template<class R>
     bool argHasType(NodeInfo::Node* arg) const;

     /* Converts lookup() calls to var() calls and lookupcnt() calls
	to cnt() calls if a reference to a string or counter is expected.
	This enables users to use the cont:var syntax even when the value
	of the variable is to be set and not only for lookup purposes */
     template<class R>
     NodeInfo::Node* referenceFix(Type2type<R>, NodeInfo::Node* node) const;

     funcmap functors_;
     NodeInfo::NodeVector postopers_;
     NodeInfo::NodeVector opers_;
     NodeInfo::NodeVector preopers_;
    TmplOpers topers_;   
};

/***************************************************************/
#include "functor.hpp"

template<class R>
bool Varprocessor::argHasType(NodeInfo::Node* arg) const {
     arg = referenceFix(Type2type<R>(), arg);
     if (arg->type == NodeInfo::T_CALL) {
	  try {
	       const Functor_base& fnc = get(arg->name, arg->args->size());
	       fnc.downcast(Type2type<R>());
	  }
	  catch (std::bad_cast &) {
	       return false;
	  }
	  return true;
     }
     else if (isTokenConst<R>(arg)) {
	  return true;
     }
     return false;
}

template<class R>
NodeInfo::Node* Varprocessor::referenceFix(Type2type<R>, NodeInfo::Node* node) const {
     std::string opname = "";
     if (node->name) {
	  opname = node->name;
     }
     if (opname == "lookup" && typeid(Type2type<R>) == typeid(Type2type<std::string&>)) {
	       getParseResult()->rename_function(node, "var");
     }
     else if (opname == "lookupcnt" && typeid(Type2type<R>) == typeid(Type2type<double&>)) {
	       getParseResult()->rename_function(node, "cnt");
     }
     return node;
}

namespace dumdidum {
     /*this silly namespace solves a bug in gcc 3.4.4 (it can't handle the
      ::getVarprocessor() call that should be used
      (we can't just call getVarprocessor() since the function is templated).*/
    Varprocessor* getVarprocessor();
}
/***************************************************************/

template<class R>
Execute<R> &Varprocessor::compile(NodeInfo::Node* oper, Type2type<R>, bool outmost) const {
     try {
	  oper = referenceFix(Type2type<R>(), oper);
	  const Functor_base &fnc = get(oper->name, oper->args->size());
	  const Functor_type<R> &f = fnc.downcast(Type2type<R>());   // throws if wrong type
	  return f.process(oper, outmost);
     }
     catch (std::bad_cast &) {
	  std::ostringstream strm;
	  strm << "Function " << oper->name << " does not return a " 
	       << type_name(Type2type<R>()) << " value";
	  throw Parse::Error(strm.str(), oper);
     }
     catch (const EvExp &e) {
	  throw Parse::Error(e.error, oper);
     }
}

template<class R, class T1>
bool Varprocessor::addFunc(std::string funName, R (*fp)(T1))
{
  Functor_base *f = new Functorone<R, T1>(fp);
  functors_.insert(funcmap::value_type(funName, f));
  return true;
}


template<class R, class T1, class T2>
bool Varprocessor::addFunc(std::string funName, R (*fp)(T1, T2))
{
  Functor_base *f = new Functortwo<R, T1, T2>(fp);
  functors_.insert(funcmap::value_type(funName, f));
  return true;
}


template<class R, class T1, class T2, class T3>
bool Varprocessor::addFunc(std::string funName, R (*fp)(T1, T2, T3))
{
  Functor_base *f = new Functorthree<R, T1, T2, T3>(fp);
  functors_.insert(funcmap::value_type(funName, f));
  return true;
}


template<class R, class T1, class T2, class T3, class T4>
bool Varprocessor::addFunc(std::string funName, R (*fp)(T1, T2, T3, T4))
{
  Functor_base *f = new Functorfour<R, T1, T2, T3, T4>(fp);
  functors_.insert(funcmap::value_type(funName, f));
  return true;
}

  
template<class R, class T1, class T2, class T3, class T4, class T5>
bool Varprocessor::addFunc(std::string funName, R (*fp)(T1, T2, T3, T4, T5))
{
  Functor_base *f = new Functorfive<R, T1, T2, T3, T4, T5>(fp);
  functors_.insert(funcmap::value_type(funName, f));
  return true;
}


template<class R, class T1, class T2, class T3, class T4, class T5, class T6>
bool Varprocessor::addFunc(std::string funName, R (*fp)(T1, T2, T3, T4, T5, T6))
{
  Functor_base *f = new Functorsix<R, T1, T2, T3, T4, T5, T6>(fp);
  functors_.insert(funcmap::value_type(funName, f));
  return true;
}


template<class R, class T1, class T2, class T3, class T4, class T5, class T6, class T7>
bool Varprocessor::addFunc(std::string funName, R (*fp)(T1, T2, T3, T4, T5, T6, T7))
{
  Functor_base *f = new Functorseven<R, T1, T2, T3, T4, T5, T6, T7>(fp);
  functors_.insert(funcmap::value_type(funName, f));
  return true;
}

  
template<class R, class T1, class T2, class T3, class T4, class T5, class T6, class T7, class T8>
bool Varprocessor::addFunc(std::string funName, R (*fp)(T1, T2, T3, T4, T5, T6, T7, T8))
{
  Functor_base *f = new Functoreight<R, T1, T2, T3, T4, T5, T6, T7, T8>(fp);
  functors_.insert(funcmap::value_type(funName, f));
  return true;
}

  
template<class R, class T1, class T2, class T3, class T4, class T5, class T6, class T7, class T8,class T9>
bool Varprocessor::addFunc(std::string funName, R (*fp)(T1, T2, T3, T4, T5, T6, T7, T8, T9))
{
  Functor_base *f = new Functornine<R, T1, T2, T3, T4, T5, T6, T7, T8, T9>(fp);
  functors_.insert(funcmap::value_type(funName, f));
  return true;
}

#endif

