/***************************************************************************
                                  xmlconf.cpp
                             -------------------
                               Wed Oct 2 2002
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "xmlconf.hpp"

#include <fstream>

#include "evutil.hpp"
#include "process.hpp"

DOMEvalCfgFilter::DOMEvalCfgFilter() {
}

short DOMEvalCfgFilter::acceptNode(const DOMNode* node) const {
     if(node->getNodeType() == DOMNode::ELEMENT_NODE)
	  return DOMNodeFilter::FILTER_ACCEPT;
     else
	  return DOMNodeFilter::FILTER_SKIP;
}

DOMEvalCfgFilter::~DOMEvalCfgFilter() {
}

EvalHandlerBase::EvalHandlerBase() {
}

void EvalHandlerBase::error(const SAXParseException &exception) {
     fatalError(exception);
}

void EvalHandlerBase::warning(const SAXParseException &exception) {
     fatalError(exception);
}

EvalHandlerBase::~EvalHandlerBase() {
}

Configurator::Configurator(Databank *databank, Varprocessor *varprc) 
     : databank_(databank)
     , varprc_(varprc) {
     // Setup internal 'if'-template
     std::string iftemplate = "exenext(#1); do(#2);";
     std::string filename = "internal";
     globalnode = getParseResult()->parse(iftemplate.c_str(), 0, 0, filename.c_str());
     varprc_->addTmpl("if_", globalnode, 2);

     // Setup internal 'ifelse'-template
     iftemplate = "exenextelse(#1); do(#2); do(#3);";
     filename = "internal";
     globalnode = getParseResult()->parse(iftemplate.c_str(), 0, 0, filename.c_str());
     varprc_->addTmpl("ifelse", globalnode, 3);
}

void Configurator::interpretEntry(const DataMap &tagMap, const DOMNode *domnode) {
     const std::string opName=tagMap.find("tagname")->second;
     if (opName=="libfile") {
	  processLibfileSection(tagMap);
     }
     else if (opName=="preprocess"  || opName=="process" || 
	      opName=="postprocess" || opName=="template") {
	  processFuncSection(tagMap, domnode);
     }
}

void Configurator::processFuncSection(const DataMap &tagMap, const DOMNode* domnode) {
     std::string fielddata = tagMap.find("#text")->second;
     ParseResult *parseResult = getParseResult();
     int line = 0;
     int column = 0;
     const char* filename = "Unknown file";
     NodePos::iterator iter = nodepos.find(domnode);
     if (iter != nodepos.end()) {
	  line = iter->second.line;
	  column = iter->second.column;
	  filename = iter->second.filename.c_str();
     }
     globalnode = parseResult->parse(fielddata.c_str(), line, column, filename);
     if (!globalnode) {
	  return;
     }
     const std::string tagname = tagMap.find("tagname")->second;
     if (tagname == "template") {
	  int args = str2int(tagMap.find("args")->second);
	  processTemplate(tagMap.find("name")->second, globalnode, args);
     }
     else if (tagname == "preprocess") {
	  processFunctions(globalnode, 0);
     }
     else if (tagname == "process") {
	  processFunctions(globalnode, 1);
     }
     else if (tagname == "postprocess") {
	  processFunctions(globalnode, 2);
     }
}

void Configurator::processFunctions(NodeInfo::Node* opers, const unsigned int funtype) const {
     ParseResult::check_type(opers, NodeInfo::T_LINE);
     for (NodeInfo::NodeVector::iterator iter = opers->args->begin(); iter != opers->args->end(); ++iter) {
	  varprc_->addOper(*iter, funtype);
     }
}

void Configurator::processLibfileSection(const DataMap &tagMap) {
     const std::string fileName = tagMap.find("#text")->second;
     ReadIncludedFile(fileName);
}

void Configurator::processTemplate(const std::string &tmplName, NodeInfo::Node* node, const unsigned int args) const {
     varprc_->addTmpl(tmplName, node, args);
}


void Configurator::ReadIncludedFile(const std::string &fileName) {
     CfgFileReader creader(fileName, this);
     creader.walk();
}

Configurator::~Configurator() {
}

CfgFileReader::CfgFileReader(const std::string &xmlFile, Configurator *prgConfig) : prgConfig_(prgConfig) {
     parser_ = new MyDOMParser(prgConfig->nodepos);
     parser_->setValidationScheme(XercesDOMParser::Val_Always);
     parser_->setDoNamespaces(true);    // optional
     parser_->setDoValidation(true);
     parser_->setDoSchema(true);
     parser_->setIncludeIgnorableWhitespace(true);
     parser_->setCreateCommentNodes(false);
     errHandler_ = (ErrorHandler*) new EvalHandlerBase();
     parser_->setErrorHandler(errHandler_);
     
     try {
	  std::ifstream test(xmlFile.c_str());
	  if (!test) {
	       EvExp e;
	       e.section = "XML Config file reader:";
	       std::stringstream strm;
	       strm << "Cannot open file '" << xmlFile << "'";
	       e.error = strm.str();
	       throw e;
	  }
	  test.close();
	  parser_->parse(xmlFile.c_str());
     }
     catch (const XMLException& toCatch) {
	  EvExp e;
	  e.section = "XML Config file reader";
	  e.error = "Xerces Error during parsing";
	  e.data = XML2Ch(toCatch.getMessage()).toStr();
	  throw e;
     }  
     catch (const DOMException& toCatch) {
	  EvExp e;
	  e.section = "XML Config file reader";
	  e.error = "Xerces Error during parsing";
	  e.data = XML2Ch(toCatch.msg).toStr();
	  throw e;
     } 
     catch (const SAXParseException& toCatch) {
	  char *filename = XMLString::transcode(toCatch.getSystemId());
	  int line = toCatch.getLineNumber();
	  int column = toCatch.getColumnNumber();
	  if (!filename) {
	       filename = "Configuration file";
	  }
	  std::ostringstream strm;
	  strm << "XML parser error:\n"
	       << XML2Ch(toCatch.getMessage()).toStr();
	  throw Parse::Error(strm.str(), line, column, filename);
     }
}

bool CfgFileReader::hasTextNode(const DOMNode *node) const {
     for (DOMNode *childnode = node->getFirstChild(); childnode != NULL; childnode=childnode->getNextSibling()) {
	  if (childnode->getNodeType() == DOMNode::TEXT_NODE)
	       return true;
     }
     return false;
}

void CfgFileReader::walk() const {
     DOMEvalCfgFilter stdfilt;
     DOMNode *node;
     DOMTreeWalker *walker = parser_->getDocument()->createTreeWalker(parser_->getDocument()->getDocumentElement(), DOMNodeFilter::SHOW_ALL, &stdfilt, true);
     while(true) {
	  bool nodeHasAttr, nodeHasText;
	  node = walker->nextNode();
	  if (node == NULL)
	       break;
	  nodeHasAttr = node->hasAttributes();
	  nodeHasText = hasTextNode(node);
	  if (nodeHasAttr || nodeHasText) {
	       DataMap tagMap;
	       
	       if(nodeHasAttr) {
		    DOMNamedNodeMap *attrmap = node->getAttributes();
		    for(XMLSize_t i = 0; i != attrmap->getLength(); ++i) {
			 DOMNode *attnode = attrmap->item(i);
			 tagMap[XML2Ch(attnode->getNodeName()).toStr()] = XML2Ch(attnode->getNodeValue()).toStr();
		    }
	       }
	       
	       if(nodeHasText) {
		    for (DOMNode *childnode = node->getFirstChild(); childnode != NULL; childnode = childnode->getNextSibling()) {
			 if (childnode->getNodeType() == DOMNode::TEXT_NODE) {
			      tagMap["#text"] = XML2Ch(childnode->getNodeValue()).toStr();
			 }
		    }
	       }
	       tagMap["tagname"] = XML2Ch(node->getNodeName()).toStr();
	       prgConfig_->interpretEntry(tagMap, node); 
	  } 
     }
}

CfgFileReader::~CfgFileReader() {
     delete parser_;
     delete errHandler_;
}
