//
// = FILENAME
//    MineField.java
//
// = FUNCTION
//    Defines the mine field panel which can load/save field and
//    generate random ones.
// 
// = AUTHOR(S)
//    Patric Jensfelt
//
// = COPYRIGHT
//    Copyright (c) 2006 Patric Jensfelt
//
/*----------------------------------------------------------------------*/

import java.util.Random;
import java.io.*;
import java.awt.*;
import javax.swing.*;
import java.awt.event.*;

class MineField extends JPanel implements MouseListener {
    public int        m_Xoffset = 15;
    public int        m_Yoffset = 30;
    
    public int        m_IWidth = 15;
    public int        m_IHeight = 15;
    
    private Image[]    m_Img;		
    private Square[]   m_Square;
    private boolean[]  m_Mine;

    private Minesweep  m_Parent;
    private boolean    m_AllowMouseOpen;
    private int        m_XSize, m_YSize;
    private int        m_NumMines;
    private int        m_NumOpened;

    /// The game is defined to be started when the first non-bomb
    /// square is opened
    private boolean    m_GameStarted;

    /// The number of mines that were opened before the game started
    private int        m_NumInitialMines;
    private boolean    m_Died;

    public MineField(Minesweep parent, int xsize, int ysize) {
        m_Parent = parent;
        m_AllowMouseOpen = true;
        m_XSize = xsize;
        m_YSize = ysize;
        
        // Read in images
        Toolkit toolkit = Toolkit.getDefaultToolkit();
        MediaTracker  trk = new MediaTracker(this) ;
        
        m_Img = new Image[14];
        for (int i = 0 ; i < 14 ; i++) {
            m_Img[i] = toolkit.getImage("ms_img/j" + (i) + ".gif");
            trk.addImage(m_Img[i], 0);
        }            
        
        try { trk.waitForAll() ; }
        catch (InterruptedException e) {	}

        calcAndSetSize();

        addMouseListener(this);
    }

    public void setAllowMouseOpen(boolean v) { m_AllowMouseOpen = v; }

    public int getNumOpened() { return m_NumOpened; }
    public int getNumInitialMines() { return m_NumInitialMines; }
    public boolean isDead() { return m_Died; }

    /// It is forbidden to call this function to calculate the number
    /// of mines in the field
    public int getMaxScore() { return (m_XSize*m_YSize - 
                                       m_NumMines - 
                                       m_NumInitialMines); }

    public int getFieldXSize() { return m_XSize; }
    public int getFieldYSize() { return m_YSize; }

    public void setFieldXSize(int s) {
        m_XSize = s;
        calcAndSetSize();
    }

    public void setFieldYSize(int s) {
        m_YSize = s;
        calcAndSetSize();
    }
    
    public void calcAndSetSize()
    {
        setPreferredSize(new Dimension(2 * m_Xoffset + m_IWidth * m_XSize,
                                       2 * m_Yoffset + m_IHeight * m_YSize));
        m_Parent.pack();
    }

    public void paint(Graphics g) {
        try {
            super.paint(g);
            
            for (int i = 0; i < m_XSize; i++) {
                for (int j = 0; j < m_YSize; j++) {
                    int index = i * m_YSize + j;
                    if (m_Square[index].isOpen()) {
                        if (m_Square[index].isMine()) {
                            g.drawImage(m_Img[9],
                                        m_Xoffset + i*m_IWidth, 
                                        m_Yoffset + j*m_IHeight, 
                                        m_IWidth,
                                        m_IHeight,
                                        this);
                        } else {
                            g.drawImage(m_Img[m_Square[index].getNumMineNeighbors()],
                                        m_Xoffset + i*m_IWidth, 
                                        m_Yoffset + j*m_IHeight, 
                                        m_IWidth,
                                        m_IHeight,
                                        this);
                        }                    
                    } else {
                        g.drawImage(m_Img[13],
                                    m_Xoffset + i*m_IWidth, 
                                    m_Yoffset + j*m_IHeight, 
                                    m_IWidth,
                                    m_IHeight,
                                    this);
                    }
                }
            }
        } catch (Exception exc) {
            //System.out.println("\n\n\nCaught exception in paint\n\n\n");
        }
    }

    public void mousePressed(MouseEvent e) { 
        if (!m_AllowMouseOpen) return;

        Point p = e.getPoint(); 

        if (p.x < m_Xoffset ||
            p.x > m_Xoffset + m_IWidth * m_XSize ||
            p.y < m_Yoffset ||
            p.y > m_Yoffset + m_IHeight * m_YSize) {
            return;
        }
            
        int i = (p.x - m_Xoffset) / m_IWidth;
        int j = (p.y - m_Yoffset) / m_IHeight;

        int ret = openSquare(i,j);

        if (ret == -1) {
            JOptionPane.showMessageDialog(m_Parent, "You are dead!!");
            m_Parent.gameOver(true);
        } else if (isAllClear()) {
            JOptionPane.showMessageDialog(m_Parent, "You won!!");
            m_Parent.gameOver(true);
        }
    }
    
    public void mouseReleased(MouseEvent e) {}
    public void mouseExited(MouseEvent e) {}
    public void mouseEntered(MouseEvent e) {}
    public void mouseClicked(MouseEvent e) {}


    public void openAll()
    {
        for (int i = 0; i < m_XSize; i++) {
            for (int j = 0; j < m_YSize; j++) {
                m_Square[i*m_YSize+j].open();
            }
        }
        m_Parent.repaint();
    }

    public boolean isAllClear() {
        return (m_NumOpened - m_NumInitialMines == 
                (m_XSize * m_YSize - m_NumMines));
    }

    public int peekSquare(int i, int j) {
        if (i < 0 || i >= m_XSize || j < 0 || j >= m_YSize) {
            System.out.println("Cannot peek square, i=" + i +
                               " j=" + j + ", it is outside field");
            return -10;
        }

        if (m_Square[i*m_YSize+j].isMine()) return -1;

        return m_Square[i*m_YSize+j].getNumMineNeighbors();       
    }
    
    public int openSquare(int i, int j) {
        if (i < 0 || i >= m_XSize || j < 0 || j >= m_YSize) {
            System.out.println("Cannot open square, i=" + i +
                               " j=" + j + ", it is outside field");
            return -10;
        }

        if (m_Square[i*m_YSize+j].isOpen()) {
            return -20;
        } else {
            m_NumOpened++;
        }
        m_Parent.updateNumOpened(m_NumOpened);

        int ret = m_Square[i*m_YSize+j].open();
        m_Parent.repaint();        

        if (ret == -1) {
            if (m_GameStarted) {
                System.out.println("You are dead!");
                m_Died = true;
                return -1;
            } else {
                System.out.println("You should really be dead");
                m_NumInitialMines++;
                return -2;
            }
        } 

        m_GameStarted = true;

        return ret;
    }

    public boolean readMinesFromFile(String filename) {
        
        int nMines = 0;

        try {
            FileReader fr = new FileReader(filename);
            BufferedReader br = new BufferedReader(fr);

            int rowLen = -1;
            int numRows = 0;
            String minelist = new String();

            // Continue to read lines while 
            // there are still some left to read
            String line = new String();
            while ((line = br.readLine()) != null) {

                if (line.charAt(0) == '#') {
                    // Skipping comment
                    continue;
                } 

                if (line.charAt(0) == 'S') {                
                    // This is the line with settings. We look for
                    // the end of the line and copy this substring
                    // out and parse it separately. The line is
                    // assumed to be
                    //
                    // S deadCost peekCost density
                    String args[] = line.split(" ");
                    System.out.println("#args=" + args.length);
                    int deadCost = -1, peekCost = -1; 
                    double density = -1;
                    try {
                        deadCost = Integer.parseInt(args[1]);
                        peekCost = Integer.parseInt(args[2]);
                        density = Double.parseDouble(args[3]);
                        m_Parent.setSettings(deadCost, peekCost, density);
                    } catch (Exception e) {
                        System.out.println("Could not parse all values for " +
                                           " deadCost peekCost density from " +
                                           "\"" + line + "\"");
                    }
                    continue;
                }


                numRows++;

                //System.out.println(line);

                int len = 0;
                for (int i = 0; i < line.length(); i++) {
                    // We break the line after the first char that is
                    // not 0 or 1
                    if (line.charAt(i) == '0' ||
                        line.charAt(i) == '1') {
                        len++;
                        minelist += line.substring(i,i+1);
                    }
                }
                                
                if (rowLen < 0) {
                    rowLen = len;
                } else if (rowLen != len) {
                    System.out.println("Length of row " + numRows +
                                       " does not match previous row");
                    return false;
                }
            }

            m_XSize = rowLen;
            m_YSize = numRows;

            m_Mine = new boolean[m_XSize*m_YSize];
            int index = 0;
            for (int j = 0; j < m_YSize; j++) {
                for (int i = 0; i < m_XSize; i++) {
                    if (minelist.charAt(index) == '1') {
                        m_Mine[i * m_YSize + j] = true;
                        nMines++;
                    } else {
                        m_Mine[i * m_YSize + j] = false;
                    }
                    index++;
                }
            }
            
        } catch(Exception e) {
            System.out.println("Got exception");
            return false;
        }

        m_NumMines = nMines;
        calcAndSetSize();
        return true;
    }

    public boolean saveMinesToFile(String filename) {
        
        int nMines = 0;

        try {
            FileWriter fw = new FileWriter(filename);
            BufferedWriter bw = new BufferedWriter(fw);

            for (int j = 0; j < m_YSize; j++) {
                for (int i = 0; i < m_XSize; i++) {
                    if (m_Mine[i * m_YSize + j]) {
                        bw.write('1');
                    } else {
                        bw.write('0');
                    }
                }
                bw.newLine();
            }
            bw.close();
            
        } catch(Exception e) {
            System.out.println("Got exception");
            return false;
        }

        return true;
    }

    public void createRandomMines(int xsize, int ysize, double density) {
        m_XSize = ysize;
        m_YSize = xsize;
        createRandomMines(density);
    }

    public void createRandomMines(double density) {
        Random randgen = new Random(System.currentTimeMillis());
        m_Mine = new boolean[m_XSize*m_YSize];
        m_NumMines = 0;
        for (int i = 0; i < m_XSize; i++) {
            for (int j = 0; j < m_YSize; j++) {
                if (randgen.nextDouble() <= density) {
                    m_Mine[i*m_YSize+j] = true;
                    m_NumMines++;
                } else {
                    m_Mine[i*m_YSize+j] = false;
                }
            }
        }
    }

    public void buildField() {
        m_GameStarted = false;
        m_NumInitialMines = 0;
        m_Died = false;
        m_NumOpened = 0;
        m_Square = new Square[m_XSize*m_YSize];        
        for (int i = 0; i < m_XSize; i++) {
            for (int j = 0; j < m_YSize; j++) {
                // Calculate number of mine around this square
                int nMines = 0;
                for (int di = -1; di <= 1; di++) {
                    for (int dj = -1; dj <= 1; dj++) {
                        if (i+di >= 0 && i+di < m_XSize &&
                            j+dj >= 0 && j+dj < m_YSize) {
                            if (m_Mine[(i+di)*m_YSize+j+dj]) nMines++;
                        }                                
                    }
                }
                m_Square[i*m_YSize+j] = new Square(m_Mine[i*m_YSize+j],nMines);
            }
        }

        m_Parent.updateNumOpened(m_NumOpened);
        m_Parent.repaint();
    }    
};

class Square {
    private boolean m_Mine;
    private boolean m_Opened;
    private int     m_NumMineNeighbors;

    public Square(boolean mine, int nMineNeighbors) {
        m_NumMineNeighbors = nMineNeighbors;
        m_Mine = mine;
        m_Opened = false;
    }


    /**
     * @return number of neighboring squares with mines, -1 if this is
     * a mine in which case you are history!!
     */
    public int open() { 
        m_Opened = true; 
        if (isMine()) return -1;
        else return m_NumMineNeighbors;
    }

    public int getNumMineNeighbors() { return m_NumMineNeighbors; }
    public boolean isOpen() { return m_Opened; }
    public boolean isMine() { return m_Mine; }
};

