//
// = FILENAME
//    MineField.java
//
// = FUNCTION
//    Defines the mine field which can load/save field and
//    generate random ones.
// 
// = AUTHOR(S)
//    Patric Jensfelt
//    Modifications by Oscar Sundbom
//
// = COPYRIGHT
//    Copyright (c) 2006 Patric Jensfelt
//
/*----------------------------------------------------------------------*/

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileReader;
import java.io.FileWriter;
import java.util.Observable;
import java.util.Random;

class MineField extends Observable {

    static class ChangeMask {
    	
    	public static final int PARAMS_CHANGED = 1;
    	public static final int FIELD_CHANGED = 2;
    	
    	private int m_mask;

    	public ChangeMask(int mask) {
    		m_mask = mask;
    	}
    	
    	public boolean getFieldParametersChanged() {
			return (m_mask & PARAMS_CHANGED) != 0;
		}
		public boolean getFieldStateChanged() {
			return (m_mask & FIELD_CHANGED) != 0;
		}
    }
	
	private Square[]   m_Square;
    private boolean[]  m_Mine;

    private int        m_XSize, m_YSize;
    private int        m_NumMines;
    private int        m_NumOpened;

    /// The game is defined to be started when the first non-bomb
    /// square is opened
    private boolean    m_GameStarted;

    /// The number of mines that were opened before the game started
    private int        m_NumInitialMines;
    private boolean    m_Died;

    private double     m_Density;
    private int        m_DeadCost = 10;
    private int        m_PeekCost = 15;
    
    public MineField(int xsize, int ysize) {
        m_XSize = xsize;
        m_YSize = ysize;
    }

    public int getNumOpened() { return m_NumOpened; }
    public int getNumInitialMines() { return m_NumInitialMines; }
    public boolean isDead() { return m_Died; }

    /// It is forbidden to call this function to calculate the number
    /// of mines in the field
    public int getMaxScore() { return (m_XSize*m_YSize - 
                                       m_NumMines - 
                                       m_NumInitialMines); }

    public void setSettings(int deadCost, int peekCost, double density) {
        m_DeadCost = deadCost;
        m_PeekCost = peekCost;
        m_Density = density;

        setChanged();
        notifyObservers(new ChangeMask(ChangeMask.PARAMS_CHANGED));
    }
    
    public int getDeadCost() { return m_DeadCost; }
    public int getPeekCost() { return m_PeekCost; }
    public double getDensity() { return m_Density; }
    
    public void setDeadCost(int deadCost) { m_DeadCost = deadCost; setChanged(); notifyObservers(new ChangeMask(ChangeMask.PARAMS_CHANGED)); }
    public void setPeekCost(int peekCost) { m_PeekCost = peekCost; setChanged(); notifyObservers(new ChangeMask(ChangeMask.PARAMS_CHANGED)); }
    public void setDensity(double density) { m_Density = density; setChanged(); notifyObservers(new ChangeMask(ChangeMask.PARAMS_CHANGED)); }

    public int getFieldXSize() { return m_XSize; }
    public int getFieldYSize() { return m_YSize; }

    public void setFieldSize(int x, int y) {
        m_XSize = x;
        m_YSize = y;
        setChanged();
        notifyObservers(new ChangeMask(ChangeMask.PARAMS_CHANGED));
    }

    public void setFieldXSize(int x) {
    	setFieldSize(x, m_YSize);
    }
    
    public void setFieldYSize(int y) {
    	setFieldSize(m_XSize,y );
    }

    public void openAll()
    {
        for (int i = 0; i < m_XSize; i++) {
            for (int j = 0; j < m_YSize; j++) {
                m_Square[i*m_YSize+j].open();
            }
        }
        setChanged();
        notifyObservers(new ChangeMask(ChangeMask.FIELD_CHANGED));
    }

    public boolean isAllClear() {
        return (m_NumOpened - m_NumInitialMines == 
                (m_XSize * m_YSize - m_NumMines));
    }
    
    public Square getSquare(int i, int j)
    {
    	return m_Square[i*m_YSize+j];
    }

    public int peekSquare(int i, int j) {
        if (i < 0 || i >= m_XSize || j < 0 || j >= m_YSize) {
            System.out.println("Cannot peek square, i=" + i +
                               " j=" + j + ", it is outside field");
            return -10;
        }

        if (m_Square[i*m_YSize+j].isMine()) return -1;

        return m_Square[i*m_YSize+j].getNumMineNeighbors();       
    }
    
    public int openSquare(int i, int j) {
        if (i < 0 || i >= m_XSize || j < 0 || j >= m_YSize) {
            System.out.println("Cannot open square, i=" + i +
                               " j=" + j + ", it is outside field");
            return -10;
        }

        if (m_Square[i*m_YSize+j].isOpen()) {
            return -20;
        } else {
            m_NumOpened++;
        }

        int ret = m_Square[i*m_YSize+j].open();

        if (ret == -1) {
            if (m_GameStarted) {
                System.out.println("You are dead!");
                m_Died = true;
                return -1;
            } else {
                System.out.println("You should really be dead");
                m_NumInitialMines++;
                return -2;
            }
        } 

        m_GameStarted = true;

        setChanged();
        notifyObservers(new ChangeMask(ChangeMask.FIELD_CHANGED));
        
        return ret;
    }

    public boolean readMinesFromFile(String filename) {
        int nMines = 0;

        try {
            FileReader fr = new FileReader(filename);
            BufferedReader br = new BufferedReader(fr);

            int rowLen = -1;
            int numRows = 0;
            String minelist = new String();

            // Continue to read lines while 
            // there are still some left to read
            String line = new String();
            while ((line = br.readLine()) != null) {

                if (line.charAt(0) == '#') {
                    // Skipping comment
                    continue;
                } 

                if (line.charAt(0) == 'S') {                
                    // This is the line with settings. We look for
                    // the end of the line and copy this substring
                    // out and parse it separately. The line is
                    // assumed to be
                    //
                    // S deadCost peekCost density
                    String args[] = line.split(" ");
//                    System.out.println("#args=" + args.length);
                    int deadCost = -1, peekCost = -1; 
                    double density = -1;
                    try {
                        deadCost = Integer.parseInt(args[1]);
                        peekCost = Integer.parseInt(args[2]);
                        density = Double.parseDouble(args[3]);
                        setSettings(deadCost, peekCost, density);
                    } catch (Exception e) {
                        System.out.println("Could not parse all values for " +
                                           " deadCost peekCost density from " +
                                           "\"" + line + "\"");
                    }
                    continue;
                }


                numRows++;

                int len = 0;
                for (int i = 0; i < line.length(); i++) {
                    // We break the line after the first char that is
                    // not 0 or 1
                    if (line.charAt(i) == '0' ||
                        line.charAt(i) == '1') {
                        len++;
                        minelist += line.substring(i,i+1);
                    }
                }
                                
                if (rowLen < 0) {
                    rowLen = len;
                } else if (rowLen != len) {
                    System.out.println("Length of row " + numRows +
                                       " does not match previous row");
                    return false;
                }
            }

            m_XSize = rowLen;
            m_YSize = numRows;

            m_Mine = new boolean[m_XSize*m_YSize];
            int index = 0;
            for (int j = 0; j < m_YSize; j++) {
                for (int i = 0; i < m_XSize; i++) {
                    if (minelist.charAt(index) == '1') {
                        m_Mine[i * m_YSize + j] = true;
                        nMines++;
                    } else {
                        m_Mine[i * m_YSize + j] = false;
                    }
                    index++;
                }
            }
            
        } catch(Exception e) {
            System.out.println("Got exception");
            e.printStackTrace();
            return false;
        }

        m_NumMines = nMines;
        m_Density = -1;
        setChanged();
        notifyObservers(new ChangeMask(ChangeMask.PARAMS_CHANGED));
        return true;
    }

    public boolean saveMinesToFile(String filename) {

        try {
            FileWriter fw = new FileWriter(filename);
            BufferedWriter bw = new BufferedWriter(fw);

            for (int j = 0; j < m_YSize; j++) {
                for (int i = 0; i < m_XSize; i++) {
                    if (m_Mine[i * m_YSize + j]) {
                        bw.write('1');
                    } else {
                        bw.write('0');
                    }
                }
                bw.newLine();
            }
            bw.close();
            
        } catch(Exception e) {
            System.out.println("Got exception");
            return false;
        }

        return true;
    }

    public void createRandomMines(int xsize, int ysize, double density) {
    	setFieldSize(xsize, ysize);
        createRandomMines(density);
    }

    public void createRandomMines(double density) {
        Random randgen = new Random(System.currentTimeMillis());
        m_Mine = new boolean[m_XSize*m_YSize];
        m_NumMines = 0;
        for (int i = 0; i < m_XSize; i++) {
            for (int j = 0; j < m_YSize; j++) {
                if (randgen.nextDouble() <= density) {
                    m_Mine[i*m_YSize+j] = true;
                    m_NumMines++;
                } else {
                    m_Mine[i*m_YSize+j] = false;
                }
            }
        }
        
        setChanged();
        notifyObservers(new ChangeMask(ChangeMask.FIELD_CHANGED));
    }

    public void createRandomMines() {
    	createRandomMines(m_Density);
    }

    public void buildField() {
        m_GameStarted = false;
        m_NumInitialMines = 0;
        m_Died = false;
        m_NumOpened = 0;
        m_Square = new Square[m_XSize*m_YSize];        
        for (int i = 0; i < m_XSize; i++) {
            for (int j = 0; j < m_YSize; j++) {
                // Calculate number of mine around this square
                int nMines = 0;
                for (int di = -1; di <= 1; di++) {
                    for (int dj = -1; dj <= 1; dj++) {
                        if (i+di >= 0 && i+di < m_XSize &&
                            j+dj >= 0 && j+dj < m_YSize) {
                            if (m_Mine[(i+di)*m_YSize+j+dj]) nMines++;
                        }                                
                    }
                }
                m_Square[i*m_YSize+j] = new Square(m_Mine[i*m_YSize+j],nMines);
            }
        }

        setChanged();
        notifyObservers(new ChangeMask(ChangeMask.FIELD_CHANGED));
    }    
};

class Square {
    private boolean m_Mine;
    private boolean m_Opened;
    private int     m_NumMineNeighbors;

    public Square(boolean mine, int nMineNeighbors) {
        m_NumMineNeighbors = nMineNeighbors;
        m_Mine = mine;
        m_Opened = false;
    }


    /**
     * @return number of neighboring squares with mines, -1 if this is
     * a mine in which case you are history!!
     */
    public int open() {
        m_Opened = true; 
        if (isMine()) return -1;
        else return m_NumMineNeighbors;
    }

    public int getNumMineNeighbors() { return m_NumMineNeighbors; }
    public boolean isOpen() { return m_Opened; }
    public boolean isMine() { return m_Mine; }
};

