//
// = FILENAME
//    MineFieldPanel.java
//
// = FUNCTION
//    Defines a mine field view as a JPanel.
// 
// = AUTHOR(S)
//    Patric Jensfelt
//    Modifications by Oscar Sundbom
//
// = COPYRIGHT
//    Copyright (c) 2006 Patric Jensfelt
//
/*----------------------------------------------------------------------*/

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.Observable;
import java.util.Observer;
import java.util.concurrent.Semaphore;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

public class MineFieldPanel extends JPanel implements MouseListener, Observer {

	private class UpdateThread extends Thread {
		
		private Semaphore m_update = new Semaphore(1);
		
		UpdateThread() {
			// Make sure we don't fire an update as soon as we run
			try {
				m_update.acquire();
			} catch(InterruptedException ex) {
				// Really cannot happen
			}
		}
		
		public void signalUpdate() {
			m_update.release();
		}
		
		@Override
		public void run() {
			System.out.println("Spawning update timer");
			try {
				while(true) {
					m_update.acquire();
					repaint();
					Thread.sleep(50);
				}
			} catch(InterruptedException ex) {
				/* Just ignore it and die */
			}
		}
	}
	
	public int m_IWidth = 15;
	public int m_IHeight = 15;
	
	public int m_Xoffset = 15;
	public int m_Yoffset = 30;
	
	private int m_LastXSize = -1;
	private int m_LastYSize = -1;

	private boolean m_LimitFramerate = false;
	private UpdateThread m_UpdateThread = null;

	private Image[] m_Img;
	private boolean m_AllowMouseOpen;

	private MineField m_MineField;					
	private MinesweepEngine m_Engine;

	MineFieldPanel(MinesweepEngine engine) {
		m_AllowMouseOpen = true;
		m_MineField = engine.getField();
		m_Engine = engine;

		// Read in images
		Toolkit toolkit = Toolkit.getDefaultToolkit();
		MediaTracker trk = new MediaTracker(this);

		m_Img = new Image[14];
		for (int i = 0; i < 14; i++) {
			m_Img[i] = toolkit.getImage("ms_img/j" + (i) + ".gif");
			trk.addImage(m_Img[i], 0);
		}

		try {
			trk.waitForAll();
		} catch (InterruptedException e) {
		}

		calcAndSetSize();

		addMouseListener(this);
		m_MineField.addObserver(this);
		m_UpdateThread = new UpdateThread();
		m_UpdateThread.start();
	}

	public void setAllowMouseOpen(boolean v) {
		m_AllowMouseOpen = v;
	}
	
	public void setLimitFramerate(boolean enableLimiting) {
		m_LimitFramerate = enableLimiting;
	}
	
	public boolean getLimitFramerate() { return m_LimitFramerate; };

	public void calcAndSetSize() {
		Dimension dim = new Dimension(2 * m_Xoffset + m_IWidth
				* m_MineField.getFieldXSize(), 2 * m_Yoffset + m_IHeight
				* m_MineField.getFieldYSize());
		
		setPreferredSize(dim);
		setSize(dim);
		
		m_LastXSize = m_MineField.getFieldXSize();
		m_LastYSize = m_MineField.getFieldYSize();

		// FIXME: HACK HACK HACK!! :((
		if(getTopLevelAncestor()!= null && getTopLevelAncestor() instanceof java.awt.Window) {
			((JFrame) getTopLevelAncestor()).pack();
		}
	}

	public void paint(Graphics g) {
		try {
			super.paint(g);

			for (int i = 0; i < m_MineField.getFieldXSize(); i++) {
				for (int j = 0; j < m_MineField.getFieldYSize(); j++) {
					// int index = i * m_YSize + j;
					if (m_MineField.getSquare(i, j).isOpen()) {
						if (m_MineField.getSquare(i, j).isMine()) {
							g.drawImage(m_Img[9], m_Xoffset + i * m_IWidth,
									m_Yoffset + j * m_IHeight, m_IWidth,
									m_IHeight, this);
						} else {
							g.drawImage(m_Img[m_MineField.getSquare(i, j)
									.getNumMineNeighbors()], m_Xoffset + i
									* m_IWidth, m_Yoffset + j * m_IHeight,
									m_IWidth, m_IHeight, this);
						}
					} else {
						g.drawImage(m_Img[13], m_Xoffset + i * m_IWidth,
								m_Yoffset + j * m_IHeight, m_IWidth, m_IHeight,
								this);
					}
				}
			}
		} catch (Exception exc) {
			// System.out.println("\n\n\nCaught exception in paint\n\n\n");
		}
	}

	public void mousePressed(MouseEvent e) {
		if (!m_AllowMouseOpen)
			return;

		Point p = e.getPoint();

		if (p.x < m_Xoffset
				|| p.x > m_Xoffset + m_IWidth * m_MineField.getFieldXSize()
				|| p.y < m_Yoffset
				|| p.y > m_Yoffset + m_IHeight * m_MineField.getFieldYSize()) {
			return;
		}

		int i = (p.x - m_Xoffset) / m_IWidth;
		int j = (p.y - m_Yoffset) / m_IHeight;

		int ret = m_MineField.openSquare(i, j);

		if (ret == -1) {
			JOptionPane.showMessageDialog(getParent(), "You are dead!!");
			m_Engine.gameOver(true);
		} else if (m_MineField.isAllClear()) {
			JOptionPane.showMessageDialog(getParent(), "You won!!");
			m_Engine.gameOver(true);
		}
	}

	public void mouseReleased(MouseEvent e) {
	}

	public void mouseExited(MouseEvent e) {
	}

	public void mouseEntered(MouseEvent e) {
	}

	public void mouseClicked(MouseEvent e) {
	}

	public void update(Observable obs, Object obj) {
		if (obs == m_MineField) {
			MineField.ChangeMask mask = (MineField.ChangeMask) obj;
			
			if(mask.getFieldParametersChanged()) {
				if (m_LastXSize != m_MineField.getFieldXSize()
						|| m_LastYSize != m_MineField.getFieldYSize()) {
					calcAndSetSize();
				}
			} 
		
			if(mask.getFieldStateChanged()) {
				if (m_MineField.isDead() || !m_LimitFramerate) {
					getParent().repaint();
					//paintImmediately(0, 0, getWidth(), getHeight());
				} else {
					m_UpdateThread.signalUpdate();
				}
			}
		}
	}

}
