//
// = FILENAME
//    Minesweep.java
//
// = FUNCTION
//
//    This is the main control for the mine sweep process. You can run it in 
//    three different modes:
//    1. Manual mode: user clicks on the mines
//    2. Auto mode  : agent plays
//    3. Server mode: agent can connect via sockets and play
// 
// = AUTHOR(S)
//    Patric Jensfelt
//    Modifications by Oscar Sundbom
//
// = COPYRIGHT
//    Copyright (c) 2005 Patric Jensfelt
//
/*----------------------------------------------------------------------*/

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.ServerSocket;
import java.net.Socket;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Observable;
import java.util.Random;

public class MinesweepEngine extends Observable {

	// ///////////////////////////////////////////////////////
	// ///////////////////////////////////////////////////////

	private class ConnectionListener extends Thread {
		int m_Port;

		public ConnectionListener(int port) {
			super("Connection listener");
			m_Port = port;
		}

		public void run() {
			try {
				ServerSocket ss = new ServerSocket(m_Port);
				System.out.println("Minesweep Server listening on port "
						+ m_Port);
				while (true) {
					Socket s = ss.accept();
					System.out.println("ConnectionListener::Got client");
					addClient(s);
				}
			} catch (Exception exc) {
				if(m_DebugLevel >= 2) {
					exc.printStackTrace();
				}
				System.out.println("Coud not open ServerSocket on port "
						+ m_Port);
			}
		}
	};

	protected class TimerThread extends Thread {

		public TimerThread() {
			super("Timer thread");
		}

		public void run() {
		
			  while (true) {
				  if (m_PlayMode == SERVER_MODE) {
					  if (m_StartTime > 0) {
						  m_GameTime = System.nanoTime() - m_StartTime;
						  long dt = m_GameTime / 1000000000L;
						  
						  if (dt > m_MaxGameTimeSec) {
							  if (!m_StoppedGame) {
								  m_StoppedGame = true; // Give clients 10s to close
								  m_StartTime += 10000000000L;
								  showMessage("Timeout", MinesweepUI.LEVEL_NOTIFY);
								  clientGameOver(true);
							  } else {
								  System.out.println("No response from clients, " + "closing them");
								  showMessage("Timeout: closing all", MinesweepUI.LEVEL_NOTIFY);
								  closeAllClients();
							  }
						  }
					  } else {
						  m_GameTime = 0;
					  }
				  } else {
					  if(m_StartTime > 0) {
						  m_GameTime = System.nanoTime() - m_StartTime;
					  } else {
						  m_GameTime = 0;
					  }
				  }
				  
				  setChanged();
				  notifyObservers(new ChangeMask(ChangeMask.TIMER_CHANGED));
			  
				  try {
					  Thread.sleep(1000);
				  } catch (Exception exc) {}
			  }
		}
	}

	protected class ControlThread extends Thread {

		private boolean m_shutdown = false;

		public ControlThread() {
			super("Control thread");
		}

		public void shutdown() {
			m_shutdown = true;
		}

		public void run() {

			while (!m_shutdown) {

				if (m_PlayMode == AUTO_MODE && m_StartPlay) {
					m_StartPlay = false;
					m_Playing = true;

					autoPlay();
				}

				if (m_PlayMode == SERVER_MODE) {

					if (m_StartPlay) {
						sendGameStart();
						m_StartPlay = false;
						m_StoppedGame = false;
						m_Playing = true;

						System.out.println("Starting new game");
						clientPlay();
					}

				}

				try {
					Thread.sleep(100);
				} catch (InterruptedException exc) {
				}
			}
		}
	}

	public class AgentClient {
		private Socket m_Socket;
		private InputStream m_InRaw;
		private BufferedReader m_In;
		private PrintWriter m_Out;
		private int m_Id;
		private String m_Name;
		private boolean m_Peeked;
		private boolean m_Active;
		private int m_Score;
		private long m_UsedTime;
		private double m_TimeFraction;

		// Stats stuff.
		private int m_NumWins;
		private double m_AvgScore;
		private int m_NumGames;
		private long m_TotalUsedTime;
		
		public int getScore() {
			return m_Score;
		}
		
		public String getName() {
			return m_Name;
		}
		
		public long getUsedTime() {
			return m_UsedTime;
		}
		
		public long getTotalUsedTime() {
			return m_TotalUsedTime;
		}

		public long getUsedTimeSecs() {
			return m_UsedTime / 1000000000L;
		}
		
		public long getTotalUsedTimeSecs() {
			return m_TotalUsedTime / 1000000000L;
		}

		public int getNumWins() {
			return m_NumWins;
		}
		
		public double getAvgScore() {
			return m_AvgScore;
		}
		
		public int getNumGames() {
			return m_NumGames;
		}
		
		public void clearStats() {
			m_NumWins = 0;
			m_AvgScore = 0;
			m_NumGames = 0;
			m_TotalUsedTime = 0;
		}
	};
	
	public static class ChangeMask {
		public static final int TIMER_CHANGED = 1;
		public static final int CLIENTS_CHANGED = 2;
		
		private int m_mask;
		
		ChangeMask(int mask) {
			m_mask = mask;
		}
		
		boolean getTimerChanged() {
			return (m_mask & TIMER_CHANGED) > 0;
		}
		
		boolean getClientsChanged() {
			return (m_mask & CLIENTS_CHANGED) > 0;
		}
	}
	
	// ///////////////////////////////////////////////////////
	// ///////////////////////////////////////////////////////

	public static final int MANUAL_MODE = 0;
	public static final int AUTO_MODE = 1;
	public static final int SERVER_MODE = 2;

	private MineField m_Field;
	private String m_MineFile;
	private MinesweepUI m_UI;
	
	private FileReader m_BatchFile;
	private BufferedReader m_BatchReader;
	private FileWriter m_BatchResFile;
	private BufferedWriter m_BatchResWriter;

	private int m_PlayMode;
	private boolean m_StartPlay;
	private boolean m_Playing;
	private int m_GameNumber;
	private int m_GameId = 0;
	private int m_AutoScore;
	private boolean m_TwoPlayerGame;
	private boolean m_SaveGameBoards;
	private boolean m_OnlySaveBoards;
	private int m_DebugLevel = 2;
	
	private boolean m_AutoRestart = true;
	private boolean m_RandomEnabled = true;
	private boolean m_RandomDensity = true;
	private boolean m_RandomBoardSize = true;
	private boolean m_RandomPeekCost = true;
	private boolean m_RandomDeadCost = true;

	private MineAgent m_Agent;
	
	private boolean m_StoppedGame;
	private boolean m_GameStopPressed;
	private long m_StartTime = -1;
	private long m_GameTime = 0;
	private int m_GameStepDelay = 0;

	private Thread m_TimerThread;
	private Thread m_ControlThread;
	private ConnectionListener m_ConnListener;
	private AgentClient m_Clients[];
	private int m_NumClients = 0;
	

	// Parameters
	private long m_MaxGameTimeSec = 300;
	private int m_ServerPort;

	public MinesweepEngine(int port, int xsize, int ysize, double density,
			int deadCost, int peekCost) {

		m_ServerPort = port;
		m_Field = new MineField(xsize, ysize);
		m_Field.setSettings(deadCost, peekCost, density);
		m_Field.createRandomMines();

		m_StartPlay = false;
		m_Playing = false;
		m_PlayMode = SERVER_MODE;

		m_ControlThread = new ControlThread();
		m_ControlThread.start();

		m_TimerThread = new TimerThread();
		m_TimerThread.start();
		 
		m_ConnListener = new ConnectionListener(m_ServerPort);
		m_ConnListener.start();
		m_NumClients = 0;
	}

	public MineField getField() {
		return m_Field;
	}

	public boolean inBatchMode() {
		return m_BatchReader != null;
	}
	
	public void disableBatchMode() {
		m_BatchReader = null;
	}
	
	public int getPlayMode() {
		return m_PlayMode;
	}

	public void setPlayMode(int mode) {
		m_PlayMode = mode;

		switch (mode) {
		case MANUAL_MODE:
			if (m_BatchReader != null) {
				System.out.println("Stopping batch run");
				m_BatchReader = null;
			}
			break;
		}
	}

	public MineAgent getAgent() {
		return m_Agent;
	}

	public void setAgent(MineAgent agent) {
		m_Agent = agent;
	}
	
	public MinesweepUI getUI() {
		return m_UI;
	}
	
	public void setUI(MinesweepUI ui) {
		m_UI = ui;
	}

	public void startGame() {
        if (m_PlayMode == MinesweepEngine.AUTO_MODE || m_PlayMode == MinesweepEngine.SERVER_MODE) {
            m_StartPlay = true;
            m_Playing = false;
            m_StartTime = System.nanoTime();
        } else if (m_PlayMode == MANUAL_MODE) {
            m_StartTime = System.nanoTime();
        }
	}
	
	public void stopGame() {
		m_GameStopPressed = true;
		if (m_PlayMode == SERVER_MODE) {
			System.out.println("Stopped the game");
			m_StoppedGame = true;
			clientGameOver(false);
		} else if (m_PlayMode == MANUAL_MODE) {
			gameOver(true);
		} else {
			gameOver(false);
		}

		if (m_BatchReader != null) {
			m_BatchReader = null;
		}
	}

	public void setTwoPlayerGame(boolean v) {
		m_TwoPlayerGame = v;
	}

	public void setGameStepDelay(int msec) {
		m_GameStepDelay = msec;
	}

	public int getGameStepDelay() {
		return m_GameStepDelay;
	}
	
	public int getGameTimeSec() {
		if(m_StartTime < 0)
			return 0;
		else {
			int tyme = (int) (m_GameTime / 1000000000L);
			return tyme;
		}
	}

	public int getMaxGameTimeSec() {
		return (int) m_MaxGameTimeSec;
	}
	
	public void rejectClient(Socket s, String msg) {
		if(m_DebugLevel >= 2) {
			System.out.println("Rejecting client: " + msg);
		}
		
		try {
			PrintWriter pw = new PrintWriter(s.getOutputStream(), true);
			pw.println(msg);
			pw.close();
			s.close();
		} catch (Exception exc) {
		}
	}

	public void sendToClient(PrintWriter pw, String msg) {
		try {
			pw.println(msg);
		} catch (Exception exc) {
		}
	}

	public void sendToAllClients(String msg) {
		for (int i = 0; i < m_NumClients; i++) {
			try {
				m_Clients[i].m_Out.println(msg);
			} catch (Exception exc) {
			}
		}
	}

	public void sendGameStart() {
		for (int i = 0; i < m_NumClients; i++) {
			m_Clients[i].m_Active = true;
			String msg = "START " + m_NumClients + " " + m_Clients[i].m_Id
					+ " " + m_Field.getFieldXSize() + " "
					+ m_Field.getFieldYSize() + " " + m_Field.getDeadCost()
					+ " " + m_Field.getPeekCost();

			sendToClient(m_Clients[i].m_Out, msg);
		}
	}
	
	public void clearStats() {
		for(int i=0;i!=m_NumClients;++i) {
			m_Clients[i].clearStats();
		}
		
		setChanged();
		notifyObservers(new ChangeMask(ChangeMask.CLIENTS_CHANGED));
	}

	public int getNumClients() {
		return m_NumClients;
	}

	public int getNumActiveClients() {
		int n = 0;
		for (int i = 0; i < m_NumClients; i++) {
			if (m_Clients[i].m_Active)
				n++;
		}

		return n;
	}
	
	public AgentClient getClient(int index) {
		if(index > m_Clients.length) {
			return null;
		} else {
			return m_Clients[index];
		}
	}

	// / This function should be called when there is a new client
	public boolean addClient(Socket s) {
		if (m_PlayMode != SERVER_MODE) {
			rejectClient(s, "ERROR Not in server mode");
			return false;
		}

		if ((!m_TwoPlayerGame && m_NumClients == 1)
				|| (m_TwoPlayerGame && m_NumClients > 1)) {
			rejectClient(s, "ERROR No more slots to connect to");
			return false;
		}

		AgentClient newClient = new AgentClient();
		newClient.m_Socket = s;
		newClient.m_Id = 0;
		newClient.m_Peeked = false;
		newClient.m_Score = 0;
		newClient.m_UsedTime = 0;
		try {
			newClient.m_InRaw = s.getInputStream();
			InputStreamReader isr = new InputStreamReader(s.getInputStream());
			newClient.m_In = new BufferedReader(isr);
			newClient.m_Out = new PrintWriter(s.getOutputStream(), true);
		} catch (Exception exc) {
			if(m_DebugLevel >= 2) {
				exc.printStackTrace();
			}
			System.out.println("Could not create I/O for client");
			return false;
		}

		sendToClient(newClient.m_Out, "WELCOME to the Minesweep server");

		System.out.println("Waiting for team name...");
		// The client should now send the name of the team
		String name = null;
		try {
			name = newClient.m_In.readLine();
		} catch (Exception exc) {
			System.out.println("Could not get agent name");
			return false;
		}

		if (name != null) {
			newClient.m_Name = name;
			System.out.println("Client with name \"" + name + "\" just joined");
		} else {
			System.out.println("Client name is null string");
			return false;
		}

		m_NumClients++;
		System.out.println("addClient: Now got " + m_NumClients + " clients");

		m_GameNumber = 0;
		m_GameId++;

		if (m_NumClients == 0) {
			m_Clients = new AgentClient[1];
			m_NumClients = 1;
			m_Clients[0] = newClient;
		} else {
			AgentClient ac[] = new AgentClient[m_NumClients];

			// We insert the new client based on the name so that the
			// list of client is sorted based on name. This way it is
			// more likely that the clients will be in the same order
			// in case they have to reconnect due to broken
			// connection, timeout, etc

			int j = 0;
			boolean addedNew = false;
			for (int i = 0; i < (m_NumClients - 1); i++) {
				if (newClient.m_Name.compareTo(m_Clients[i].m_Name) < 0) {
					ac[j] = newClient;
					ac[j].m_Id = j;
					j++;
					addedNew = true;
				}

				ac[j] = m_Clients[i];
				ac[j].m_Id = j;
				j++;
			}
			if (!addedNew) {
				ac[m_NumClients - 1] = newClient;
				ac[m_NumClients - 1].m_Id = m_NumClients - 1;
			}
			m_Clients = ac;
		}

		setChanged();
		notifyObservers(new ChangeMask(ChangeMask.CLIENTS_CHANGED));
		
		return true;
	}

	public void closeAllClients() {
		int n = m_NumClients;
		for (int i = 0; i < n; i++) {
			delClient(i);
		}
	}

	public boolean delClient(int i) {
		AgentClient oldClient = m_Clients[i];
		System.out.println("Removing client with name \"" + m_Clients[i].m_Name
				+ "\"");

		if (m_NumClients - 1 > 0) {
			AgentClient ac[] = new AgentClient[m_NumClients - 1];
			int n = 0;
			for (int j = 0; j < m_NumClients; j++) {
				if (j != i) {
					ac[n] = m_Clients[j];
					ac[n].m_Id = 0;
					n++;
				}
			}
			m_Clients = ac;
		} else {
			m_Clients = null;
		}
		m_NumClients--;

		try {
			oldClient.m_InRaw.close();
			oldClient.m_In.close();
			oldClient.m_Socket.close();
		} catch (Exception e) {
		}

		setChanged();
		notifyObservers(new ChangeMask(ChangeMask.CLIENTS_CHANGED));

		return true;
	}

	// / In this function we check for data from client(s)
	public void clientPlay() {
		m_StartTime = System.nanoTime();

		boolean run = true;
		boolean firstRound = true;

		if (m_DebugLevel >= 1) {
			System.out.println("Game number " + (m_GameNumber + 1));
		}

		m_StoppedGame = false;

		if (m_OnlySaveBoards) {
			clientGameOver(false);
			m_Playing = false;
			return;
		}

		while (run) {

			for (int i = 0; run && i < m_NumClients; i++) {

				if (m_StoppedGame) {
					if (m_DebugLevel >= 1) {
						System.out.println("Game stopped/timeout");
					}
					run = false;
					break;
				}

				long splitTime = System.nanoTime();

				// In the first round we skip players in such as way
				// that the pllayers take turn to start
				if (firstRound) {
					if (i < (m_GameNumber % m_NumClients)) {
						if (m_DebugLevel >= 3) {
							System.out.println("Skipping client " + i
									+ " for game #" + m_GameNumber);
						}
						continue;
					}
					firstRound = false;
				}

				if (!m_Clients[i].m_Active)
					continue;

				if (m_DebugLevel >= 3) {
					System.out.println("Sending MOVE to client " + i);
				}

				String msg = "MOVE";
				sendToClient(m_Clients[i].m_Out, msg);

				String move = "";
				try {
					move = m_Clients[i].m_In.readLine();
				} catch (Exception exc) {
					if (m_DebugLevel >= 3) {
						exc.printStackTrace();
					} else if (m_DebugLevel >= 2) {
						System.out
								.println("Exception in MinesweepEngine::clientPlay(): "
										+ exc.getMessage());
					}
					delClient(i);
					run = false;
					break;
				}

				if (m_DebugLevel >= 3) {
					System.out.println("Got response from client " + i);
				}

				if (move == null) {
					if (m_DebugLevel >= 2) {
						System.out
								.println("Null string read for move from client "
										+ i);
					}
					delClient(i);
					run = false;
					break;
				}

				if (m_StoppedGame) {
					if (m_DebugLevel >= 1) {
						System.out.println("Game stopped/timeout");
					}
					run = false;
					break;
				}

				long timeTaken =  System.nanoTime() - splitTime;
				m_Clients[i].m_UsedTime += timeTaken;
				m_Clients[i].m_TotalUsedTime += timeTaken;

				// System.out.println(move);

				String cmd[] = move.split(" ");

				if (cmd[0].equals("OPEN")) {
					int x = Integer.parseInt(cmd[1]);
					int y = Integer.parseInt(cmd[2]);
					int ret = m_Field.openSquare(x, y);

					if (ret == -10) {

						String repl = new String("ILLEGAL_MOVE "
								+ m_Clients[i].m_Id + " " + x + " " + y);
						sendToClient(m_Clients[i].m_Out, repl);
						// Let this player play again, sicne we do not
						// want clients using this as a strategy not
						// to have to say STOP
						i--;
					} else if (ret == -20) {
						if (m_DebugLevel >= 2) {
							System.out
									.println("Square already open, stop or open a !!");
						}
						String repl = new String("ALREADY_OPEN "
								+ m_Clients[i].m_Id + " " + x + " " + y);
						sendToClient(m_Clients[i].m_Out, repl);

						// Let this player play again, sicne we do not
						// want clients using this as a strategy not
						// to have to say STOP
						i--;
					} else {

						if (ret == -1) {
							m_Clients[i].m_Active = false;
							m_Clients[i].m_Score -= m_Field.getDeadCost();

							if (getNumActiveClients() <= 1) {
								clientGameOver(false);
								run = false;
								break;
							}
						} else if (ret == -2) {

						} else {
							// A safe square was open and we should
							// increment the score for the active
							// clients
							for (int j = 0; j < m_NumClients; j++) {
								if (m_Clients[j].m_Active) {
									m_Clients[j].m_Score++;
								}
							}
						}

						String repl = new String("SQUARE_INFO "
								+ m_Clients[i].m_Id + " " + x + " " + y + " "
								+ ret);
						sendToAllClients(repl);
					}
				} else if (cmd[0].equals("STOP")) {

					String repl = new String("PLAYER_STOPPED "
							+ m_Clients[i].m_Id);
					sendToAllClients(repl);
					m_Clients[i].m_Active = false;

					if (getNumActiveClients() == 0) {
						clientGameOver(false);
						run = false;
						break;
					}
				} else if (cmd[0].equals("PEEK")) {
					if (m_Clients[i].m_Peeked) {
						String repl = new String("ALREADY_PEEKED "
								+ m_Clients[i].m_Id);
						sendToClient(m_Clients[i].m_Out, repl);
					} else {
						int x = Integer.parseInt(cmd[1]);
						int y = Integer.parseInt(cmd[2]);
						int ret = m_Field.peekSquare(x, y);
						System.out.println("Client " + i + " peeking at " + x
								+ " " + y);
						String repl = new String("SQUARE_INFO "
								+ m_Clients[i].m_Id + " " + x + " " + y + " "
								+ ret);
						sendToClient(m_Clients[i].m_Out, repl);
						m_Clients[i].m_Peeked = true;
					}
					// This client should move again
					i--;
				} else {
					if (m_DebugLevel >= 2) {
						System.out.println("Unkown command \"" + move + "\"");
					}
					String repl = new String("UNKNOWN_CMD " + move);
					sendToClient(m_Clients[i].m_Out, repl);
					// This player has to go again
					i--;
				}

				// Check if all squares are open or the game is over
				if (m_Field.isAllClear() || m_Field.isDead()) {
					clientGameOver(false);
					run = false;
					break;
				}

				if (m_GameStepDelay > 0) {
					try {
						Thread.sleep(m_GameStepDelay);
					} catch (Exception exc) {
					}
				}
			}
		}

		System.out.println("Game finished");
		m_GameNumber++;
		m_Playing = false;
	}

	public void autoPlay() {
		m_StartTime = System.nanoTime();

		System.out.println("Starting autoplay");

		int playerID = 0;
		if (m_Agent.startNewGame(1, playerID, m_Field.getFieldXSize(), m_Field
				.getFieldYSize(), m_Field.getDeadCost(), m_Field.getPeekCost()) == false) {
			showMessage("Could not start agent", MinesweepUI.LEVEL_NOTIFY);
			return;
		}

		MineMove mm;

		m_AutoScore = 0;

		int nSquares = m_Field.getFieldXSize() * m_Field.getFieldYSize();

		boolean peeked = false;

		if (m_OnlySaveBoards) {
			gameOver(false);
			return;
		}

		while (true) {
			mm = m_Agent.getNextMove();

			if (mm == null || mm.m_Move == MineMove.STOP
					|| (nSquares - m_Field.getNumOpened() == 0)) {
				gameOver(false);
				return;
			}

			if (mm.m_Move == MineMove.OPEN_SQUARE) {
				int ret = m_Field.openSquare(mm.m_X, mm.m_Y);
				MineInfo mi = new MineInfo();
				mi.m_PlayerID = playerID;
				mi.m_X = mm.m_X;
				mi.m_Y = mm.m_Y;
				if (ret == -10) {
					mi.m_InfoType = MineInfo.ILLEGAL_MOVE;
				} else {

					if (ret == -1) {
						m_AutoScore -= m_Field.getDeadCost();
						gameOver(false);
						return;
					} else if (ret == -2) {
						showMessage("You should be dead, but you're not", MinesweepUI.LEVEL_NOTIFY);
					} else {
						m_AutoScore++;
					}
					mi.m_Value = ret;
					mi.m_InfoType = MineInfo.SQUARE_INFO;
				}
				m_Agent.handleInfo(mi);
			} else if (mm.m_Move == MineMove.PEEK_SQUARE) {
				MineInfo mi = new MineInfo();
				mi.m_PlayerID = playerID;
				if (peeked) {
					mi.m_InfoType = MineInfo.ALREADY_PEEKED;
				} else {
					mi.m_X = mm.m_X;
					mi.m_Y = mm.m_Y;
					mi.m_Value = m_Field.peekSquare(mm.m_X, mm.m_Y);
					mi.m_InfoType = MineInfo.SQUARE_INFO;
					peeked = true;
				}
				m_Agent.handleInfo(mi);
			}

			if (m_GameStepDelay > 0) {
				try {
					Thread.sleep(m_GameStepDelay);
				} catch (Exception exc) {
				}
			}
		}

	}

	/** Simply proxy to the currently active MinesweepUI, if any. */
	public void showMessage(String msg, int level) {
		if(m_UI != null)
			m_UI.showMessage(msg, level);
	}

	public void runBatchFileTest(String batchfile) {
		String statsfile = "";
		try {
			m_BatchFile = new FileReader(batchfile);
			m_BatchReader = new BufferedReader(m_BatchFile);
		} catch (Exception exc) {
			showMessage("Failed to open batch file \"" + batchfile + "\"", MinesweepUI.LEVEL_ERROR);
			return;
		}
		try {
			SimpleDateFormat dateFormat = new SimpleDateFormat(
					"yyyy-MM-dd_HH-mm-ss.SSS");

			statsfile = "batchresults";
			if(m_NumClients >= 1) {
				statsfile += "-" + m_Clients[0].m_Name;
			}
			if (m_NumClients == 2) {
				statsfile += "-vs-" + m_Clients[1].m_Name;
			}

			statsfile += "-" + dateFormat.format(new Date()) + ".txt";
			m_BatchResFile = new FileWriter(statsfile);
			m_BatchResWriter = new BufferedWriter(m_BatchResFile);
		} catch (Exception exc) {
			System.out.println("Failed to open statistics file \"" + statsfile
					+ "\"");
			return;
		}

		m_RandomEnabled = false;
		newGame();
		m_StartPlay = true;
	}

	public void clientGameOver(boolean timeout) {
		m_GameId++;
		m_StartTime = -1;
		m_Playing = false;

		int maxScore = m_Field.getMaxScore();
		String m = "Game finished\n" + "Max score for board " + maxScore + "\n";

		long totTime = 1;
		for (int i = 0; i < m_NumClients; i++) {
			totTime += m_Clients[i].m_UsedTime;
		}
		for (int i = 0; i < m_NumClients; i++) {
			m_Clients[i].m_TimeFraction = 1.0 * m_Clients[i].m_UsedTime
					/ totTime;
		}

		if (timeout && m_NumClients > 1) {
			if (m_Clients[0].m_TimeFraction > 2.0 / 3) {
				System.out.println("Punished agent \"" + m_Clients[0].m_Name
						+ "\" for using too much time");
				m_Clients[0].m_Score -= m_Field.getPeekCost();
			} else if (m_Clients[1].m_TimeFraction > 2.0 / 3) {
				System.out.println("Punished agent \"" + m_Clients[1].m_Name
						+ "\" for using too much time");
				m_Clients[1].m_Score -= m_Field.getPeekCost();
			}
		}

		if (m_NumClients == 1)
			writeResult(m_Clients[0].m_Score);
		else
			writeResult(-1);

		
		  if (!m_GameStopPressed) {
			  if (m_NumClients == 1) {
				  if(m_Clients[0].m_Score >= maxScore) {
					  // Count a complete sweep as a win.
					  // (Score may be larger if client initially steps on a mine)
					  ++m_Clients[0].m_NumWins;
				  }
			  } else if (m_NumClients == 2) {
				  boolean win1 = (m_Clients[0].m_Score > m_Clients[1].m_Score);
				  boolean win2 = (m_Clients[1].m_Score > m_Clients[0].m_Score);

				  if(win1) {
					  ++m_Clients[0].m_NumWins;
				  } else if(win2) {
					  ++m_Clients[1].m_NumWins;
				  }
			  }
			  
			  for(int i=0;i!=m_NumClients;++i) {
				  AgentClient c = m_Clients[i];
				  double frac = (double) c.m_Score / (double) maxScore;
				   c.m_AvgScore = (c.m_AvgScore * c.m_NumGames + frac) / (c.m_NumGames + 1);
				  ++c.m_NumGames;
			  }
			  
			  setChanged();
			  notifyObservers(new ChangeMask(ChangeMask.CLIENTS_CHANGED));
		  }
		 
		for (int i = 0; i < m_NumClients; i++) {
			if (m_NumClients == 1) {
				m += "\nClient" + i + " (\"" + m_Clients[i].m_Name
						+ "\") got score=" + m_Clients[i].m_Score + " time="
						+ (long) (1e-6 * m_Clients[i].m_UsedTime) + "ms";
			} else {
				m += "\nClient" + i + " (\"" + m_Clients[i].m_Name
						+ "\") got score=" + m_Clients[i].m_Score + " time="
						+ (long) (1e-6 * m_Clients[i].m_UsedTime) + "ms ("
						+ (long) (100.0 * m_Clients[i].m_TimeFraction) + "%)";
			}
			String msg = "DONE " + m_Clients[i].m_Score;
			sendToClient(m_Clients[i].m_Out, msg);
			m_Clients[i].m_Score = 0;
			m_Clients[i].m_Active = true;
			m_Clients[i].m_Peeked = false;
			m_Clients[i].m_UsedTime = 0;
		}

		showMessage(m, MinesweepUI.LEVEL_NOTIFY);

		System.out.println(m);

		if (!m_GameStopPressed) {
			if (m_BatchReader != null) {
				if(newGame()) {
					m_StartPlay = true;
				} else {
					if(m_UI != null) {
						m_UI.batchRunEnded();
					}
				}
			} else if (m_AutoRestart && !m_GameStopPressed) {
				if (m_PlayMode == SERVER_MODE) {
					newGame();
					m_StartPlay = true;
				}
			}
		}

	}

	public void gameOver(boolean calcScore) {
		m_GameId++;
		m_StartTime = -1;
		m_Playing = false;

		int score = m_AutoScore;
		if (calcScore) {
			score = m_Field.getNumOpened() - m_Field.getNumInitialMines();
			if (m_Field.isDead()) {
				score -= m_Field.getDeadCost();
			}
		}
		/*
		 * m_StatWin.addStats(1.0 * score / m_Field.getMaxScore(), false, 0,
		 * false);
		 */
		showMessage("Game finished\n" + "You got score " + score, MinesweepUI.LEVEL_NOTIFY);

		if (m_BatchReader != null) {
			writeResult(score);
			newGame();
			m_StartPlay = true;
		} else if (m_AutoRestart && !m_GameStopPressed) {
			if (m_PlayMode == AUTO_MODE) {
				newGame();
				m_StartPlay = true;
			}
		}
	}

	private void writeResult(int score) {
		if (m_BatchReader == null)
			return;

		String line = new String("");
		if (m_TwoPlayerGame && m_PlayMode == SERVER_MODE) {
			line += m_MineFile + " " + m_Field.getFieldXSize() + " "
					+ m_Field.getFieldYSize() + " " + m_Field.getMaxScore()
					+ " " + m_Field.getNumOpened() + " "
					+ m_Field.getNumInitialMines() + " " + m_Clients[0].m_Name
					+ " " + m_Clients[1].m_Name + " " + m_Clients[0].m_Score
					+ " " + m_Clients[1].m_Score + " "
					+ (long) (1e-6 * m_Clients[0].m_UsedTime) + " "
					+ (long) (1e-6 * m_Clients[1].m_UsedTime) + " "
					+ m_Clients[0].m_TimeFraction + " "
					+ m_Clients[1].m_TimeFraction;

		} else {

			line += m_MineFile + " " + m_Field.getFieldXSize() + " "
					+ m_Field.getFieldYSize() + " " + m_Field.getMaxScore()
					+ " " + m_Field.getNumOpened() + " "
					+ m_Field.getNumInitialMines() + " " + score + " "
					+ (double) score / (double) m_Field.getMaxScore();
		}
		try {
			m_BatchResWriter.write(line, 0, line.length());
			m_BatchResWriter.newLine();
			m_BatchResWriter.flush();
		} catch (Exception exc) {
			System.out.println("Failed to write result to file");
		}
	}

	private boolean readNextBatchFile() {
		try {
			m_MineFile = m_BatchReader.readLine();
		} catch (Exception e) {
		}

		return (m_MineFile != null);
	}

	public boolean newGame() {

		showMessage("newGame called", MinesweepUI.LEVEL_DEBUG);

		if (m_RandomEnabled == false) {

			if (m_BatchReader != null) {
				showMessage("In batch mode", MinesweepUI.LEVEL_DEBUG);
				if (!readNextBatchFile()) {
					m_BatchReader = null;
					m_StartPlay = false;
					m_GameStopPressed = true;
					m_Playing = false;
					m_StoppedGame = true;
					m_AutoRestart = false;
					showMessage("Out of batch files", MinesweepUI.LEVEL_DEBUG);
					try {
						m_BatchResWriter.close();
					} catch (Exception exc) {
					}

					showMessage("Batch test done", MinesweepUI.LEVEL_NOTIFY);
					return false;
				} else {
					showMessage("Found batch file \"" + m_MineFile + "\"", MinesweepUI.LEVEL_DEBUG);
				}
			}

			if (m_MineFile == null) {
				showMessage("You must specify mine file", MinesweepUI.LEVEL_ERROR);
			} else {
				if (m_Field.readMinesFromFile(m_MineFile)) {
					showMessage("Loaded mine file \"" + m_MineFile + "\"", MinesweepUI.LEVEL_DEBUG);
				} else {
					showMessage("Could not open mine file, playing same game again.", MinesweepUI.LEVEL_ERROR);
				}
			}
		} else {
			Random r = new Random();
			  if (m_RandomDensity) {
				  m_Field.setDensity(1e-3 * (long)(1e3 * (0.02 + 0.23*r.nextDouble())));
			  }
			  
			  if (m_RandomPeekCost) {
				  if (r.nextInt(10) > 8) {
					  	m_Field.setPeekCost(10 + r.nextInt(10));
				  } else {
					  m_Field.setPeekCost(r.nextInt(10));
			  }
			  
			  if (m_RandomDeadCost) {
				  m_Field.setDeadCost(5 + r.nextInt(15));
			  }
			  
			  if (m_RandomBoardSize) {
				  int xSize = 10 + r.nextInt(40);
				  int ySize = 10 + r.nextInt(40);
				  m_Field.setFieldXSize(xSize);
				  m_Field.setFieldYSize(ySize);
			  }
			  
			  m_Field.createRandomMines();
			  
			  if (m_SaveGameBoards) {
				  String filename = "minefile-" + m_GameId + ".msf";
				  m_Field.saveMinesToFile(filename);
				  // Add the settings to the minefile
				  try {
					  FileWriter fw = new FileWriter(filename, true);
					  fw.write("S " + m_Field.getDeadCost() + " " + m_Field.getPeekCost() + " " + m_Field.getDensity() + " " + "\n");
					  fw.close();
				  } catch (Exception e) {
				  }
			  }
			}
		}

		m_Field.buildField();
		return true;
	}

	public boolean getAutoRestart() {
		return m_AutoRestart;
	}

	public void setAutoRestart(boolean autoRestart) {
		m_AutoRestart = autoRestart;
	}

	public boolean getRandomBoardSize() {
		return m_RandomBoardSize;
	}

	public void setRandomBoardSize(boolean randomBoardSize) {
		m_RandomBoardSize = randomBoardSize;
	}

	public boolean getRandomDeadCost() {
		return m_RandomDeadCost;
	}

	public void setRandomDeadCost(boolean randomDeadCost) {
		m_RandomDeadCost = randomDeadCost;
	}

	public boolean getRandomDensity() {
		return m_RandomDensity;
	}

	public void setRandomDensity(boolean randomDensity) {
		m_RandomDensity = randomDensity;
	}

	public boolean getRandomEnabled() {
		return m_RandomEnabled;
	}

	public void setRandomEnabled(boolean randomEnabled) {
		m_RandomEnabled = randomEnabled;
		if(randomEnabled) {
			disableBatchMode();
		}
	}

	public boolean getRandomPeekCost() {
		return m_RandomPeekCost;
	}

	public void setRandomPeekCost(boolean randomPeekCost) {
		m_RandomPeekCost = randomPeekCost;
	}
	
	public boolean getSaveGameBoards() {
		return m_SaveGameBoards;
	}
	
	public void setSaveGameBoards(boolean save) {
		m_SaveGameBoards = save;
	}
	
	public boolean getOnlySaveBoards() {
		return m_OnlySaveBoards;
	}
	
	public void setOnlySaveBoards(boolean onlySave) {
		m_OnlySaveBoards = onlySave;
	}
};
