//
// = FILENAME
//    Minesweep.java
//
// = FUNCTION
//
//    This is the main control for the mine sweep process. You can run it in 
//    three different modes:
//    1. Manual mode: user clicks on the mines
//    2. Auto mode  : agent plays
//    3. Server mode: agent can connect via sockets and play
// 
// = AUTHOR(S)
//    Patric Jensfelt
//    Modifications by Oscar Sundbom
//
// = COPYRIGHT
//    Copyright (c) 2005 Patric Jensfelt
//
/*----------------------------------------------------------------------*/

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FileDialog;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;
import java.io.FilenameFilter;
import java.util.Observable;
import java.util.Observer;

import javax.swing.Box;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JTextField;

public class MinesweepGUI extends JFrame implements MinesweepUI, ItemListener,
                                                 ActionListener,
                                                 Observer {

	private MineFieldPanel  m_FieldPanel = null; 

    JCheckBoxMenuItem  m_UseRandom;

    private JTextField m_TextXSize;
    private JTextField m_TextYSize;
    private JTextField m_TextDens;
    private JTextField m_TextDeadCost;
    private JTextField m_TextPeekCost;
    private JTextField m_TextNumOpen;

    private JCheckBoxMenuItem m_SlowSpeedAuto = null;
    private JCheckBoxMenuItem m_TwoPlayerGame = null;
    private JCheckBoxMenuItem m_SaveGameBoards = null;
    private JCheckBoxMenuItem m_AutoRestart = null;
    private JCheckBoxMenuItem m_OnlySave = null;
    private JCheckBoxMenuItem m_LimitFramerateCheckbox = null;
    
    private JCheckBoxMenuItem m_ShowStats = null;

    private JCheckBoxMenuItem m_RandomPeekCost = null;
    private JCheckBoxMenuItem m_RandomDeadCost = null;
    private JCheckBoxMenuItem m_RandomDensity = null;
    private JCheckBoxMenuItem m_RandomBoardSize = null;

//    private Thread m_TimerThread = null;
    private StatWindow m_StatWin = null;
    
    private MinesweepEngine m_Engine;

    public MinesweepGUI(MinesweepEngine engine) {
        setTitle("Mine sweep");
        
        m_Engine = engine;
        
        //Create the menu bar.
        JMenuBar menuBar = new JMenuBar();

        JMenu fmenu = new JMenu("File");

        JMenuItem loadItem = new JMenuItem("Load mine file");
        loadItem.addActionListener(this);
        fmenu.add(loadItem);

        JMenuItem saveItem = new JMenuItem("Save mine file");
        saveItem.addActionListener(this);
        fmenu.add(saveItem);

        m_UseRandom = new JCheckBoxMenuItem("Randomize");
        m_UseRandom.addItemListener(this);
        m_UseRandom.setSelected(true);
        fmenu.add(m_UseRandom);

        JMenuItem batchItem = new JMenuItem("Run batch file test");
        batchItem.addActionListener(this);
        fmenu.add(batchItem);

        JMenuItem quitItem = new JMenuItem("Quit");
        quitItem.addActionListener(this);
        fmenu.add(quitItem);        

        menuBar.add(fmenu);        

        JMenu mmenu = new JMenu("Mode");

        ButtonGroup group = new ButtonGroup();
        JRadioButtonMenuItem rbMenuItem = new JRadioButtonMenuItem("Manual mode");
        rbMenuItem.setSelected(false);
        rbMenuItem.addItemListener(this);
        mmenu.add(rbMenuItem);
        group.add(rbMenuItem);
        rbMenuItem = new JRadioButtonMenuItem("Auto mode");
        rbMenuItem.setSelected(false);
        rbMenuItem.addItemListener(this);
        mmenu.add(rbMenuItem);
        group.add(rbMenuItem);
        rbMenuItem = new JRadioButtonMenuItem("Server mode");
        rbMenuItem.setSelected(true);
        rbMenuItem.addItemListener(this);
        mmenu.add(rbMenuItem);
        group.add(rbMenuItem);
        
        menuBar.add(mmenu);

        JMenu smenu = new JMenu("Settings");
        m_SlowSpeedAuto = new JCheckBoxMenuItem("Slow auto");
        m_SlowSpeedAuto.addItemListener(this);
        m_SlowSpeedAuto.setSelected(false);
        smenu.add(m_SlowSpeedAuto);
        m_LimitFramerateCheckbox = new JCheckBoxMenuItem("Limit frame rate");
        m_LimitFramerateCheckbox.addItemListener(this);
        m_LimitFramerateCheckbox.setSelected(false);
        smenu.add(m_LimitFramerateCheckbox);
        m_TwoPlayerGame = new JCheckBoxMenuItem("Two player game");
        m_TwoPlayerGame.addItemListener(this);
        m_TwoPlayerGame.setSelected(false);
        smenu.add(m_TwoPlayerGame);
        m_SaveGameBoards = new JCheckBoxMenuItem("Save game boards");
        m_SaveGameBoards.addItemListener(this);
        m_SaveGameBoards.setSelected(false);
        smenu.add(m_SaveGameBoards);
        m_AutoRestart = new JCheckBoxMenuItem("Auto restart");
        m_AutoRestart.addItemListener(this);
        m_AutoRestart.setSelected(true);
        smenu.add(m_AutoRestart);
        m_OnlySave = new JCheckBoxMenuItem("Only save files");
        m_OnlySave.addItemListener(this);
        m_OnlySave.setSelected(false);
        smenu.add(m_OnlySave);
        
        menuBar.add(smenu);


        smenu = new JMenu("Statistics");
        m_ShowStats = new JCheckBoxMenuItem("Show Statistics");
        m_ShowStats.setSelected(true);
        m_ShowStats.addActionListener(this);
        smenu.add(m_ShowStats);
        JMenuItem mi = new JMenuItem("Clear statistics");
        mi.addActionListener(this);
        smenu.add(mi);

        menuBar.add(smenu);


        JMenu rmenu = new JMenu("Random");
        m_RandomDensity = new JCheckBoxMenuItem("Random density");
        m_RandomDensity.addItemListener(this);
        m_RandomDensity.setSelected(true);
        rmenu.add(m_RandomDensity);
        m_RandomDeadCost = new JCheckBoxMenuItem("Random deadCost");
        m_RandomDeadCost.addItemListener(this);
        m_RandomDeadCost.setSelected(true);
        rmenu.add(m_RandomDeadCost);
        m_RandomPeekCost = new JCheckBoxMenuItem("Random peekCost");
        m_RandomPeekCost.addItemListener(this);
        m_RandomPeekCost.setSelected(true);
        rmenu.add(m_RandomPeekCost);
        m_RandomBoardSize = new JCheckBoxMenuItem("Random board size");
        m_RandomBoardSize.addItemListener(this);
        m_RandomBoardSize.setSelected(true);
        rmenu.add(m_RandomBoardSize);

        menuBar.add(rmenu);

        menuBar.add(Box.createHorizontalGlue());


        JMenu hmenu = new JMenu("Help");

        JMenuItem helpItem = new JMenuItem("Help");
        helpItem.addActionListener(this);
        hmenu.add(helpItem);

        JMenuItem aboutItem = new JMenuItem("About");
        aboutItem.addActionListener(this);
        hmenu.add(aboutItem);

        menuBar.add(hmenu);

        setJMenuBar(menuBar);

        getContentPane().setLayout(new BorderLayout());        

        JPanel buttons = new JPanel();
        JButton newGame = new JButton("New Game");
        newGame.addActionListener(this);
        buttons.add(newGame);        
        JButton stopGame = new JButton("Stop Game");
        stopGame.addActionListener(this);
        buttons.add(stopGame);        
        JButton openAll = new JButton("Open all");
        openAll.addActionListener(this);
        buttons.add(openAll);        

        JPanel ctrl = new JPanel();
        ctrl.setLayout(new GridLayout(3,2));

        ctrl.add(new JLabel("x-size"));
        m_TextXSize = new JTextField();
        m_TextXSize.addActionListener(this);
        ctrl.add(m_TextXSize);

        ctrl.add(new JLabel("density"));
        m_TextDens = new JTextField();
        m_TextDens.addActionListener(this);
        ctrl.add(m_TextDens);

        ctrl.add(new JLabel("y-size"));        
        m_TextYSize = new JTextField();
        m_TextYSize.addActionListener(this);
        ctrl.add(m_TextYSize);
        makeTextEditable(m_UseRandom.isSelected());

        ctrl.add(new JLabel("dead cost"));
        m_TextDeadCost = new JTextField();
        m_TextDeadCost.addActionListener(this);
        ctrl.add(m_TextDeadCost);

        ctrl.add(new JLabel("#open"));
        m_TextNumOpen = new JTextField("0");
        m_TextNumOpen.setEditable(false);
        ctrl.add(m_TextNumOpen);

        ctrl.add(new JLabel("peek cost"));
        m_TextPeekCost = new JTextField();
        m_TextPeekCost.addActionListener(this);
        ctrl.add(m_TextPeekCost);

        buttons.add(ctrl, BorderLayout.EAST);

        getContentPane().add(buttons, BorderLayout.SOUTH);

        m_FieldPanel = new MineFieldPanel(m_Engine);
        getContentPane().add( m_FieldPanel, BorderLayout.CENTER);

        m_Engine.addObserver(this);
        m_Engine.getField().addObserver(this);
        m_Engine.setUI(this);
        m_Engine.newGame();

        Dimension d = m_FieldPanel.getSize();
        setSize((int)d.getWidth(), (int)(d.getHeight() + 100));
        
        pack();

        m_StatWin = new StatWindow(m_Engine);
        m_StatWin.setVisible(m_ShowStats.isSelected());

    }

    public void setTwoPlayerGame(boolean v)
    {
        m_TwoPlayerGame.setSelected(v);
    }

    public void makeTextEditable(boolean e) {
        m_TextXSize.setEditable(e);
        m_TextYSize.setEditable(e);
        m_TextDens.setEditable(e);
    }

/*
    public boolean runBatchFileTest(String batchfile)
    {
    	String statsfile = "";
    	
    	try {
            m_BatchFile = new FileReader(batchfile);
            m_BatchReader = new BufferedReader(m_BatchFile);
        } catch (Exception exc) {
            showMessage("Failed to open batch file \"" + batchfile + "\"");
            return false;
        }

        try {
        	SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss.SSS");
            
        	statsfile = "batchresults-" + m_Clients[0].m_Name;
            if(m_NumClients == 2) {
            	statsfile += "-vs-" + m_Clients[1].m_Name;
            }
            
            statsfile += "-" + dateFormat.format(new Date()) +  ".txt";
            m_BatchResFile = new FileWriter(statsfile);
            m_BatchResWriter = new BufferedWriter(m_BatchResFile);
        } catch (Exception exc) {
            System.out.println("Failed to open statistics file \"" + statsfile + "\"");
            return false;
        }
            
        newGame();                
        m_StartPlay = true;

        return true;
    }
     */    

    public void actionPerformed(ActionEvent e) {
        if (e.getSource() == m_TextXSize) {
            System.out.println("X changed");
            try {
                int s = Integer.parseInt(m_TextXSize.getText());
                if (s < 0 || s > 50) {
                    System.out.println("xsize in [0,50]");
                    JOptionPane.showMessageDialog(this, "xsize in [0,50]");
                    m_TextXSize.setText(Integer.toString(m_Engine.getField().getFieldXSize()));
                } else {
                    m_Engine.getField().setFieldXSize(s);
                    m_Engine.newGame();
                }
            } catch (Exception ne) {
                JOptionPane.showMessageDialog(this, "xsize must be integer");
                m_TextXSize.setText(Integer.toString(m_Engine.getField().getFieldXSize()));
            }
        } else if (e.getSource() == m_TextYSize) {
            System.out.println("Y changed");
            try {
                int s = Integer.parseInt(m_TextYSize.getText());
                if (s < 0 || s > 50) {
                    System.out.println("ysize in [0,50]");
                    JOptionPane.showMessageDialog(this, "ysize in [0,50]");
                    m_TextYSize.setText(Integer.toString(m_Engine.getField().getFieldYSize()));
                } else {
                	m_Engine.getField().setFieldYSize(s);
                	m_Engine.newGame();
                }
            } catch (Exception ne) {
                JOptionPane.showMessageDialog(this, "ysize must be integer");
                m_TextYSize.setText(Integer.toString(m_Engine.getField().getFieldYSize()));
            }
        } else if (e.getSource() == m_TextDens) {
            System.out.println("Density changed");
            try {
                double d = Double.parseDouble(m_TextDens.getText());
                if (d < 0) {
                    System.out.println("density < 0");
                    JOptionPane.showMessageDialog(this, "0.0<density<1.0");
                    m_TextDens.setText(Double.toString(m_Engine.getField().getDensity()));
                } else {
                	m_Engine.getField().setDensity(d);
                    m_Engine.newGame();
                }
            } catch (Exception ne) {
                JOptionPane.showMessageDialog(this, "density must be integer");
                m_TextDens.setText(Double.toString(m_Engine.getField().getDensity()));
            }
        } else if (e.getSource() == m_TextDeadCost) {
            System.out.println("Dead cost changed");
            try {
                int c = Integer.parseInt(m_TextDeadCost.getText());
                m_Engine.getField().setDeadCost(c);
                m_Engine.newGame();
            } catch (Exception ne) {
                JOptionPane.showMessageDialog(this, 
                                              "Failed to set dead cost");
                m_TextDeadCost.setText(Integer.toString(m_Engine.getField().getDeadCost()));
            }
        } else if (e.getSource() == m_TextPeekCost) {
            System.out.println("Peek cost changed");
            try {
                int c = Integer.parseInt(m_TextPeekCost.getText());
                m_Engine.getField().setPeekCost(c);
                m_Engine.newGame();
            } catch (Exception ne) {
                JOptionPane.showMessageDialog(this, 
                                              "Failed to set peek cost");
                m_TextPeekCost.setText(Integer.toString(m_Engine.getField().getPeekCost()));
            }
        } else if (e.getSource() == m_ShowStats) {
            m_StatWin.setVisible(m_ShowStats.isSelected());
        } else if ("New Game".equals(e.getActionCommand())) {
        	int playMode = m_Engine.getPlayMode(); 
        	
            if (playMode == MinesweepEngine.SERVER_MODE) {
                if (!m_TwoPlayerGame.isSelected() && m_Engine.getNumClients() < 1) {
                    showMessage("Must have a client before we can start", LEVEL_ERROR);
                    return;
                } else if (m_TwoPlayerGame.isSelected() && m_Engine.getNumClients() < 2) {
                    showMessage("Must have 2 clients to play, have " +
                    		m_Engine.getNumClients() + " now", LEVEL_ERROR);
                    return;
                }
                
            }

            m_Engine.newGame();
            m_Engine.startGame();
 
        } else if ("Open all".equals(e.getActionCommand())) {

            m_Engine.getField().openAll();

        } else if ("Stop Game".equals(e.getActionCommand())) {

            m_Engine.stopGame();

        } else if ("Quit".equals(e.getActionCommand())) {
            System.out.println("Quitting application");
            System.exit(0);
        } else if ("Run batch file test".equals(e.getActionCommand())) {

        	if (m_Engine.getPlayMode() == MinesweepEngine.MANUAL_MODE) {
                showMessage("You cannot run batch test in manual mode", LEVEL_ERROR);
                return;
            }

            class SuffixFilter implements FilenameFilter {
                public boolean accept(File dir, String name) {
                    return name.endsWith(".msb");
                }
            }
            FileDialog fd = new FileDialog(this, "FileDialog");            
            fd.setMode( FileDialog.LOAD );
            fd.setFilenameFilter(new SuffixFilter());
            fd.setVisible(true);
            if ( fd.getDirectory() != null && fd.getFile() != null) {
                m_Engine.runBatchFileTest(fd.getDirectory() + fd.getFile());
            } else {
//                m_BatchReader = null;
                System.out.println("Load batch file aborted");
            }

        } else if ("Load mine file".equals(e.getActionCommand())) {

        	m_Engine.disableBatchMode();
        	
        	class SuffixFilter implements FilenameFilter {
                public boolean accept(File dir, String name) {
                    return name.endsWith(".msf");
                }
            }
            FileDialog fd = new FileDialog(this, "FileDialog");            
            fd.setMode( FileDialog.LOAD );
            fd.setFilenameFilter(new SuffixFilter());
            fd.setVisible(true);
            if ( fd.getDirectory() != null && fd.getFile() != null) {
            	// FIXME: Maybe wrong!
            	m_Engine.getField().readMinesFromFile(fd.getDirectory() + fd.getFile());
            	
//            	m_MineFile = new String(fd.getDirectory() + fd.getFile());
                m_Engine.setRandomEnabled(false);
                makeTextEditable(false);
                m_Engine.newGame();
            } else {
                System.out.println("Load mine file aborted");
            }

        } else if ("Save mine file".equals(e.getActionCommand())) {
            class SuffixFilter implements FilenameFilter {
                public boolean accept(File dir, String name) {
                    return name.endsWith(".msf");
                }
            }
            FileDialog fd = new FileDialog(this, "FileDialog");            
            fd.setMode( FileDialog.SAVE );
            fd.setFilenameFilter(new SuffixFilter());
            fd.setVisible(true);
            if ( fd.getDirectory() != null && fd.getFile() != null) {
                m_Engine.getField().saveMinesToFile(new String(fd.getDirectory() + 
                                                   fd.getFile()));
            } else {
                System.out.println("Load mine file aborted");
            }

        } else if ("Help".equals(e.getActionCommand())) {
            JOptionPane.showMessageDialog(this, 
                                          "New Game - Start a new game (reads from file if specified)\n" +
                                          "End Game - Open all squares\n" +
                                          "Auto - If checked function autoPlay will be used to play\n" + 
                                          "Density - one mine per density squares\n" +
                                          "\n" +
                                          "You should implement a MineAgent and assign it to m_Agent!!\n" +
                                          "You are of course free to write your own code from scratch\n");
        } else if ("About".equals(e.getActionCommand())) {
            JOptionPane.showMessageDialog(this, 
                                          "Written by Patric Jensfelt 2005\n" +
                                          "for the AI course");
        } else if ("Clear statistics".equals(e.getActionCommand())) {

        	m_Engine.clearStats();

        } else {
            System.out.println("Action \"" + e.getActionCommand() +
                               "\" not implemented");
        }       
    }

    public void itemStateChanged(ItemEvent e) {
    	Object item = e.getItem();
    	if(item instanceof JRadioButtonMenuItem) {
            if (e.getStateChange() == ItemEvent.SELECTED) {
	            JRadioButtonMenuItem rb = (JRadioButtonMenuItem)e.getItem();
	            if ("Manual mode".equals(rb.getText())) {
	                m_Engine.setPlayMode(MinesweepEngine.MANUAL_MODE);
	                System.out.println("Manual mode");
	                m_FieldPanel.setAllowMouseOpen(true);
	                makeTextEditable(true);
	
	            } else if ("Auto mode".equals(rb.getText())) {
	                m_FieldPanel.setAllowMouseOpen(false);
	                m_Engine.setPlayMode(MinesweepEngine.AUTO_MODE);
	
	                showMessage("Exchange StupidAgent for your agent", LEVEL_NOTIFY);
	
	                // Write your own MineAgent and change the line below to use
	                // your version instead of StupidAgent
	                m_Engine.setAgent(new StupidAgent());
	        
	                System.out.println("Auto mode");
	            } else if ("Server mode".equals(rb.getText())) {
	                m_FieldPanel.setAllowMouseOpen(false);
	                System.out.println("Server mode");
	                m_Engine.setPlayMode(MinesweepEngine.SERVER_MODE);
	            }
            }
    	} else {
    		if (item == m_OnlySave) {
                if (m_OnlySave.isSelected()) {
                    m_SaveGameBoards.setSelected(true);
                }
                m_Engine.setOnlySaveBoards(m_OnlySave.isSelected());

    		} else if (item == m_SlowSpeedAuto) {
            	m_Engine.setGameStepDelay(m_SlowSpeedAuto.isSelected()?25:0);
            } else if (item == m_AutoRestart) {
            	m_Engine.setAutoRestart(m_AutoRestart.isSelected());
            } else if (item == m_LimitFramerateCheckbox) {
            	m_FieldPanel.setLimitFramerate(m_LimitFramerateCheckbox.isSelected());
            } else if (item == m_TwoPlayerGame) {
            	m_Engine.setTwoPlayerGame(m_TwoPlayerGame.isSelected());
            } else if (item == m_SaveGameBoards) {
                m_Engine.setSaveGameBoards(m_SaveGameBoards.isSelected());
            } else if (item == m_UseRandom) {
        		m_Engine.setRandomEnabled(m_UseRandom.isSelected());
            } else if(item == m_RandomBoardSize) {
            	m_Engine.setRandomBoardSize(m_RandomBoardSize.isSelected());
            } else if(item == m_RandomDeadCost) {
            	m_Engine.setRandomDeadCost(m_RandomDeadCost.isSelected());
            } else if(item == m_RandomDensity) {
            	m_Engine.setRandomDensity(m_RandomDensity.isSelected());
            } else if(item == m_RandomPeekCost) {
            	m_Engine.setRandomPeekCost(m_RandomPeekCost.isSelected());
            } else {
                showMessage("Unkown item changed state", LEVEL_DEBUG);
            }
    	}
        return;
    }

    //@Override
    public void showMessage(String msg, int level) {
        // Do not show popups in batch mode
        if((level == LEVEL_NOTIFY || level == LEVEL_ERROR) &&
           !(m_Engine.inBatchMode() || m_AutoRestart.isSelected())) {
            
            JOptionPane.showMessageDialog(this, msg);
        } else {
            System.out.println(msg);
        }
    }
    
    //@Override
    public void batchRunEnded() {
    	
    }
    
    public void update(Observable obs, Object obj) {
    	MineField field = m_Engine.getField();
    	if(obs == field) {
    		MineField.ChangeMask mask = (MineField.ChangeMask) obj;
    		if(mask.getFieldStateChanged()) {
    	        m_TextNumOpen.setText(Integer.toString(m_Engine.getField().getNumOpened()));
    		}
    		
    		if(mask.getFieldParametersChanged()) {
    	        m_TextXSize.setText(Integer.toString(m_Engine.getField().getFieldXSize()));
    	        m_TextYSize.setText(Integer.toString(m_Engine.getField().getFieldYSize()));

    	        m_TextPeekCost.setText(Integer.toString(m_Engine.getField().getPeekCost()));
    	        m_TextDeadCost.setText(Integer.toString(m_Engine.getField().getDeadCost()));
    	        
    	        if(m_Engine.getField().getDensity() >= 0.0) {
    	        	m_TextDens.setText(Double.toString(m_Engine.getField().getDensity()));
    	        } else {
    	        	m_TextDens.setText("?");
    	        }
    		}
		} else if(obs == m_Engine) {
    		MinesweepEngine.ChangeMask mask = (MinesweepEngine.ChangeMask) obj;
    		if(mask.getTimerChanged()) {
/*
    			if(m_Engine.getNumClients() == 2) {
    				m_StatWin.setTime(m_Engine.getClient(0).getUsedTimeSecs(),
    											m_Engine.getClient(1).getUsedTimeSecs());
    			} else {
    				m_StatWin.setTime(m_Engine.getGameTimeSec(), m_Engine.getMaxGameTimeSec());
    			}
*/
    			m_StatWin.updateTimes();
    		}
    		
    		if(mask.getClientsChanged()) {
/*
    			int clients = m_Engine.getNumClients();
    			if(clients == 0) {
    				m_StatWin.setTeamNames("-", "-");
    				m_StatWin.setTime(m_Engine.getGameTimeSec(), m_Engine.getMaxGameTimeSec());
    			} else if(clients == 1) {
    				MinesweepEngine.AgentClient ac = m_Engine.getClient(0);
    				m_StatWin.setTeamNames(ac.getName(), "-");
    			}  else if(clients == 2) {
    				MinesweepEngine.AgentClient ac = m_Engine.getClient(0);
    				MinesweepEngine.AgentClient ac2 = m_Engine.getClient(1);
    				m_StatWin.setTeamNames(ac.getName(), ac2.getName());
    			}
*/
    			m_StatWin.updateStats();
    		}
		}
	}

};

