% [F, N, T, Fo, No, vp, wp] = dynamics(ROBOT, SPEED, ACC, POS1,
% POS2, GRAV)
%
% calculates the (F)orces, mome(N)ts, and (T)ourques for robot
% ROBOT with current speed and acceleration SPEED and ACC (in terms
% of angular velocities and accelerations for the joints).
%
% (The remaining outputs are used for debugging)
% GRAV can be used to specify a gravitation vector.
%
% POS1 and POS2 are the positions of the centers of mass of the
% cubes. For details see initialize_robot.m
%
% All values are indexed in terms of joint indexes.
%
% The algorithm used is as described on p.200 in J.J.Craig (2.ed)
%
% This implementation is customized for the high-performance
% manipulator 'Popeye', but should be easy to modify
% for use with other configurations, as it is written in a
% fairly general formulation.

global cube_mass cube_size cube_moment dyn_inertia_calculated dyn_I

function [f, n, t, F, N, vp, wp] = dynamics(robot, tp, tpp, POS1, POS2, grav_vec)

% globals are used both to pass parameters and as statics, making
% all subsequent calls of this function faster.

global cube_mass cube_size cube_moment ...
       dyn_inertia_calculated dyn_I Pc

if(length(dyn_inertia_calculated)<2)
  dyn_inertia_calculated = zeros(1,6);
  Pc = zeros(3,7);
end

g = 9.81;

if nargin<6
  grav_vec = [0 0 g]';
end


F = zeros(3,6);
N = zeros(3,6);
T = zeros(1,6);

%output
f = zeros(3,6);
n = zeros(3,6);
t = zeros(1,6);

w = zeros(3,7);
v = zeros(3,7);
wp = zeros(3,7);
vp = [grav_vec zeros(3,6)];

% used to rtoate the gravity vector
% g_t = 45*pi/180;
% grav_rot = [1 0 0; 0 cos(g_t) -sin(g_t); 0 sin(g_t) cos(g_t)];
% vp(:,1)=grav_rot*vp(:,1);


vpc = zeros(3,7);
Z = [0 0 1]';
P = zeros(3,7);

for k = 0:5

  i0 = k+1;    % used for variables that need to be indexed from 0

  % Generate rotation matrix R{k+1} (meaning the rotation mapping
  % from frame k to k+1)
  T = generate_T(k+1,k,robot);
  R{k+1} = T(1:3,1:3);
  Rk=R{k+1};
  % calculate omega (w) i+1
  w(:,i0+1) = Rk*w(:,i0) + tp(k+1)*Z;
  
  % calculate domega/dt (wp) i+1
  wp(:,i0+1) = Rk*wp(:,i0) +...
      crossF((Rk*w(:,i0)),(tp(k+1)*Z)) +...
      tpp(k+1)*Z;
  
  % calculate dv/dt (vp) i+1
  Tinv=inv(T);
  P(:,i0)=Tinv(1:3,4);
  vp(:,i0+1) = Rk * (...
      crossF(wp(:,i0),P(:,i0))+...
      crossF(w(:,i0),crossF(w(:,i0),P(:,i0)))+...
      vp(:,i0)  );
  
  % calculate dv_c/dt (vpc) i+1
  % note that the outermost cubes follow a different configuration pattern.
  if(dyn_inertia_calculated(i0)==0)
    if (k<4)
      Pc(:,k+1) = (POS2(1:3,i0)*cube_mass(i0)+POS1(1:3,i0+1)*cube_mass(i0+1))/...
	  sum(cube_mass([i0 i0+1]));
      sum(cube_mass([i0 i0+1]))
    else
      Pc(:,k+1) = POS2(1:3,i0);
      cube_mass(i0)
    end
  end
  

  
  vpc(:,i0+1) = crossF(wp(:,i0+1),Pc(:,k+1)) +...
      crossF(w(:,i0+1),crossF(w(:,i0+1),Pc(:,k+1)))+...
      vp(:,i0+1);
  
  % calculate F i+1
  if (k<4)
    F(:,k+1) = sum(cube_mass([i0 i0+1]))*vpc(:,i0+1);
  else
    F(:,k+1) = cube_mass(i0)*vpc(:,i0+1);
  end
          
  % Calculate N i+1 (note the calculation of the inertia tensors in
  % matrix I)
  if(dyn_inertia_calculated(i0)==0)
    dyn_inertia_calculated(i0)=1;
    i_kub1 = (cube_mass(i0)/12)*(2*(cube_size(i0)^2));
    cm1 = cube_mass(i0);
    p_kub1 = POS2(1:3,i0)-Pc(:,k+1);
    if (k<4)
      i_kub2 = cube_mass(i0+1)/12*(2*(cube_size(i0+1)^2));
      cm2 = cube_mass(i0+1);
      p_kub2 = POS1(1:3,i0+1)-Pc(:,k+1);
    else
      i_kub2 = 0;
      cm2 = 0;
      p_kub2 = [0 0 0]';
    end  

    dyn_I{k+1} = cm1*(p_kub1'*p_kub1*eye(3) - p_kub1*p_kub1') + (i_kub1+i_kub2)*eye(3) +...
	cm2*(p_kub2'*p_kub2*eye(3) - p_kub2*p_kub2');
  end
    
    
  N(:,k+1) =dyn_I{k+1}*wp(:,i0+1) + ...
	   crossF(w(:,i0+1),dyn_I{k+1}*w(:,i0+1));

end

f(:,6) = F(:,6);
n(:,6) = N(:,6) + crossF(Pc(:,6),F(:,6));
t(6) = n(3,6);

for k=5:-1:1
  Ri = inv(R{k+1});
  f(:,k) = Ri*f(:,k+1)+F(:,k);
  n(:,k) = N(:,k) + Ri*n(:,k+1) + crossF(Pc(:,k),F(:,k)) + ...
	   crossF(P(:,k+1),Ri*f(:,k+1));
  t(k) = n(3,k);
end
