% function THETAS = inv_kin(POS, t4, PARAMS)
%
% This function calculates the theta values (angles) for a
% Puma560-type robot when the desired position and orientation of
% the end effector is given as the 6*1 position vector POS.
% The form of POS should be [x y z p t a]', where [x y z] is 
% the cartesian position of the end effector, [p t a] is the
% orinetation. 
%
% [p t a] = [0 0 0] will point straight up (z-direction), 
% [p t a] = [PI/2 PI/2 0] will point along the positive y axis 
%
% The parameter t4 is the current value for theta_4. When the
% configurations is in or close to a singular position, the value
% for theta_4 is left unchanged. 
%
% The vector PARAMS = [a2 d3 d4 d6] gives the values for these 
% parameters. These values default to 0.31, 0, 0.51, and 0.09 resp.
%
% The calculation follows the formulas derived by J.J. Craig

function thetas = inv_kin(pos, t4, params)

if (nargin <3)
  params = [0.31 0 0.51 0.09];
end

% Transform [p t a] to a rotation matrix representation:

p = pos(4);
t = pos(5);
a = pos(6);

% calculate values often needed:
sp=sin(p);
st=sin(t);
sa=sin(a);
cp=cos(p);
ct=cos(t);
ca=cos(a);
  
%create rotation matrix:
r13=cp*st;
r23=sp*st;
r33=ct;
r12=-cp*ct*sa - sp*ca;
r22=cp*ca - sp*ct*sa;
r32=st*sa;

% calculate the last column from the cross product
r11 = r22*r33 - r23*r32;
r21 = r32*r13 - r12*r33;
r31 = r12*r23 - r13*r22;
  
% Get position of end effector (tool)
pxt = pos(1);
pyt = pos(2);
pzt = pos(3);

% get D-H params
a2 = params(1);
d3 = params(2);
d4 = params(3);
d6 = params(4);
 
% Calculate position of joint 4
px=pxt - (d6*r13);
py=pyt - (d6*r23);
pz=pzt - (d6*r33);
  

px2=px*px;
py2=py*py;
pz2=pz*pz;

d32=d3*d3;

a_1 = atan2(py,px);

t1_a = a_1 - atan2(-d3,sqrt(px2+py2-d32));
t1_b = a_1 - atan2(-d3,-sqrt(px2+py2-d32));

K3 = (a2*a2+d32+d4*d4 - (px2+py2+pz2))/(2*d4*a2);

t3_a = atan2(K3,sqrt(1-(K3*K3)));
t3_b = atan2(K3,-sqrt(1-(K3*K3)));


c1_a = cos(t1_a);
c1_b = cos(t1_b);
s1_a = sin(t1_a);
s1_b = sin(t1_b);

c3_a = cos(t3_a);
c3_b = cos(t3_b);
s3_a = K3;
s3_b = K3;

% Used for debugging purposes:   %%%%%%%%%%%%%%%%%%%%%%
 s3_a_diff = K3 - sin(t3_a);                          %
 s3_b_diff = K3 - sin(t3_b);                          %
 if(abs(s3_a_diff)>0.0001)|(abs(s3_b_diff)>0.0001)    %
   K3                                                 %
   t3_a                                               %
   t3_b                                               %
   sin(t3_a)                                          %
   sin(t3_b)                                          %
 end                                                  %
% end debug   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


s23_aa = ((-a2*c3_a)*pz - (c1_a*px+s1_a*py)*(d4-a2*s3_a))/...
	 (pz2+(c1_a*px+s1_a*py)^2);
s23_ba = ((-a2*c3_a)*pz - (c1_b*px+s1_b*py)*(d4-a2*s3_a))/...
	 (pz2+(c1_b*px+s1_b*py)^2);
s23_ab = ((-a2*c3_b)*pz - (c1_a*px+s1_a*py)*(d4-a2*s3_b))/...
	 (pz2+(c1_a*px+s1_a*py)^2);
s23_bb = ((-a2*c3_b)*pz - (c1_b*px+s1_b*py)*(d4-a2*s3_b))/...
	 (pz2+(c1_b*px+s1_b*py)^2);


c23_aa = ((a2*s3_a-d4)*pz + (a2*c3_a*(c1_a*px+s1_a*py)))/...
	 (pz2+(c1_a*px+s1_a*py)^2);
c23_ba = ((a2*s3_a-d4)*pz + (a2*c3_a*(c1_b*px+s1_b*py)))/...
	 (pz2+(c1_b*px+s1_b*py)^2);
c23_ab = ((a2*s3_b-d4)*pz + (a2*c3_b*(c1_a*px+s1_a*py)))/...
	 (pz2+(c1_a*px+s1_a*py)^2);
c23_bb = ((a2*s3_b-d4)*pz + (a2*c3_b*(c1_b*px+s1_b*py)))/...
	 (pz2+(c1_b*px+s1_b*py)^2);

t23_aa = atan2(s23_aa,c23_aa);
t23_ba = atan2(s23_ba,c23_ba);
t23_ab = atan2(s23_ab,c23_ab);
t23_bb = atan2(s23_bb,c23_bb);

t2_aa = t23_aa-t3_a;
t2_ab = t23_ab-t3_b;
t2_ba = t23_ba-t3_a;
t2_bb = t23_bb-t3_b;

alpha_a = -r13*s1_a+r23*c1_a;
alpha_b = -r13*s1_b+r23*c1_b;

beta_aa = -r13*c1_a*c23_aa - r23*s1_a*c23_aa + r33*s23_aa;
beta_ba = -r13*c1_b*c23_ba - r23*s1_b*c23_ba + r33*s23_ba;
beta_ab = -r13*c1_a*c23_ab - r23*s1_a*c23_ab + r33*s23_ab;
beta_bb = -r13*c1_b*c23_bb - r23*s1_b*c23_bb + r33*s23_bb;

tol_4 = 0.0000001;

if(abs(alpha_a)<tol_4)
  if(abs(beta_aa)<tol_4)
    t4_aa = t4;
    warning('t4 collapse!')
  else
    t4_aa = atan2(alpha_a,beta_aa);
  end
  if(abs(beta_ab)<tol_4)
    t4_ab = t4;
    warning('t4 collapse!')
  else
    t4_ab = atan2(alpha_a,beta_ab);
  end
else
  t4_aa = atan2(alpha_a,beta_aa);
  t4_ab = atan2(alpha_a,beta_ab);
end

if(abs(alpha_b)<tol_4)
  if(abs(beta_ba)<tol_4)
    t4_ba = t4;
    warning('t4 collapse!')
  else
    t4_ba = atan2(alpha_b,beta_ba);
  end
  if(abs(beta_bb)<tol_4)
    t4_bb = t4;
    warning('t4 collapse!')
  else
    t4_bb = atan2(alpha_b,beta_bb);
  end
else
  t4_ba = atan2(alpha_b,beta_ba);
  t4_bb = atan2(alpha_b,beta_bb);
end

c4_aa = cos(t4_aa);
c4_ba = cos(t4_ba);
c4_ab = cos(t4_ab);
c4_bb = cos(t4_bb);

s4_aa = sin(t4_aa);
s4_ba = sin(t4_ba);
s4_ab = sin(t4_ab);
s4_bb = sin(t4_bb);

s5_aa = (c4_aa*c23_aa*c1_a+s4_aa*s1_a)*r13 +...
	(c4_aa*c23_aa*s1_a-s4_aa*c1_a)*r23 -...
	(c4_aa*s23_aa)*r33;
s5_ba = (c4_ba*c23_ba*c1_b+s4_ba*s1_b)*r13 +...
	(c4_ba*c23_ba*s1_b-s4_ba*c1_b)*r23 -...
	(c4_ba*s23_ba)*r33;
s5_ab = (c4_ab*c23_ab*c1_a+s4_ab*s1_a)*r13 +...
	(c4_ab*c23_ab*s1_a-s4_ab*c1_a)*r23 -...
	(c4_ab*s23_ab)*r33;
s5_bb = (c4_bb*c23_bb*c1_b+s4_bb*s1_b)*r13 +...
	(c4_bb*c23_bb*s1_b-s4_bb*c1_b)*r23 -...
	(c4_bb*s23_bb)*r33;

c5_aa = (-s23_aa*c1_a)*r13 +...
	(-s23_aa*s1_a)*r23 -...
	(c23_aa)*r33;
c5_ba = (-s23_ba*c1_b)*r13 +...
	(-s23_ba*s1_b)*r23 -...
	(c23_ba)*r33;
c5_ab = (-s23_ab*c1_a)*r13 +...
	(-s23_ab*s1_a)*r23 -...
	(c23_ab)*r33;
c5_bb = (-s23_bb*c1_b)*r13 +...
	(-s23_bb*s1_b)*r23 -...
	(c23_bb)*r33;


t5_aa = atan2(s5_aa,c5_aa);
t5_ba = atan2(s5_ba,c5_ba);
t5_ab = atan2(s5_ab,c5_ab);
t5_bb = atan2(s5_bb,c5_bb);

s6_aa = (-s4_aa*c23_aa*c1_a + c4_aa*s1_a)*r11+...
	(-s4_aa*c23_aa*s1_a - c4_aa*c1_a)*r21+...
	(s4_aa*s23_aa)*r31;
s6_ba = (-s4_ba*c23_ba*c1_b + c4_ba*s1_b)*r11+...
	(-s4_ba*c23_ba*s1_b - c4_ba*c1_b)*r21+...
	(s4_ba*s23_ba)*r31;
s6_ab = (-s4_ab*c23_ab*c1_a + c4_ab*s1_a)*r11+...
	(-s4_ab*c23_ab*s1_a - c4_ab*c1_a)*r21+...
	(s4_ab*s23_ab)*r31;
s6_bb = (-s4_bb*c23_bb*c1_b + c4_bb*s1_b)*r11+...
	(-s4_bb*c23_bb*s1_b - c4_bb*c1_b)*r21+...
	(s4_bb*s23_bb)*r31;

c6_aa = ((c5_aa*c4_aa*c23_aa+s5_aa*s23_aa)*c1_a + c5_aa*s4_aa*s1_a)*r11+...
	((c5_aa*c4_aa*c23_aa+s5_aa*s23_aa)*s1_a - c5_aa*s4_aa*c1_a)*r21+...
	(-c5_aa*c4_aa*s23_aa+s5_aa*c23_aa)*r31;

c6_ba = ((c5_ba*c4_ba*c23_ba+s5_ba*s23_ba)*c1_b + c5_ba*s4_ba*s1_b)*r11+...
	((c5_ba*c4_ba*c23_ba+s5_ba*s23_ba)*s1_b - c5_ba*s4_ba*c1_b)*r21+...
	(-c5_ba*c4_ba*s23_ba+s5_ba*c23_ba)*r31;

c6_ab = ((c5_ab*c4_ab*c23_ab+s5_ab*s23_ab)*c1_a + c5_ab*s4_ab*s1_a)*r11+...
	((c5_ab*c4_ab*c23_ab+s5_ab*s23_ab)*s1_a - c5_ab*s4_ab*c1_a)*r21+...
	(-c5_ab*c4_ab*s23_ab+s5_ab*c23_ab)*r31;

c6_bb = ((c5_bb*c4_bb*c23_bb+s5_bb*s23_bb)*c1_b + c5_bb*s4_bb*s1_b)*r11+...
	((c5_bb*c4_bb*c23_bb+s5_bb*s23_bb)*s1_b - c5_bb*s4_bb*c1_b)*r21+...
	(-c5_bb*c4_bb*s23_bb+s5_bb*c23_bb)*r31;

t6_aa = atan2(s6_aa,c6_aa);
t6_ba = atan2(s6_ba,c6_ba);
t6_ab = atan2(s6_ab,c6_ab);
t6_bb = atan2(s6_bb,c6_bb);

thetas = [t1_a  t1_b  t1_a  t1_b;...
	  t2_aa t2_ba t2_ab t2_bb;...
	  t3_a t3_a t3_b t3_b;...
	  t4_aa t4_ba t4_ab t4_bb;...
	  t5_aa t5_ba t5_ab t5_bb;...
	  t6_aa t6_ba t6_ab t6_bb];

% generate all the flipped configurations
for k = 1:4
  thetas = [thetas thetas(:,k).*[1 1 1 1 -1 1]'+[0 0 0 pi 0 pi]'];
end


% Normalize angles

thetas(find(thetas>pi)) = thetas(find(thetas>pi))-(2*pi);
thetas(find(thetas<-pi)) = thetas(find(thetas<-pi))+(2*pi);