% result = plot_robot3(ROBOT, POS1, POS2)
%
% Draws 3D plot of robot whose Denavit-Hartenberg representation is
% given by ROBOT in the following pattern:
%
% ROBOT is a matrix such that 
% ROBOT = [al_im1, a_im1, d_i, theta_i]
%
% Where
%
% col vector al_im1 contains the alpha-values, indexed from 0 (the
%    notation 'im1' meaning 'i minus 1') 
%
% col vector a_im1 contains the a_values, indexed from 0
%
% col vector d_i contains the d values, indexed from 1
%
% col vector theta_i contains the theta values, indexed from 1
%
%
% NB: the Denavit-Hartenberg notation is done according to
% J.J. Craig
%
% The matrix POSx contains the position of the mass center for the
% PowerCubes, such that the position for the cube controlling joint
% i is given in frame {i}.
%
% The i:th column of POSx is the position of the i:th cube, given in
% homogenous coordinates,
%
% POS1 is the position of the part of the cube fastened to previous
% link, POS2 is the position of the part fastened to the next.
%
% Setting a position's x-value to be imaginary will result in that
% particular cube not being drawn. Useful when using wrists or
% grippers that consist of a single cube. 

function result = plot_robot3(robot, pos1, pos2)

global cube_size

clean_slate = 0;
draw_links=1;
draw_principal_axis=0;

joints =[0 0 0 1;...
	 0 0 0 1;...
	 0 0 0 1;...
	 0 0 0 1;...
	 0 0 0 1;...
	 0 0 0 1;]';

colors = ['rgbcmyk'];
colormap([[0.5 0.5 1]; [1 1 0.5]*0.80; [1 1 1]*0.9]);

if(clean_slate)
  clf;
end
% set up limits for drawing 
plot3([-1 -1 1 -1]*0.8+0.2,[-1 -1 -1 1]*0.7,[1 -0.4 -0.4 -0.4],'.w')
hold on
axis equal    
oldp = pos1(:,1);

% generate values used to draw unit cube centered over origin:
x=[-1 -1 -1 -1 -1  -1 -1 -1 -1 -1   1 1 1 1 1    1 1 1 1 1]/2;
y=[0 0 0 0 0       -1 1 1 -1 -1    -1 1 1 -1 -1  0 0 0 0 0]/2;
z=[0 0 0 0 0       -1 -1 1 1 -1    -1 -1 1 1 -1  0 0 0 0 0]/2;
ett=ones(size(z));

% Parameters used to draw the cylindrical link segments:
if(draw_links)
  %colormap([1 1 0.5]*0.75)
  link_res = 10;
  t_link=linspace(0,2*pi,link_res);
  
  rotx=[0, 0, 0, 0, 0, 0];
  roty=[-pi/2, 0, 0, -pi/2, 0, -pi/2];
  rotz=[0, 0, pi/2, 0, 0, 0];
  
  link_offX=[pos2(3,1)+cube_size(1)/2, cube_size(2)/2, cube_size(3)/2,...
	    pos2(3,4)+cube_size(4)/2 0 cube_size(5)/2];
  link_offY=[0 0 0 0 0 0];
  link_offZ=[0 -cube_size(2) 0 0 0 0];
  link_length = [-(sum(cube_size([1 2])))/2-pos2(3,1),...
		 robot(3,2)- sum(cube_size([2 3]))/2,...
		 pos1(2,4) - sum(cube_size([3 4]))/2,...
		 -pos2(3,4) - sum(cube_size([4 5]))/2,...
		 0,...
		 pos2(3,6) - sum(cube_size([5 6]))/2];
  
  link_radius = cube_size/2 - 0.01;
  link_radius = [link_radius link_radius(end)];
  
  for k=1:6
  Rx = [1              0             0;...
        0   cos(rotx(k)) -sin(rotx(k));...
        0   sin(rotx(k))  cos(rotx(k))];
    
  Ry = [ cos(roty(k)) 0 sin(roty(k));...
	 0            1            0;...
	-sin(roty(k)) 0 cos(roty(k))];
    
  Rz = [cos(rotz(k)) -sin(rotz(k))  0;...
        sin(rotz(k))  cos(rotz(k))  0;...
        0             0             1];

  R=eye(3);
  R=Rz*Ry*Rx*R;
  link_rot{k}=[[R; 0 0 0] [0 0 0 1]'];  
  end
  
end


% Draw all 6 cubes. For robots not using 6 cubes, other function
% might bee needed.
T = eye(4);
for k = 1:6
  
  % Draw cube fastened to previous link:
  if(isreal(pos1(1,k)))
    xk=pos1(1,k)+x*cube_size(k);
    yk=pos1(2,k)+y*cube_size(k);
    zk=pos1(3,k)+z*cube_size(k);
    cube=T*[xk;yk;zk;ett];
    
    xk = reshape(cube(1,:),5,4)';
    yk = reshape(cube(2,:),5,4)';
    zk = reshape(cube(3,:),5,4)';
    surf(xk,yk,zk,zk*0+1)
  end
 
  % Generate transformation matrix for this link:
  T = T*generate_T(k-1,k,robot);
  
   % draw link:
  if(draw_links)
    X_link=[zeros(1,link_res)+link_offX(k),...
	    ones(1,link_res)*link_length(k)+link_offX(k)];  
    Y_link=[cos(t_link)*link_radius(k),...
	    cos(t_link)*(link_radius(k+1))]+link_offY(k);
    Z_link=[sin(t_link)*link_radius(k),...
	    sin(t_link)*link_radius(k+1)]+link_offZ(k);
    
    pos_links=T*link_rot{k}*[X_link;...
		    Y_link;...
		    Z_link;...
		    ones(1,link_res*2)];
    
    Xl = reshape(pos_links(1,:),link_res,2)';
    Yl = reshape(pos_links(2,:),link_res,2)';
    Zl = reshape(pos_links(3,:),link_res,2)';

    surf(Xl,Yl,Zl,Zl*0+2);
  end
  
  
  % Draw cube fastened to this link:
  if(isreal(pos2(1,k)))
    xk=pos2(1,k)+x*cube_size(k);
    yk=pos2(2,k)+y*cube_size(k);
    zk=pos2(3,k)+z*cube_size(k);
    cube=T*[xk;yk;zk;ett];
    
    xk = reshape(cube(1,:),5,4)';
    yk = reshape(cube(2,:),5,4)';
    zk = reshape(cube(3,:),5,4)';
  
    if(k~=6)
      surf(xk,yk,zk,zk*0+1)
    else
      surf(xk,yk,zk,zk*0+3)
    end
    
  end

  % draw joints and links
  if(draw_principal_axis)
    newp=T*joints(:,k);
    plot3(newp(1),newp(2),newp(3),['*' colors(k)]);
    plot3([oldp(1) newp(1)], [oldp(2) newp(2)], [oldp(3) newp(3)],colors(k));
    oldp=newp;
  end
  
    
end

result = 1;