//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// this is a -*- C++ -*- file
//----------------------------------------------------------------------
/*!\file    Dynamics.C
 *
 * \author   Christian Smith
 * \date    2005.05.02
 * \update  2005.05.23
 * $Revision: 1.5 $
 * $Id: NullModule.C,v 1.5 2003/02/24 14:36:21 gassmann Exp $
 *
 */
//----------------------------------------------------------------------
// Includes
//----------------------------------------------------------------------

#include "Dynamics.h"
#include "vectorMath.h"

//----------------------------------------------------------------------
// Module Debug
//----------------------------------------------------------------------
//#undef MODULE_DEBUG
#define MODULE_DEBUG
//#define TIMING_TEST
#define DYNAMICS_LOGFILE "logs/dynamics.log"
#define KINEMATICS_LOGFILE "logs/kinematics.log"
//#define LOG_INVERSE_DYNAMICS
//#define LOG_FORCES
//#define LOG_CONTROL_SEQUENCE
#include <module_debug.h>

//----------------------------------------------------------------------
// defines and consts
//----------------------------------------------------------------------

//----------------------------------------------------------------------
// class Dynamics constructor
//----------------------------------------------------------------------
mDynamics::mDynamics(tParent *parent,
		     tDescription name,bool fixit)
  :tModule(parent,name,
	   eSI_DIMENSION,eSO_DIMENSION,eCI_DIMENSION,eCO_DIMENSION,
	   si_description,so_description,ci_description,co_description,
	   fixit)
{
  InitParameter(ePAR_DIMENSION,
		new tParameterFloat("max torque waist",0, 500, 134),
		new tParameterFloat("max torque shld",0, 500, 267),
		new tParameterFloat("max torque elb",0, 500, 134),
		new tParameterFloat("max torque wr rot",0, 500, 23),
		new tParameterFloat("max torque wr bnd",0, 100, 35),
		new tParameterFloat("max torque flange",0, 50, 8)
		);
}

//----------------------------------------------------------------------
// class Dynamics destructor
//----------------------------------------------------------------------
mDynamics::~mDynamics()
{
}

//----------------------------------------------------------------------
// class Dynamics Control
//----------------------------------------------------------------------
void  mDynamics::Control()
{
  
  if (ControllerInputChanged() || DYN_sensor_changed)
    {
      VM_7vector t;
      if(ControllerInputChanged()){
	theta_pp.coord[0] = (double)0.0;
	theta_pp_ctrl.coord[1] = (double)(ControllerInput(eCI_JOINT_ACC_WAIST)); 
	theta_pp_ctrl.coord[2] = (double)(ControllerInput(eCI_JOINT_ACC_SHOULDER));
	theta_pp_ctrl.coord[3] = (double)(ControllerInput(eCI_JOINT_ACC_ELBOW)); 
	theta_pp_ctrl.coord[4] = (double)(ControllerInput(eCI_JOINT_ACC_WRIST_ROT));
	theta_pp_ctrl.coord[5] = (double)(ControllerInput(eCI_JOINT_ACC_WRIST_BEND));
	theta_pp_ctrl.coord[6] = (double)(ControllerInput(eCI_JOINT_ACC_FLANGE));
      }

      /*
      //debug************************
      theta_ctrl.coord[0] = (double)0.0;
      theta_ctrl.coord[1] = (double)0.0;
      theta_ctrl.coord[2] = (double)0.0;
      theta_ctrl.coord[3] = (double)0.0;
      theta_ctrl.coord[4] = (double)0.0;
      theta_ctrl.coord[5] = (double)0.0;
      theta_ctrl.coord[6] = (double)0.0;
      
      theta_p_ctrl.coord[0] = (double)0.0;
      theta_p_ctrl.coord[1] = (double)1.0;
      theta_p_ctrl.coord[2] = (double)1.0;
      theta_p_ctrl.coord[3] = (double)1.0;
      theta_p_ctrl.coord[4] = (double)1.0;
      theta_p_ctrl.coord[5] = (double)1.0;
      theta_p_ctrl.coord[6] = (double)1.0;
      
      theta_pp_ctrl.coord[0] = (double)0.0;
      theta_pp_ctrl.coord[1] = (double)2.0;
      theta_pp_ctrl.coord[2] = (double)4.0;
      theta_pp_ctrl.coord[3] = (double)2.0;
      theta_pp_ctrl.coord[4] = (double)2.0;
      theta_pp_ctrl.coord[5] = (double)2.0;
      theta_pp_ctrl.coord[6] = (double)2.0;
    // end debug***************************
    */


#ifdef LOG_CONTROL_SEQUENCE     
      FILE *control_logfile = fopen("logs/control.log","a");

      fprintf(control_logfile," control acc in:\n ");
      for(int i=1;i<7;i++){
	fprintf(control_logfile," %d:%f ",i,theta_pp_ctrl.coord[i]);
      }
      fprintf(control_logfile,"\n\n");


      fprintf(control_logfile," control vel in:\n ");
      for(int i=1;i<7;i++){
	fprintf(control_logfile," %d:%f ",i,theta_p.coord[i]);
      }
      fprintf(control_logfile,"\n\n");

      fprintf(control_logfile," control ang in:\n ");
      for(int i=1;i<7;i++){
	fprintf(control_logfile," %d:%f ",i,theta.coord[i]);
      }
      fprintf(control_logfile,"\n\n");
      
      fprintf(control_logfile,"\n Calling torque calc:\n");
#endif //#ifdef LOG_CONTROL_SEQUENCE     

      VM_3vector grav;
      for(int i = 0;i<3; i++){
	grav.coord[i]= ((i==2)*(PA_PH_G/1000));
      }
      
      t = InverseDynamics(theta, theta_p, theta_pp_ctrl, grav, 0);      
     
#ifdef LOG_CONTROL_SEQUENCE     
      fprintf(control_logfile," control torques before limit:\n ");
      for(int i=1;i<7;i++){
	fprintf(control_logfile," %d:%f ",i,t.coord[i]);
      }
      fprintf(control_logfile,"\n\n");
#endif //#ifdef LOG_CONTROL_SEQUENCE     

      // Limit Torques to possible values
      for(int i=0; i<6; i++){
	t.coord[i+1] = MAX(t.coord[i+1],-max_torque[i]);
	t.coord[i+1] = MIN(t.coord[i+1],max_torque[i]);
      }

#ifdef LOG_CONTROL_SEQUENCE     
      fprintf(control_logfile," control torques after limit:\n ");
      for(int i=1;i<7;i++){
	fprintf(control_logfile," %d:%f ",i,t.coord[i]);
      }
      fprintf(control_logfile,"\n\n");
#endif //#ifdef LOG_CONTROL_SEQUENCE     


      theta_pp_ctrl = ForwardDynamics(theta, theta_p, t);

#ifdef LOG_CONTROL_SEQUENCE     
      fprintf(control_logfile," control acc out:\n ");
      for(int i=1;i<7;i++){
	fprintf(control_logfile," %d:%f ",i,theta_pp_ctrl.coord[i]);
      }
      fprintf(control_logfile,"\n\n");
      
      fflush(control_logfile);
      fclose(control_logfile);
#endif //#ifdef LOG_CONTROL_SEQUENCE     

      for (int i=0;i<ControllerInputDimension();i++){
	ControllerOutput(i)=theta_pp_ctrl.coord[i+1];
      }
      SetControllerOutputChanged();
      DYN_sensor_changed = 0;
      ClearControllerInput();
    }
  else
    ClearControllerOutput();
}

//----------------------------------------------------------------------
// class Dynamics Sense
//----------------------------------------------------------------------
void  mDynamics::Sense()
{
  // output variable:
  VM_7vector t;
  static int sense_first=1;

  if (SensorInputChanged() || sense_first){

    sense_first = 0;
    DYN_sensor_changed = 1;

#ifdef TIMING_TEST
    D_sense_timer.FromNow();
#endif

    // For simplifying access to input values:
    theta.coord[0] = (double)0.0;
    theta.coord[1] = (double)(SensorInput(eSI_WAIST));       
    theta.coord[2] = (double)(SensorInput(eSI_SHOULDER));
    theta.coord[3] = (double)(SensorInput(eSI_ELBOW));   
    theta.coord[4] = (double)(SensorInput(eSI_WRIST_ROT));  
    theta.coord[5] = (double)(SensorInput(eSI_WRIST_BEND)); 
    theta.coord[6] = (double)(SensorInput(eSI_FLANGE));     
    
    theta_p.coord[0] = (double)0.0;
    theta_p.coord[1] = (double)(SensorInput(eSI_WAIST_VEL)); 
    theta_p.coord[2] = (double)(SensorInput(eSI_SHOULDER_VEL));
    theta_p.coord[3] = (double)(SensorInput(eSI_ELBOW_VEL)); 
    theta_p.coord[4] = (double)(SensorInput(eSI_WRIST_ROT_VEL));
    theta_p.coord[5] = (double)(SensorInput(eSI_WRIST_BEND_VEL));
    theta_p.coord[6] = (double)(SensorInput(eSI_FLANGE_VEL));
    
    theta_pp.coord[0] = (double)0.0;
    theta_pp.coord[1] = (double)(SensorInput(eSI_WAIST_ACC)); 
    theta_pp.coord[2] = (double)(SensorInput(eSI_SHOULDER_ACC));
    theta_pp.coord[3] = (double)(SensorInput(eSI_ELBOW_ACC)); 
    theta_pp.coord[4] = (double)(SensorInput(eSI_WRIST_ROT_ACC));
    theta_pp.coord[5] = (double)(SensorInput(eSI_WRIST_BEND_ACC));
    theta_pp.coord[6] = (double)(SensorInput(eSI_FLANGE_ACC));

    // defining gravity (in m/s).
    VM_3vector gravitation;
    for(int i = 0;i<3; i++){
      gravitation.coord[i]= ((i==2)*(PA_PH_G/1000));
    }


    t = InverseDynamics(theta, theta_p, theta_pp, gravitation, -1);
        
    for (int i=0;i<7;i++)
      SensorOutput(i)=t.coord[i+1];
    // Es wurde was gendert, also
    SetSensorOutputChanged();
    // Eingang wurde damit abgearbeitet, also nderung zurcksetzen:
    ClearSensorInput();

#ifdef TIMING_TEST
    double D_elapsed_time =(tTime::Now()-D_sense_timer).ToUSec();
    fprintf(stdout,"Dynamics calc time (usec)= %f\n",D_elapsed_time);
#endif


#ifdef KINEMATICS_LOGFILE
  if(kin_logfile_opened){
    for(int i=1;i<7;i++){
      fprintf(kin_logfile,"%f ",theta.coord[i]);
      fprintf(kin_logfile,"%f ",theta_p.coord[i]);
      fprintf(kin_logfile,"%f ",theta_pp.coord[i]);
    }
    fprintf(kin_logfile,"\n");
  }
#endif

#ifdef DYNAMICS_LOGFILE
  if(dyn_logfile_opened){
    for(int i=1;i<7;i++){
      fprintf(dyn_logfile,"%f ",t.coord[i]);
      /*
      for(int j=0;j<3;j++){
	fprintf(dyn_logfile,"%f ",0.0);
      }
      for(int j=0;j<3;j++){
	fprintf(dyn_logfile,"%f ",0.0);
	}*/
    }
    fprintf(dyn_logfile,"\n");
  }
#endif

  }
  else{
    ClearSensorOutput();
  }
}

//----------------------------------------------------------------------
// class Dynamics Exception
//----------------------------------------------------------------------
void mDynamics::Exception(tExceptionType type)
{
  switch (type)
    {
    case eET_ParameterChanged:
      {
	max_torque[0]=ParameterFloat(ePAR_MAX_TORQUE_WAIST); 
	max_torque[1]=ParameterFloat(ePAR_MAX_TORQUE_SHOULDER);
	max_torque[2]=ParameterFloat(ePAR_MAX_TORQUE_ELBOW);
	max_torque[3]=ParameterFloat(ePAR_MAX_TORQUE_WRIST_ROT); 
	max_torque[4]=ParameterFloat(ePAR_MAX_TORQUE_WRIST_BEND);
	max_torque[5]=ParameterFloat(ePAR_MAX_TORQUE_FLANGE); 
	ClearParameter();
      }
      break;
    case eET_PrepareForFirstSense:
      DYN_sensor_changed = 1;
      InitializePositions();
#ifdef DYNAMICS_LOGFILE
      if ((dyn_logfile = fopen(DYNAMICS_LOGFILE, "w")) == NULL){
	fprintf(stderr, "Cannot open %s\n", DYNAMICS_LOGFILE);
	dyn_logfile_opened = 0;
      }
      else{
	dyn_logfile_opened = 1;
      }
#endif
#ifdef KINEMATICS_LOGFILE
      if ((kin_logfile = fopen(KINEMATICS_LOGFILE, "w")) == NULL){
	fprintf(stderr, "Cannot open %s\n", KINEMATICS_LOGFILE);
	kin_logfile_opened = 0;
      }
      else{
	kin_logfile_opened = 1;
      }
#endif
      break;
    case eET_PrepareForFirstControl:
      break;
    case eET_PrepareForBreak:
#ifdef DYNAMICS_LOGFILE
      if(dyn_logfile_opened){
	fclose(dyn_logfile);
	dyn_logfile_opened = 0;
      }
#endif
#ifdef KINEMATICS_LOGFILE
      if(kin_logfile_opened){
	fclose(kin_logfile);
	kin_logfile_opened = 0;
      }
#endif
      break;
    case eET_PrepareForRestart:
      break;
    }
}
// Internal functions
VM_3matrix mDynamics::RotationMatrix(int from, int to){
  // This generates a rotation matrix in accordance to
  // J.J. Craig sect 3.5. NB: the standard case here is
  // the inverse of Craig's matrix!
  if(from>6 || to > 6){
    fprintf(stdout,"ERROR!! cannot generate rotation matrix from frame %d to %d!\n",from,to);
  }
  VM_3matrix _R;
  if((to-from)==1){
    static int _iscalculated[] = {0, 0, 0, 0, 0, 0};
    static double _c_alpha[6];
    static double _s_alpha[6];
    if(!_iscalculated[from]){
    fprintf(stdout,"Rotate using angles = [%f %f %f %f %f %f].\n",PA_dh_alpha[0],PA_dh_alpha[1],PA_dh_alpha[2],PA_dh_alpha[3],PA_dh_alpha[4],PA_dh_alpha[5]);
      _c_alpha[from]=cos(PA_dh_alpha[from]);
      _s_alpha[from]=sin(PA_dh_alpha[from]);
      _iscalculated[from]=1; 
      fprintf(stdout,"c_alpha[%d] = %f, s_alpha[%d]= %f\n",from,_c_alpha[from],from,_s_alpha[from]);
    }
    double _ct = cos(theta.coord[to]);
    double _st = sin(theta.coord[to]);
    _R.coord[0][0] = _ct;
    _R.coord[0][1] = _st*_c_alpha[from];
    _R.coord[0][2] = _st*_s_alpha[from];
    
    _R.coord[1][0] = -_st;
    _R.coord[1][1] = _ct*_c_alpha[from];
    _R.coord[1][2] = _ct*_s_alpha[from];
    
    _R.coord[2][0] = 0;
    _R.coord[2][1] = -_s_alpha[from];
    _R.coord[2][2] = _c_alpha[from];
  }
  else{
    // for the less simple case, use more general function.
    _R = RotationMatrix2(from,to);
  }
  return _R;
}

// This function is slower but more general.
VM_3matrix mDynamics::RotationMatrix2(int from, int to){
  VM_3matrix _R;
  int _inv = (to<from);
  
  // generate identity matrix
  for(int i=0;i<3;i++){
    for(int j=0;j<3;j++){
      _R.coord[i][j]=(i==j);
    }
  }
  
  // Multiply together all the rotations:
  for(int i = MIN(from,to); i<MAX(from,to); i++){
    _R = VM_mul(RotationMatrix(i,i+1), _R);
  }
  
  // Invert if going from larger to smaller
  // inverse of rotation matrix is equal to transpose
  if(_inv){
    _R = VM_trans(_R);
  }
  
  return _R;
}


VM_3vector mDynamics::TranslationVector(int from, int to){
  // This generates a translation vector in accordance to
  // J.J. Craig sect 3.5. 
  VM_3vector _P;
  if(from>6 || to > 6){
    fprintf(stdout,"ERROR!! cannot generate translation vector from frame %d to %d!\n",from,to);
  }
  if((from-to)==1){  
    static int _iscalculated[] = {0, 0, 0, 0, 0, 0};
    static double _c_alpha[6];
    static double _s_alpha[6];
    if(!_iscalculated[to]){
      _c_alpha[to]=cos(PA_dh_alpha[to]);
      _s_alpha[to]=sin(PA_dh_alpha[to]);
      _iscalculated[to]=1; 
    }
    _P.coord[0] = PA_dh_a[to];
    _P.coord[1] = -_s_alpha[to]*PA_dh_d[from];
    _P.coord[2] = _c_alpha[to]*PA_dh_d[from];
  }
  else{
    // use more general function
    _P = TranslationVector2(from, to);
  }

  return _P;
}

VM_3vector mDynamics::TranslationVector2(int from, int to){
  // This generates a translation vector in accordance to
  // J.J. Craig sect 3.5.
  // This is a more general function.
  VM_3vector _P;
  VM_4matrix _T;
  
  // generate identity matrix
  for(int i=0;i<4;i++){
    for(int j=0;j<4;j++){
      _T.coord[i][j]=(i==j);
    }
  }
  
  int _inv = (to>from);
  
  // Generate series of multiplications of _T
  for(int i=MAX(to,from); i>MIN(to,from); i--){
    VM_4matrix _Ttemp;
    VM_3matrix _R = RotationMatrix(i,i-1);
    VM_3vector _Ptemp = TranslationVector(i,i-1);
    
    // Generate transformation matrix Ttemp
    for(int i=0;i<3;i++){
      for(int j=0;j<3;j++){
	_Ttemp.coord[i][j]=_R.coord[i][j];
      }
      _Ttemp.coord[i][4] = _Ptemp.coord[i];
    }
    for(int j=0;j<4;j++){
      _Ttemp.coord[4][j] = 0;
    }
    
    // Multiply with last matrix
    _T = VM_mul(_Ttemp,_T);
  }
  
  for(int i = 0; i<3; i++){
    _P.coord[i] = _T.coord[i][4];
  }
  
  // Invert if necessary
  if(_inv){
    VM_3matrix _R = RotationMatrix(from,to);
    _P = VM_mul(-1.0,(VM_mul(VM_trans(_R),_P)));
  }
  return _P;
}

// Calculates the center of mass for link 'frame'
VM_3vector mDynamics::MassCenter(int frame){
  // this doesn't change during execution, 
  // and only needs calculating once
  static int _iscalculated[] = {0, 0, 0, 0, 0, 0, 0};
  static VM_3vector _mc[7];
  VM_3vector _temp1;
  VM_3vector _temp2;
  if (!_iscalculated[frame]){
    _temp2 = VM_mul(PA_rb_mass_a[frame],PA_rb_pos1[frame+1]);
    _temp1 = VM_mul(PA_rb_mass_b[frame],PA_rb_pos2[frame]);
    _mc[frame] = VM_mul((1/(Mass(frame))),VM_add(_temp1,_temp2));
    _iscalculated[frame]=1;

  }
  return _mc[frame];
}

double mDynamics::Mass(int frame){
  // this doesn't change during execution, 
  // and only needs calculating once
  static int _iscalculated[] = {0, 0, 0, 0, 0, 0, 0};
  static double _m[7];
  if (!_iscalculated[frame]){
    _m[frame] =  PA_rb_mass_a[frame] + PA_rb_mass_b[frame];
    _iscalculated[frame]=1;
  }
  return _m[frame];
}

VM_3matrix mDynamics::InertiaTensor(int frame){
  // this doesn't change during execution, 
  // and only needs calculating once
  static int _iscalculated[] = {0, 0, 0, 0, 0, 0, 0};
  static VM_3matrix _I[7];
  if (!_iscalculated[frame]){
    
    // 3*3 identity matrix:
    VM_3matrix _Ident3;
    for(int i=0;i<3;i++){
      for(int j=0;j<3;j++){
	_Ident3.coord[i][j]=(i==j);
      }
    }
    
    double _icube1;
    double _icube2;
    VM_3vector _p_cube1;
    VM_3vector _p_cube2;
    
    VM_3matrix _TEMP1;
    VM_3matrix _TEMP1a;
    VM_3matrix _TEMP1b;
    
    VM_3matrix _TEMP2;    
    VM_3matrix _TEMP2a;
    VM_3matrix _TEMP2b;
    
    VM_3matrix _TEMP3;

    // There is no cube 1 in the last frame...
    if(frame<6){
      _icube1 = (PA_rb_mass_a[frame+1]/12)*(2*(PA_rb_size[frame+1])*(PA_rb_size[frame+1]));
    }else{
      _icube1 = 0;
    }
    _icube2 = (PA_rb_mass_b[frame]/12)*(2*(PA_rb_size[frame])*(PA_rb_size[frame]));
    
    _p_cube1 = VM_sub(PA_rb_pos1[frame+1],MassCenter(frame));
    _p_cube2 = VM_sub(PA_rb_pos2[frame],MassCenter(frame));
    
    _TEMP1a = VM_mul(VM_dot( _p_cube1 , _p_cube1), _Ident3);
    _TEMP1b = VM_mul(-1.0, VM_matrixProd( _p_cube1 , _p_cube1));
    _TEMP1 = VM_mul(PA_rb_mass_b[frame+1],VM_add(_TEMP1a,_TEMP1b));

    _TEMP2a = VM_mul(VM_dot( _p_cube2 , _p_cube2), _Ident3);
    _TEMP2b = VM_matrixProd( _p_cube2 , _p_cube2);
    _TEMP2 = VM_mul(PA_rb_mass_a[frame],VM_sub(_TEMP2a,_TEMP2b));
    
    _TEMP3 = VM_mul((_icube1+_icube2),_Ident3);
    
    _I[frame] = VM_add(_TEMP1,VM_add(_TEMP2,_TEMP3));

    _iscalculated[frame]=1;

    /*
    fprintf(stdout,"Initial:\n I[%d] = [%f %f %f]\n [%f %f %f]\n [%f %f %f]\n",frame,
	    _I[frame].coord[0][0],_I[frame].coord[0][1],_I[frame].coord[0][2],
	    _I[frame].coord[1][0],_I[frame].coord[1][1],_I[frame].coord[1][2],
	    _I[frame].coord[2][0],_I[frame].coord[2][1],_I[frame].coord[2][2]);
    */
  }
  /*
  fprintf(stdout,"I[%d] = [%f %f %f]\n [%f %f %f]\n [%f %f %f]\n",frame,
	  _I[frame].coord[0][0],_I[frame].coord[0][1],_I[frame].coord[0][2],
	  _I[frame].coord[1][0],_I[frame].coord[1][1],_I[frame].coord[1][2],
	  _I[frame].coord[2][0],_I[frame].coord[2][1],_I[frame].coord[2][2]);
  */
  return _I[frame];
}

// Calculates inverse dynamics (the torques necessary for a guven position, velocity and acceleration). _position is used by the forward dynamics function. Only the effects from link _position and outwards will be calculated.
VM_7vector mDynamics::InverseDynamics(VM_7vector _theta, VM_7vector _theta_p, VM_7vector _theta_pp, VM_3vector _grav, int _position){
  VM_7vector _t;


  VM_3vector zero_vec;

  // initializing the values for link 0
  for(int i=0;i<3;i++){
    zero_vec.coord[i]=0.0;
    w[0].coord[i]=0.0;
    wp[0].coord[i]=0.0;
    vp[0].coord[i]=_grav.coord[i];
  }
  
  // defining the unit Z-vector
  VM_3vector Z;
  Z.coord[0]=0.0;
  Z.coord[1]=0.0;
  Z.coord[2]=1.0;
  
  // The rotations and translations from one frame to the next
  // R[i] is the rotation that transform coordinates in frame
  // i to frame i+1.
  // P[i] is the translation that transforms coordinates in frame
  // i+1 to frame i.
  VM_3matrix R[7];
  static VM_3vector P[7];
  static int _is_calculated_P[]={0,0,0,0,0,0,0};

  // Pc[i] is the location of the center of mass in link i
  VM_3vector Pc[7];
  
  // Moment of inertia tensor for link i, around center of mass.
  VM_3matrix I[7];
  
  // Some temp variables used to store values for clarity
  // in complex expressions:
  VM_3vector temp1;
  VM_3vector temp2;
  VM_3vector temp3;
  VM_3vector temp4;
  
  // First we calculate the accelerations forces and torques per link
  // going outwards (see section 6.5 (p200) in J.J. Craig)
  
  int stationary = 1;
  
  for(int i=0;i<6;i++){
    
    //fprintf(stdout,"i=%d ",i);
    
    R[i] = RotationMatrix(i,i+1);
    
    /* debug messages 
       fprintf(stdout,"Theta.Coord[%d] = %f \n",i+1,_theta.coord[i+1]);
       fprintf(stdout,"R[%d] = [%f %f %f]\n [%f %f %f]\n [%f %f %f]\n",i,
       R[i].coord[0][0],R[i].coord[0][1],R[i].coord[0][2],
       R[i].coord[1][0],R[i].coord[1][1],R[i].coord[1][2],
       R[i].coord[2][0],R[i].coord[2][1],R[i].coord[2][2]);
    */
    
    stationary = (((i+1)<_position) && stationary && _theta_p.coord[i+1]==0 && _theta_pp.coord[i+1]==0 && _grav.coord[2]==0 && _grav.coord[1]==0 && _grav.coord[0]==0); 
   
    Pc[i+1] = MassCenter(i+1);
    if(!_is_calculated_P[i]){
      P[i] = TranslationVector(i+1,i);
      _is_calculated_P[i] = 1;
    }

    // some things need only be calculated if we know that we are 
    // moving
    if(!stationary){
      w[i+1] = VM_add(VM_mul(R[i],w[i]),VM_mul(_theta_p.coord[i+1],Z));
    
      temp1 = VM_mul(R[i],wp[i]);
      temp2 = VM_cross(VM_mul(R[i],w[i]),VM_mul(_theta_p.coord[i+1],Z));
      temp3 = VM_mul(_theta_pp.coord[i+1],Z);
      
      wp[i+1] = VM_add(VM_add(temp1,temp2),temp3);
      
      temp1 = VM_cross(wp[i],P[i]);
      temp2 = VM_cross(w[i],VM_cross(w[i],P[i]));
      temp3 = vp[i];
      temp4 = VM_add(temp1,VM_add(temp2,temp3));
      vp[i+1] = VM_mul(R[i],temp4);
      
      temp1 = VM_cross(wp[i+1],Pc[i+1]);
      temp2 = VM_cross(w[i+1],VM_cross(w[i+1],Pc[i+1]));
      temp3 = vp[i+1];
      vcp[i+1] = VM_add(temp1,VM_add(temp2,temp3));
      
      m[i+1] = Mass(i+1);
      F[i+1] = VM_mul(m[i+1],vcp[i+1]);
      
      I[i+1] = InertiaTensor(i+1);
      
      temp1 = VM_mul(I[i+1],wp[i+1]);
      temp2 = VM_cross(w[i+1], VM_mul(I[i+1],w[i+1]));
      N[i+1] = VM_add(temp1,temp2);
    }
    else{
      for(int j=0;j<3;j++){
	w[i+1].coord[j]=wp[i+1].coord[j]=vp[i+1].coord[j]=F[i+1].coord[j]=N[i+1].coord[j]=0;
      }
    }  
  }

#ifdef LOG_INVERSE_DYNAMICS
  //DEBUG**********
  FILE *inv_dyn_logfile = fopen("logs/inverse_dynamics.log","a");    
    if(_position==-1 && 1){ 

    for(int kk=1; kk<7; kk++){
      fprintf(inv_dyn_logfile,"Mass[%d] = %f   (_pos=%d)\n",kk,m[kk],_position);
    }

    for(int kk=1; kk<7; kk++){
      fprintf(inv_dyn_logfile,"F[%d] = %f %f %f   (_pos=%d)\n",kk,F[kk].coord[0],F[kk].coord[1],F[kk].coord[2],_position);
    }
    
    for(int kk=1; kk<7; kk++){
      fprintf(inv_dyn_logfile,"N[%d] = %f %f %f   (_pos=%d)\n",kk,N[kk].coord[0],N[kk].coord[1],N[kk].coord[2],_position);
    }
    
    for(int kk=1; kk<7; kk++){
      fprintf(inv_dyn_logfile,"vp[%d] = %f %f %f   (_pos=%d)\n",kk,vp[kk].coord[0],vp[kk].coord[1],vp[kk].coord[2],_position);
    }
    
    for(int kk=1; kk<7; kk++){
      fprintf(inv_dyn_logfile,"wp[%d] = %f %f %f   (_pos=%d)\n",kk,wp[kk].coord[0],wp[kk].coord[1],wp[kk].coord[2],_position);
    }
  }
#endif //#ifdef LOG_INVERSE_DYNAMICS


  //END DEBUG**********

  // Next, we work our way inwards, adding upp all the forces
  // and torques from the outward links.
  // (see section 6.5 (p200) in J.J. Craig)
  // The first step is given special treatment, for speed.
  f[6] = F[6];
  n[6] = VM_add(N[6],VM_cross(Pc[6],F[6]));
  _t.coord[6] = n[6].coord[2];
  //fprintf(stdout,"Torque %d = %f",6,t[6]);
  
  for(int i=5; i>0; i--){
    if(i<_position){
      _t.coord[i] = 0;
    }
    else{
      VM_3matrix Rinv = VM_trans(R[i]);
      f[i] = VM_add(VM_mul(Rinv,f[i+1]),F[i]);
      
      temp1 = N[i];
      temp2 = VM_mul(Rinv,n[i+1]);
      temp3 = VM_cross(Pc[i],F[i]);
      temp4 = VM_cross(P[i],VM_mul(Rinv,f[i+1]));
      n[i] = VM_add(VM_add(temp1,temp2),VM_add(temp3,temp4));
      
      _t.coord[i] = n[i].coord[2];
    }
  }   

#ifdef LOG_FORCES
  FILE *forces_logfile = fopen("logs/forces.log","a");    
    
  for(int kk=1; kk<7; kk++){
    fprintf(forces_logfile," %f %f %f",f[kk].coord[0],f[kk].coord[1],f[kk].coord[2]);
  }
  fprintf(forces_logfile,"\n");
  fclose(forces_logfile);
#endif //#ifdef LOG_FORCES

  //DEBUG**********
#ifdef LOG_INVERSE_DYNAMICS
  if(_position==-1){ 
    for(int kk=1; kk<7; kk++){
      fprintf(inv_dyn_logfile,"Pc[%d] = %f %f %f   (_pos=%d)\n",kk,Pc[kk].coord[0],Pc[kk].coord[1],Pc[kk].coord[2],_position);
    }

    for(int kk=1; kk<7; kk++){
      fprintf(inv_dyn_logfile,"P[%d] = %f %f %f   (_pos=%d)\n",kk,P[kk].coord[0],P[kk].coord[1],P[kk].coord[2],_position);
    }

    for(int kk=1; kk<7; kk++){
      fprintf(inv_dyn_logfile,"f[%d] = %f %f %f   (_pos=%d)\n",kk,f[kk].coord[0],f[kk].coord[1],f[kk].coord[2],_position);
    }
    
    for(int kk=1; kk<7; kk++){
      fprintf(inv_dyn_logfile,"n[%d] = %f %f %f   (_pos=%d)\n",kk,n[kk].coord[0],n[kk].coord[1],n[kk].coord[2],_position);
      }
  }
  fclose(inv_dyn_logfile);
#endif //#ifdef LOG_INVERSE_DYNAMICS
  
  /*
#ifdef DYNAMICS_LOGFILE
  if(dyn_logfile_opened && _position == -1){
    for(int i=1;i<7;i++){
      fprintf(dyn_logfile,"%f ",_t.coord[i]);
      for(int j=0;j<3;j++){
	fprintf(dyn_logfile,"%f ",n[i].coord[j]);
      }
      for(int j=0;j<3;j++){
	fprintf(dyn_logfile,"%f ",f[i].coord[j]);
      }
    }
    fprintf(dyn_logfile,"\n");
  }
#endif
  */
  return _t;   
}


// This function uses 'Method 2' presented by Walker and Orin, "Efficient Dynamic Computer Simulation of Robotic Mechanisms" to calculate the forward dynamics.
VM_7vector mDynamics::ForwardDynamics(VM_7vector _theta, VM_7vector _theta_p, VM_7vector _torque7){

  VM_6matrix _M;
  VM_7vector _acc;
  
  VM_3vector _gravity;
  for(int i=0;i<3;i++){
    _gravity.coord[i] = 0;
  }

  VM_7vector _nominalAcc;
  VM_7vector _zeroVec7;
  for(int i=0;i<7; i++){
    _nominalAcc.coord[i] = 0;
    _zeroVec7.coord[i] = 0;
  }

  for(int i=0;i<6;i++){
    _nominalAcc.coord[i] = 0;
    _nominalAcc.coord[i+1] = 1;
    VM_7vector _inertialTorques = InverseDynamics(_theta, _zeroVec7, _nominalAcc, _gravity, i);
    for(int j=i; j<6; j++){
      _M.coord[j][i] = _M.coord[i][j] = _inertialTorques.coord[j+1];
      
    }
  }
  

  /*
  fprintf(stdout,"\n M:\n");
  for(int ii=0;ii<6;ii++){  
      fprintf(stdout,"[ ");
    for(int iii=0;iii<6;iii++){
      fprintf(stdout," %f",_M.coord[ii][iii]);
    }
    fprintf(stdout," ]\n");
    fflush(stdout);
  }
  */


  VM_7vector _tau7;
  _gravity.coord[2] = (PA_PH_G/1000);
  _tau7 = InverseDynamics(_theta, _theta_p, _zeroVec7, _gravity,0);
  
  //resize to fit calculations
  VM_6vector _tau6;
  VM_6vector _torque6;
  VM_6vector _acc6;
  for(int i=0; i<6; i++){
    _tau6.coord[i]=_tau7.coord[i+1];
    _torque6.coord[i] = _torque7.coord[i+1];
  }

  _acc6 = VM_mul(VM_inverseSym(_M),VM_sub(_torque6,_tau6));

  _acc.coord[0]=0;
  for(int i=1;i<7;i++){
    _acc.coord[i]=_acc6.coord[i-1];
  }
  
  
  return _acc;

}


void mDynamics::InitializePositions(void){
  fprintf(stdout,"\nINITIALIZING VALUES");

  static int _isinitialized = 0;
  if(!_isinitialized){
    _isinitialized = 1;

    theta.coord[0] = (double)0.0;
    theta.coord[1] = (double)0.0;
    theta.coord[2] = (double)0.0;
    theta.coord[3] = (double)0.0;
    theta.coord[4] = (double)0.0;
    theta.coord[5] = (double)0.0;
    theta.coord[6] = (double)0.0;
    fprintf(stdout,".");
    
    theta_p.coord[0] = (double)0.0;
    theta_p.coord[1] = (double)0.0;
    theta_p.coord[2] = (double)0.0;
    theta_p.coord[3] = (double)0.0;
    theta_p.coord[4] = (double)0.0;
    theta_p.coord[5] = (double)0.0;
    theta_p.coord[6] = (double)0.0;
    fprintf(stdout,".");    

    theta_pp.coord[0] = (double)0.0;
    theta_pp.coord[1] = (double)0.0;
    theta_pp.coord[2] = (double)0.0;
    theta_pp.coord[3] = (double)0.0;
    theta_pp.coord[4] = (double)0.0;
    theta_pp.coord[5] = (double)0.0;
    theta_pp.coord[6] = (double)0.0;
    fprintf(stdout,".");

    // indexed so that the index correlates to index of reference frame
    PA_rb_mass_a[0] = PA_RB_MASS_A1;
    PA_rb_mass_a[1] = PA_RB_MASS_A2;
    PA_rb_mass_a[2] = PA_RB_MASS_A3;
    PA_rb_mass_a[3] = PA_RB_MASS_A4;
    PA_rb_mass_a[4] = PA_RB_MASS_A5;
    PA_rb_mass_a[5] = PA_RB_MASS_A6;
    PA_rb_mass_a[6] = 0;
    fprintf(stdout,".");

    PA_rb_mass_b[0] = 0;
    PA_rb_mass_b[1] = PA_RB_MASS_B1;
    PA_rb_mass_b[2] = PA_RB_MASS_B2;
    PA_rb_mass_b[3] = PA_RB_MASS_B3;
    PA_rb_mass_b[4] = PA_RB_MASS_B4;
    PA_rb_mass_b[5] = PA_RB_MASS_B5;
    PA_rb_mass_b[6] = PA_RB_MASS_B6;
    fprintf(stdout,".");

    // We want values in meters for SI unit calculations
    PA_rb_size[0] = 0;
    PA_rb_size[1] = PA_RB_SIZE_1/1000;
    PA_rb_size[2] = PA_RB_SIZE_2/1000;
    PA_rb_size[3] = PA_RB_SIZE_3/1000;
    PA_rb_size[4] = PA_RB_SIZE_4/1000;
    PA_rb_size[5] = PA_RB_SIZE_5/1000;
    PA_rb_size[6] = PA_RB_SIZE_6/1000;
    fprintf(stdout,".");

    PA_dh_alpha[0] = PA_DH_ALPHA0;
    PA_dh_alpha[1] = PA_DH_ALPHA1;
    PA_dh_alpha[2] = PA_DH_ALPHA2;
    PA_dh_alpha[3] = PA_DH_ALPHA3;
    PA_dh_alpha[4] = PA_DH_ALPHA4;
    PA_dh_alpha[5] = PA_DH_ALPHA5;
    fprintf(stdout,".initializing angles = [%f %f %f %f %f %f].",PA_dh_alpha[0],PA_dh_alpha[1],PA_dh_alpha[2],PA_dh_alpha[3],PA_dh_alpha[4],PA_dh_alpha[5]);

    PA_dh_a[0] = 0/1000;
    PA_dh_a[1] = PA_DH_A1/1000;
    PA_dh_a[2] = PA_DH_A2/1000;
    PA_dh_a[3] = PA_DH_A3/1000;
    PA_dh_a[4] = PA_DH_A4/1000;
    PA_dh_a[5] = PA_DH_A5/1000;
    fprintf(stdout,".");

    PA_dh_d[0] = 0;
    PA_dh_d[1] = PA_DH_D1/1000;
    PA_dh_d[2] = PA_DH_D2/1000;
    PA_dh_d[3] = PA_DH_D3/1000;
    PA_dh_d[4] = PA_DH_D4/1000;
    PA_dh_d[5] = PA_DH_D5/1000;
    PA_dh_d[6] = PA_DH_D6/1000;
    fprintf(stdout,".");
    /* positions of (center of) cubes in frame that cube is stationary in [mm] */
    PA_rb_pos1[0].coord[0] = 0;
    PA_rb_pos1[0].coord[1] = 0;
    PA_rb_pos1[0].coord[2] = 0;
    
    PA_rb_pos1[1].coord[0] = 0;
    PA_rb_pos1[1].coord[1] = 0;
    PA_rb_pos1[1].coord[2] = ((-(3*PA_RB_SIZE_1 + PA_RB_SIZE_2)/2) - PA_RB_LENGTH_1)/1000;
    
    PA_rb_pos1[2].coord[0] = 0;
    PA_rb_pos1[2].coord[1] = 0;
    PA_rb_pos1[2].coord[2] = 0; 
    
    PA_rb_pos1[3].coord[0] = ((PA_RB_SIZE_2 + PA_RB_SIZE_3)/2 + PA_RB_LENGTH_2)/1000;
    PA_rb_pos1[3].coord[1] = 0;
    PA_rb_pos1[3].coord[2] = (-PA_RB_SIZE_2)/1000;
    
    PA_rb_pos1[4].coord[0] = 0;
    PA_rb_pos1[4].coord[1] = ((PA_RB_SIZE_3 + PA_RB_SIZE_4)/2 + PA_RB_LENGTH_3)/1000;
    PA_rb_pos1[4].coord[2] = 0;
    
    PA_rb_pos1[5].coord[0] = 0;
    PA_rb_pos1[5].coord[1] = 0;
    PA_rb_pos1[5].coord[2] = 0;
    
    PA_rb_pos1[6].coord[0] = 0;
    PA_rb_pos1[6].coord[1] = 0;
    PA_rb_pos1[6].coord[2] = 0;
    
    PA_rb_pos1[7].coord[0] = 0;
    PA_rb_pos1[7].coord[1] = 0;
    PA_rb_pos1[7].coord[2] = 0;
    
    PA_rb_pos2[0].coord[0] = 0;
    PA_rb_pos2[0].coord[1] = 0;
    PA_rb_pos2[0].coord[2] = 0;
    
    PA_rb_pos2[1].coord[0] = 0;
    PA_rb_pos2[1].coord[1] = 0;
    PA_rb_pos2[1].coord[2] = (-(PA_RB_SIZE_1 + PA_RB_SIZE_2)/2 - PA_RB_LENGTH_1)/1000;
    
    PA_rb_pos2[2].coord[0] = 0;
    PA_rb_pos2[2].coord[1] = 0;
    PA_rb_pos2[2].coord[2] = (-PA_RB_SIZE_2)/1000;
    
    PA_rb_pos2[3].coord[0] = 0;
    PA_rb_pos2[3].coord[1] = 0;
    PA_rb_pos2[3].coord[2] = 0;
    
    PA_rb_pos2[4].coord[0] = 0;
    PA_rb_pos2[4].coord[1] = 0;
    PA_rb_pos2[4].coord[2] = (-(PA_RB_SIZE_4 + PA_RB_SIZE_5)/2 - PA_RB_LENGTH_4)/1000;
    
    PA_rb_pos2[5].coord[0] = 0;
    PA_rb_pos2[5].coord[1] = 0;
    PA_rb_pos2[5].coord[2] = 0;
    
    PA_rb_pos2[6].coord[0] = 0;
    PA_rb_pos2[6].coord[1] = 0;
    PA_rb_pos2[6].coord[2] = 0.14;//((PA_RB_SIZE_5 + PA_RB_SIZE_6)/2 + PA_RB_LENGTH_6)/1000;
    //fprintf(stdout,"PA_rb_pos2[%d] = [%f %f %f]\n",6,PA_rb_pos2[6].coord[0],PA_rb_pos2[6].coord[1],PA_rb_pos2[6].coord[2]);
  }
  fprintf(stdout,".\n");
  fprintf(stdout,"Initialized to angles = [%f %f %f %f %f %f].\n",PA_dh_alpha[0],PA_dh_alpha[1],PA_dh_alpha[2],PA_dh_alpha[3],PA_dh_alpha[4],PA_dh_alpha[5]);
}
