// This program enables simple vector operations. It is currently
// neither geared towards generality or speed, but should be geared
// towards the latter eventually.
//
// Written by Christian Smith Apr 28 2005
// updated by Christian Smith Apr 28 2005

#ifndef _VECTOR_MATH_H
#define _VECTOR_MATH_H

#include <math.h>

#ifndef MAX
#define MAX(x, y)   ((x)>(y)?(x):(y))
#endif

#ifndef MIN
#define MIN(x, y)   ((x)<(y)?(x):(y))
#endif

#ifndef PI
#define PI 3.1415926
#endif

// Datatypes. Should be selfexplanatory:
struct VM_3vector{
  double coord[3];
};

struct VM_4vector{
  double coord[4];
};

struct VM_6vector{
  double coord[6];
};

struct VM_7vector{
  double coord[7];
};

struct VM_3matrix{
  double coord[3][3];
};

struct VM_4matrix{
  double coord[4][4];
};

struct VM_6matrix{
  double coord[6][6];
};

// Functions:
// addition of two 3-vectors
inline VM_3vector VM_add(const VM_3vector a, const VM_3vector b){
  VM_3vector c;
  for(int i=0;i<3;i++){
    c.coord[i] = a.coord[i]+b.coord[i];
  }
  return c;
}

// addition of two 4-vectors
inline VM_4vector VM_add(const VM_4vector a, const VM_4vector b){
  VM_4vector c;
  for(int i=0;i<4;i++){
    c.coord[i] = a.coord[i]+b.coord[i];
  }
  return c;
} 

// addition of two 6-vectors
inline VM_6vector VM_add(const VM_6vector a, const VM_6vector b){
  VM_6vector c;
  for(int i=0;i<6;i++){
    c.coord[i] = a.coord[i]+b.coord[i];
  }
  return c;
}

// addition of two 7-vectors
inline VM_7vector VM_add(const VM_7vector a, const VM_7vector b){
  VM_7vector c;
  for(int i=0;i<7;i++){
    c.coord[i] = a.coord[i]+b.coord[i];
  }
  return c;
}

// subtraction of two 3-vectors
inline VM_3vector VM_sub(const VM_3vector a, const VM_3vector b){
  VM_3vector c;
  for(int i=0;i<3;i++){
    c.coord[i] = a.coord[i]-b.coord[i];
  }
  return c;
}

// subtraction of two 4-vectors
inline VM_4vector VM_sub(const VM_4vector a, const VM_4vector b){
  VM_4vector c;
  for(int i=0;i<4;i++){
    c.coord[i] = a.coord[i]-b.coord[i];
  }
  return c;
} 

// subtraction of two 6-vectors
inline VM_6vector VM_sub(const VM_6vector a, const VM_6vector b){
  VM_6vector c;
  for(int i=0;i<6;i++){
    c.coord[i] = a.coord[i]-b.coord[i];
  }
  return c;
}

// subtraction of two 7-vectors
inline VM_7vector VM_sub(const VM_7vector a, const VM_7vector b){
  VM_7vector c;
  for(int i=0;i<7;i++){
    c.coord[i] = a.coord[i]-b.coord[i];
  }
  return c;
}

// addition of two 3-matrices
inline VM_3matrix VM_add(const VM_3matrix A, const VM_3matrix B){
  VM_3matrix C;
  for(int i=0;i<3;i++){
    for(int j=0;j<3;j++){
      C.coord[i][j] = A.coord[i][j]+B.coord[i][j];
    }
  }
  return C;
} 

// addition of two 4-matrices
inline VM_4matrix VM_add(const VM_4matrix A, const VM_4matrix B){
  VM_4matrix C;
  for(int i=0;i<4;i++){
    for(int j=0;j<4;j++){
      C.coord[i][j] = A.coord[i][j]+B.coord[i][j];
    }
  }
  return C;
} 

// addition of two 6-matrices
inline VM_6matrix VM_add(const VM_6matrix A, const VM_6matrix B){
  VM_6matrix C;
  for(int i=0;i<6;i++){
    for(int j=0;j<6;j++){
      C.coord[i][j] = A.coord[i][j]+B.coord[i][j];
    }
  }
  return C;
} 


// subtraction of two 3-matrices
inline VM_3matrix VM_sub(const VM_3matrix A, const VM_3matrix B){
  VM_3matrix C;
  for(int i=0;i<3;i++){
    for(int j=0;j<3;j++){
      C.coord[i][j] = A.coord[i][j]-B.coord[i][j];
    }
  }
  return C;
} 

// subtraction of two 4-matrices
inline VM_4matrix VM_sub(const VM_4matrix A, const VM_4matrix B){
  VM_4matrix C;
  for(int i=0;i<4;i++){
    for(int j=0;j<4;j++){
      C.coord[i][j] = A.coord[i][j]-B.coord[i][j];
    }
  }
  return C;
} 

// subtraction of two 6-matrices
inline VM_6matrix VM_sub(const VM_6matrix A, const VM_6matrix B){
  VM_6matrix C;
  for(int i=0;i<6;i++){
    for(int j=0;j<6;j++){
      C.coord[i][j] = A.coord[i][j]-B.coord[i][j];
    }
  }
  return C;
} 


// transpose of 3-matrix (is also inverse of rotation matrix)
inline VM_3matrix VM_trans(const VM_3matrix A){
  VM_3matrix C;
  for(int i=0;i<3;i++){
    for(int j=0;j<3;j++){
      C.coord[i][j] = A.coord[j][i];
    }
  }
  return C;
} 

// multiplication of scalar and 3-vector
inline VM_3vector VM_mul(const double a, const VM_3vector b){
  VM_3vector c;
  for(int i=0;i<3;i++){
    c.coord[i] = a*b.coord[i];
  }
  return c;
} 

// multiplication of scalar and 3-matrix
inline VM_3matrix VM_mul(const double a, const VM_3matrix B){
  VM_3matrix C;
  for(int i=0;i<3;i++){
    for(int j=0;j<3;j++){
      C.coord[i][j] = B.coord[i][j]*a;
    }
  }
  return C;
}

// multiplication of 3-matrix and 3-vector
inline VM_3vector VM_mul(const VM_3matrix A, const VM_3vector b){
  VM_3vector c;
  for(int i=0;i<3;i++){
    c.coord[i] = A.coord[i][0]*b.coord[0] + A.coord[i][1]*b.coord[1] + A.coord[i][2]*b.coord[2];
  }
  return c;
} 

// multiplication of two 3-matrices
inline VM_3matrix VM_mul(const VM_3matrix A, const VM_3matrix B){
  VM_3matrix C;
  for(int i=0;i<3;i++){
    for(int j=0;j<3;j++){
      C.coord[i][j] = A.coord[i][0]*B.coord[0][j] + A.coord[i][1]*B.coord[1][j] + A.coord[i][2]*B.coord[2][j];
    }
  }
  return C;
} 

// multiplication of two 4-matrices
inline VM_4matrix VM_mul(const VM_4matrix A, const VM_4matrix B){
  VM_4matrix C;
  for(int i=0;i<4;i++){
    for(int j=0;j<4;j++){
      C.coord[i][j] = A.coord[i][0]*B.coord[0][j] + A.coord[i][1]*B.coord[1][j] + A.coord[i][2]*B.coord[2][j];
    }
  }
  return C;
} 

// multiplication of 4-matrix and 4-vector
inline VM_4vector VM_mul(const VM_4matrix A, const VM_4vector b){
  VM_4vector c;
  for(int i=0;i<4;i++){
    c.coord[i] = A.coord[i][0]*b.coord[0] + A.coord[i][1]*b.coord[1] + A.coord[i][2]*b.coord[2] + A.coord[i][3]*b.coord[3];
  }
  return c;
} 

// multiplication of 6-matrix and 6-vector
inline VM_6vector VM_mul(const VM_6matrix A, const VM_6vector b){
  VM_6vector c;
  for(int i=0;i<6;i++){
    c.coord[i] = A.coord[i][0]*b.coord[0] + A.coord[i][1]*b.coord[1] + A.coord[i][2]*b.coord[2] + 
      A.coord[i][3]*b.coord[3] + A.coord[i][4]*b.coord[4] + A.coord[i][5]*b.coord[5];
  }
  return c;
} 

// dot product for 3-vectors
inline double VM_dot(const VM_3vector a, const VM_3vector b){
  return (a.coord[0]*b.coord[0] + a.coord[1]*b.coord[1] + a.coord[2]*b.coord[2]);
} 

// dot product for 4-vectors
inline double VM_dot(const VM_4vector a, const VM_4vector b){
  return (a.coord[0]*b.coord[0] + a.coord[1]*b.coord[1] + a.coord[2]*b.coord[2] + a.coord[3]*b.coord[3]);
} 

// cross product for 3-vectors
inline VM_3vector VM_cross(const VM_3vector a, const VM_3vector b){
  VM_3vector c;
  for(int i=0;i<3;i++){
    c.coord[i] = a.coord[(i+1)%3]*b.coord[(i+2)%3] - a.coord[(i+2)%3]*b.coord[(i+1)%3];
  }
  return c;
} 

// matrix product for 3-vectors
inline VM_3matrix VM_matrixProd(const VM_3vector a, const VM_3vector b){
  VM_3matrix C;
  for(int i=0;i<3;i++){
    for(int j=0;j<3;j++){
      C.coord[i][j] = a.coord[i]*b.coord[j];
    }
  }
  return C;
} 

// Inverts symmetric 6-matrix:
inline VM_6matrix VM_inverseSym(const VM_6matrix A){
  VM_6matrix _C;
  VM_6matrix _I;
  int _ri[6]; // row index

  for(int i=0;i<6;i++){
    for(int j=i;j<6;j++){
      _C.coord[i][j] = (_C.coord[j][i] =  A.coord[i][j]);
      _I.coord[i][j] = (_I.coord[j][i] = (double)(i==j));
    }
    _ri[i]=i;
  }
  
  for(int i=0;i<6;i++){
    
    // find largest element in column i
    double _max_val=0.0;
    int _max_pos = 0;
    for(int j=i;j<6;j++){
      if(fabs(_C.coord[_ri[j]][i])>_max_val){
	_max_val = fabs(_C.coord[_ri[j]][i]);
	_max_pos = j;
      }
    }

    //switch rows:
    int _tmp=_ri[i];
    _ri[i]=_ri[_max_pos];
    _ri[_max_pos]=_tmp;

    // normalize row i
    double _temp_div = _C.coord[_ri[i]][i];
    for(int j=0;j<6;j++){
      _I.coord[_ri[i]][j]=_I.coord[_ri[i]][j]/_temp_div;
      _C.coord[_ri[i]][j]=_C.coord[_ri[i]][j]/_temp_div;
    }

    // eliminate lower rows
    if(i<5){
      for(int j=i+1; j<6; j++){
	double _temp_mul =  _C.coord[_ri[j]][i];
	for(int k=0; k<6; k++){
	  _I.coord[_ri[j]][k] -= _temp_mul*_I.coord[_ri[i]][k];
	  _C.coord[_ri[j]][k] -= _temp_mul*_C.coord[_ri[i]][k];
	}
      }
    }

  }
 
  // eliminate upper rows
  for(int i=4; i>(-1); i--){
    for(int j=0; j<(i+1); j++){
      for(int k=0; k<6; k++){
	_I.coord[_ri[j]][k] -= _C.coord[_ri[j]][i+1]*_I.coord[_ri[i+1]][k];
      }
    }
  }

  VM_6matrix _Iout;
  for(int i=0; i<6; i++){
    for(int j=0; j<6; j++){
      _Iout.coord[i][j] = _I.coord[_ri[i]][j];
    }
  }

  return _Iout;
}

// the following functions are not as convenient to call, but are 
// faster than the above. Does not use the the structs defined in
// this header, but general arrays.
/*
inline void VM_3vector_add(const double a_[3], const double b_[3], double d_[3]){
  for(int i=0;i<3;i++){
    d_[i] = a_[i]+b_[i];
  }
} 

inline double VM_3vector_dot(const double a_[3], const double b_[3]){
  double c_ = 0;
  for(int i=0;i<3;i++){
    c_ += a_[i]*b_[i];
  }
  return c_;
} 

inline void VM_3vector_cross(const double a_[3], const double b_[3], double d_[3]){
  for(int i=0;i<3;i++){
    d_[i] = a_[(i+1)%3]*b_[(i+2)%3] - a_[(i+2)%3]*b_[(i+1)%3];
  }
} 

inline void VM_3matrix_vector_mult(const double M_[3][3], const double v_[3], double d_[3]){
  for(int i=0;i<3;i++){
    d_[i] = M_[i][0]*v_[0] + M_[i][1]*v_[1] + M_[i][2]*v_[2];
  }
}
*/
#endif // #ifndef _VECTOR_MATH_H_
