//
// You received this file as part of MCA2
// Modular Controller Architecture Version 2
//
//Copyright (C) Forschungszentrum Informatik Karlsruhe
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// this is a -*- C++ -*- file
//----------------------------------------------------------------------
/*!\file    JointSimulation.C
 *
 * \author   Christian Smith
 * \date    25.04.05
 * \update  04.05.05
 *
 * $Revision: 1.5 $
 * $Id: NullModule.C,v 1.5 2003/02/24 14:36:21 gassmann Exp $
 *
 */
//----------------------------------------------------------------------
// Includes
//----------------------------------------------------------------------

#include "JointSimulation.h"



//----------------------------------------------------------------------
// Module Debug
//----------------------------------------------------------------------
//#undef MODULE_DEBUG
#define MODULE_DEBUG
#include <module_debug.h>

//----------------------------------------------------------------------
// defines and consts
//----------------------------------------------------------------------

//----------------------------------------------------------------------
// class JointSimulation constructor
//----------------------------------------------------------------------
tJointSimulation::tJointSimulation(tParent *parent,
			   tDescription name,bool fixit)
  :tModule(parent,name,
	       eSI_DIMENSION,eSO_DIMENSION,eCI_DIMENSION,eCO_DIMENSION,
	       si_description,so_description,ci_description,co_description,
	       fixit)
{
  InitParameter(ePAR_DIMENSION,
		new tParameterFloat("min angle value",-2*M_PI,2*M_PI,-M_PI), //min,max,value
		new tParameterFloat("max angle value",-2*M_PI,2*M_PI,M_PI), //min,max,value
		new tParameterFloat("max angular velocity",0,100,6), //min,max,value
		new tParameterFloat("max angular acc",0,100,100.0) //min,max,value
		);
}

//----------------------------------------------------------------------
// class JointSimulation destructor
//----------------------------------------------------------------------
tJointSimulation::~tJointSimulation()
{
}

//----------------------------------------------------------------------
// class JointSimulation Control
//----------------------------------------------------------------------
void  tJointSimulation::Control()
{
  //  double t;
  static double target_acc = 0;
  static int lapcounter = 0;
  //#define JS_CONTROL_FREE_LAPS (1)
  if (ControllerInputChanged())
    {
      // calculate the elapsed time from last access to control
      double elapsed_time =(tTime::Now()-sense_timer).ToUSec()*0.000001;
      // reset timer
      sense_timer.FromNow();
      target_acc = ControllerInput(eCI_JOINT_ACC);
	
      //fprintf(stdout,"Joint: acc=%f old_angle=%f ",target_acc, old_angle);
      
      // calculate two first terms of Taylor series for new angle
      double diff_angle = elapsed_time * angular_vel;
      double diff_angle2 = (target_acc*(elapsed_time*elapsed_time))/2;
      
      // update the angle and velocity and check if we're close enough
#define SPEED_TOLERANCE 0.0001
      if((fabs(angular_vel) > SPEED_TOLERANCE) || (fabs(target_acc) > SPEED_TOLERANCE)){
	new_angle = old_angle + diff_angle + diff_angle2;
	angular_vel += (target_acc*elapsed_time);
      }
      else{
	angular_vel=target_acc=0;
	ClearControllerInput();
      }
      
      // Out of Min/Max
      if (new_angle < min_angle_value){
	new_angle=min_angle_value;
	angular_vel = 0;
      }
      if (new_angle > max_angle_value){
	new_angle=max_angle_value;
	angular_vel = 0;
      }
      //save the stat
      old_angle=new_angle;
	
      //fprintf(stdout,"vel=%f new_angle=%f \n",angular_vel, new_angle);
 
      //fprintf(stdout,"new angle=%f\n",(double)new_angle);
      
      SensorOutput(eSO_CURRENT_JOINT_VALUE)=new_angle;
      SensorOutput(eSO_JOINT_VELOCITY)=angular_vel;
      SensorOutput(eSO_JOINT_ACC)=target_acc;
      
      SetSensorOutputChanged();
    }
  else
    {
      //
      sense_timer.FromNow();
      ClearSensorOutput();
      angular_vel=0;
    }
}

//----------------------------------------------------------------------
// class JointSimulation Sense
//----------------------------------------------------------------------
void  tJointSimulation::Sense()
{
  if (SensorInputChanged())
    {
      //fprintf(stdout,"MUPP:OUTPUT=%f\n",SensorOutput(eSO_CURRENT_JOINT_VALUE));
      for (int i=0;i<SensorInputDimension();i++)
	SensorOutput(i)=SensorInput(i);
      // Es wurde was gendert, also
      SetSensorOutputChanged();
      // Eingang wurde damit abgearbeitet, also nderung zurcksetzen:
      ClearSensorInput();
    }
  else{
    //  ClearSensorOutput();
    //fprintf(stdout,"OUTPUT=%f\n",SensorOutput(eSO_CURRENT_JOINT_VALUE));
  }
}

//----------------------------------------------------------------------
// class JointSimulation Exception
//----------------------------------------------------------------------
void tJointSimulation::Exception(tExceptionType type)
{
  switch (type)
    {

    case eET_ParameterChanged:
      min_angle_value=ParameterFloat(ePAR_MIN_ANGLE);
      max_angle_value=ParameterFloat(ePAR_MAX_ANGLE);
      max_angular_velocity=ParameterFloat(ePAR_ANGULAR_VELOCITY);
      ClearParameter();
      break;

    case eET_PrepareForFirstSense:
      break;
    case eET_PrepareForFirstControl:
      break;
    case eET_PrepareForBreak:
      break;
    case eET_PrepareForRestart:
      break;
    }
}
