// this is a -*- C++ -*- file
//----------------------------------------------------------------------
/*!\file    Kinematic.h
 *
 * \author   Kay-Ulrich Scholl
 * \date    06.11.02
 *
 * $Revision: 1.1 $
 * $Id: Kinematic.h,v 1.1 2002/11/06 16:08:23 scholl Exp $
 *
 * \brief   Contains tKinematic
 *
 * \b tKinematic
 *
 * Kinematic for PUMA robot. Just got this
 * from Hochholdinger and converted to MCA2.3
 *
 */
//----------------------------------------------------------------------

#ifndef _Kinematic_h_
#define _Kinematic_h_
//----------------------------------------------------------------------
// System Includes - include with <>
//----------------------------------------------------------------------
// z.B. #include <stdio.h>
#include <Module.h>
#include <descr.h>
//----------------------------------------------------------------------
// Project Includes - include with ""
//----------------------------------------------------------------------

//----------------------------------------------------------------------
// Extern methods and global vars
//----------------------------------------------------------------------
#ifndef PI
#define PI 3.141592654
#endif
//----------------------------------------------------------------------
// Forward class declarations
//----------------------------------------------------------------------

//----------------------------------------------------------------------
// Implementation
//----------------------------------------------------------------------

//! Short description of tKinematic
/*! Kinematics for PUMA robot
*/
class tKinematic:public tModule
{
public:
  /*!
    Anonymous enumeration typ which contains the indices of the
    controller inputs.
   */
  _DESCR(tKinematic::ci_description,4,Natural,cDATA_VECTOR_END_MARKER)
  enum {
    eCI_TOOL_X,   /*! X-Coordinate of the position-vector */
    eCI_TOOL_Y,   /*! Y-Coordinate of the position-vector */
    eCI_TOOL_Z,   /*! Z-Coordinate of the position-vector */
    eCI_TOOL_O,   /*! Orientation angle of the tool [rad] */
    eCI_TOOL_A,   /*! Altitude angle of the tool [rad]    */
    eCI_TOOL_T,   /*! Tool angle of the tool [rad]        */
    eCI_ARM,      /*! Switch for the arm-orientation [left/right]    */
    eCI_ELBOW,    /*! Switch for the elbow-orientation [above/below] */
    eCI_WRIST,    /*! Switch for the wrist-orientation [down/up]     */
    eCI_FLIP,     /*! Toggle the other way round */
    eCI_DIMENSION /*!< Endmarker and Dimension */
  };

  /*!
    Anonymous enumeration typ which contains the indices of the
    controller outputs.
   */
  _DESCR(tKinematic::co_description,4,Natural,cDATA_VECTOR_END_MARKER)
  enum {
    eCO_WAIST,       /*! Angle of the waist-joint [rad]          */
    eCO_SHOULDER,    /*! Angle of the shoulder-joint [rad]       */
    eCO_ELBOW,       /*! Angle of the elbow-joint [rad]          */
    eCO_WRIST_ROT,   /*! Angle of the wrist_rotation-joint [rad] */
    eCO_WRIST_BEND,  /*! Angle of the wrist_bend-joint [rad]     */
    eCO_FLANGE,      /*! Angle of the flange-joint [rad]         */
    eCO_DIMENSION /*!< Endmarker and Dimension */
  };

  /*!
    Anonymous enumeration typ which contains the indices of the
    sensor inputs.
   */
  _DESCR(tKinematic::si_description,4,Natural,cDATA_VECTOR_END_MARKER)
  enum {
    eSI_WAIST,       /*! Angle of the waist-joint [rad]          */
    eSI_SHOULDER,    /*! Angle of the shoulder-joint [rad]       */
    eSI_ELBOW,       /*! Angle of the elbow-joint [rad]          */
    eSI_WRIST_ROT,   /*! Angle of the wrist_rotation-joint [rad] */
    eSI_WRIST_BEND,  /*! Angle of the wrist_bend-joint [rad]     */
    eSI_FLANGE,      /*! Angle of the flange-joint [rad]         */
    eSI_DIMENSION /*!< Endmarker and Dimension */
  };

  /*!
    Anonymous enumeration typ which contains the indices of the
    sensor outputs.
   */
  _DESCR(tKinematic::so_description,4,Natural,cDATA_VECTOR_END_MARKER)
  enum {
    eSO_TOOL_X,     /*! X-Coordinate of the position-vector */
    eSO_TOOL_Y,     /*! Y-Coordinate of the position-vector */
    eSO_TOOL_Z,     /*! Z-Coordinate of the position-vector */
    eSO_TOOL_O,     /*! Orientation angle of the tool [rad] */
    eSO_TOOL_A,     /*! Altitude angle of the tool [rad]    */
    eSO_TOOL_T,     /*! Tool angle of the tool [rad]        */
    eSO_ARM,        /*! Indicator of the arm-orientation [left/right]    */
    eSO_ELBOW,      /*! Indicator of the elbow-orientation [above/below] */
    eSO_WRIST,      /*! Indicator of the wrist-orientation [down/up]     */
    eSO_FLIP,       /*! Toggle the other way round */
    eSO_OVERFLOW,   /*! Indicator if the position is out of range */
    eSO_DIMENSION /*!< Endmarker and Dimension */
  };

  /*!
    Anonymous enumeration typ which contains the indices of the
    parameters.
   */
  enum {
    ePAR_A2, /*!< (Denavit-Hartenberg Parameters) */
    ePAR_D2, /*!<            - " -                */
    ePAR_A3, /*!<            - " -                */
    ePAR_D4, /*!<            - " -                */
    ePAR_D6, /*!<            - " -                */
    ePAR_D3, /*!<            - " -                */
    /*! Min and Max range of the waist angles [rad]*/
    ePAR_WAIST_MIN,
    ePAR_WAIST_MAX,
    /*! Min and Max range of the shoulder angles [rad]*/
    ePAR_SHOULDER_MIN,
    ePAR_SHOULDER_MAX,
    /*! Min and Max range of the elbow angles [rad]*/
    ePAR_ELBOW_MIN,
    ePAR_ELBOW_MAX,
    /*! Min and Max range of the wrist-rotation angles [rad]*/
    ePAR_WRIST_ROT_MIN,
    ePAR_WRIST_ROT_MAX,
    /*! Min and Max range of the wrist-bend angles [rad]*/
    ePAR_WRIST_BEND_MIN,
    ePAR_WRIST_BEND_MAX,
    /*! Min and Max range of the flange angles [rad] */
    ePAR_FLANGE_MIN,
    ePAR_FLANGE_MAX,
    ePAR_DIMENSION /*!< Endmarker and Dimension */
  };

  /*!
    Descriptions of the controller inputs
  */
  static const tDescription ci_description[];
  /*!
    Descriptions of the controller outputs
  */
  static const tDescription co_description[];
  /*!
    Descriptions of the sensor inputs
   */
  static const tDescription si_description[];
  /*!
    Descriptions of the sensor outputs
   */
  static const tDescription so_description[];
  /*!
   tKinematic initializes the Sensor-,ControllerIn and Output and the
    Parameters. The Parameters are also stored in our privat area.
   */
  tKinematic(tParent *parent,
	     tDescription name="Kinematic",bool fixtit=true);
  /*!
    Nothing to be done in the destructor
   */
  ~tKinematic();
  /*!
   In Controle is the inverse kinematics implemeted.
    If the angles are out of bounds or not calculatable, we set an
    overflow-flag and an reflector for the sense-function to give a Signal on
    SensorOut. The angles are kept until the new ones are in bounds.
   */
  void Control();
  /*!
    The Sense functions represents the direct kinematics.
    We calculate the direction of the arm, elbow and wrist.
    Also show the "overflow-case" if the position wanted in the inverse
    kinematic is not in range.
  */
  void Sense();
  /*!
   */
  void Exception(tExceptionType type);
private:
  /*!
    This small helper-function sets the value to zero if it is near zero.
    This is necessary becaus of the atan2-function calculating with rounding
    errrors.
  */
  void inline Small(long double *value);

  /*!
    This function is true if the input-angle is in range
  */
  bool InRange(long double *angle, long double *par_min, long double *par_max);

  /*!
    convert the borders in the range from -PI to PI
  */

  void Convert(long double *par_min, long double *par_max);

  /*!
    The parameters are kept here for internal use
  */

  // The Denavit-Hartenberg parameters:
  long double a2,d2,a3,d4,d6,d3;

  // the range of the angles are stored here
  long double waist_min, waist_max;
  long double shoulder_min, shoulder_max;
  long double elbow_min, elbow_max;
  long double wrist_rot_min, wrist_rot_max;
  long double wrist_bend_min, wrist_bend_max;
  long double flange_min, flange_max;

  // keep the old angles here for fallback purpose
  long double d_temp[6];
  long double K_input_d[6];

  // We want to show the overflow
  long double overflow;

  // We have to reflect the overflow from Controle to Sense
  bool reflect;

  tTime K_sense_timer;
};

#endif
