//Copyright (C) CAS, Christian Smith
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// this is a -*- C++ -*- file
//----------------------------------------------------------------------
/*!\file    JointControl.C
 *
 * \author   Christian Smith
 * \date    2005.05.24
 * \update  2005.05.24
 *
 * $Revision: 1.5 $
 * $Id: NullModule.C,v 1.5 2003/02/24 14:36:21 gassmann Exp $
 *
 */
//----------------------------------------------------------------------
// Includes
//----------------------------------------------------------------------

#include "JointControl.h"


//----------------------------------------------------------------------
// Module Debug
//----------------------------------------------------------------------
//#undef MODULE_DEBUG
#define MODULE_DEBUG
#include <module_debug.h>
//#define LOG_JOINT_CONTROL

//----------------------------------------------------------------------
// defines and consts
//----------------------------------------------------------------------

//----------------------------------------------------------------------
// class JointControl constructor
//----------------------------------------------------------------------
tJointControl::tJointControl(tParent *parent,
			   tDescription name,bool fixit)
  :tModule(parent,name,
	       eSI_DIMENSION,eSO_DIMENSION,eCI_DIMENSION,eCO_DIMENSION,
	       si_description,so_description,ci_description,co_description,
	       fixit)
{
  InitParameter(ePAR_DIMENSION,
		new tParameterFloat("max ang vel waist",0, 10, 8.2),
		new tParameterFloat("max ang vel shld",0, 10, 4.1),
		new tParameterFloat("max ang vel elb",0, 10, 8.2),
		new tParameterFloat("max ang vel wr rot",0, 10, 8.2),
		new tParameterFloat("max ang vel wr bnd",0, 10, 4.3),
		new tParameterFloat("max ang vel flange",0, 10, 6.2)
		);
}

//----------------------------------------------------------------------
// class JointControl destructor
//----------------------------------------------------------------------
tJointControl::~tJointControl()
{
}

//----------------------------------------------------------------------
// class JointControl Control
//----------------------------------------------------------------------
void  tJointControl::Control()
{
  //  double t;
  static int timecount=0;

  if (ControllerInputChanged())
    {
      // calculate the elapsed time from last access to control
      
      // the number of loops that the time is averaged over, to avoid division by zero
#define JC_TIMERSTEPS 500
      static double elapsed_time=0.01;
     
      if(++timecount>JC_TIMERSTEPS){
	elapsed_time =((tTime::Now()-sense_timer).ToUSec()*0.000001/timecount)+0.000001;
	// reset timer
	sense_timer.FromNow();
	timecount=0;
      }

#define TARGET_VEL_FACTOR 15.0
#define TARGET_VEL_FACTOR1 15.0
#define TARGET_VEL_FACTOR2 2.0
#define MAXIMUM_DECELERATION 80
      

      // Get the desired accelerations
      for(int i=0; i<6; i++){
	old_angle[i]=SensorInput(i);
	new_angle[i]=ControllerInput(i);
	angular_vel[i]=SensorInput(i+6);

	// calculate desired speeds and accelerations;
	double target_distance = (new_angle[i] - old_angle[i]);
	double target_vel = sqrt(fabs(target_distance)*2*MAXIMUM_DECELERATION)*Sgn(target_distance)/1.3;
	
	if(fabs(target_distance)<0.07){
	  target_vel=(target_distance/elapsed_time)*0.35;
	}
	
	double vel_dir;  
	// we don't desire impossible speeds:
	(target_vel > 0) ? vel_dir = 1 : vel_dir= -1;
	target_vel = MIN(fabs(target_vel),max_angular_velocity[i])*vel_dir;
	
	target_acc[i] = (target_vel - angular_vel[i])/(elapsed_time);

#ifdef LOG_JOINT_CONTROL
	FILE *controllog=fopen("logs/joint_control.log","a");
	
	fprintf(controllog,"jnt[%d] old_a=%f new_a=%f c_vel=%f t_vel=%f t_acc=%f\n",
		i,old_angle[i],new_angle[i],angular_vel[i],target_vel,target_acc[i]);
	
	fclose(controllog);
#endif //#ifdef LOG_JOINT_CONTROL

      }
      for(int i=0;i<6;i++){
	ControllerOutput(i)=target_acc[i];
      } 
      SetControllerOutputChanged();
    }
  else
    {
    if((++timecount%JC_TIMERSTEPS)==0){
	// reset timer
	sense_timer.FromNow();
	timecount=0;
      }

      ClearSensorOutput();
    }
}

//----------------------------------------------------------------------
// class JointControl Sense
//----------------------------------------------------------------------
void  tJointControl::Sense()
{
  if (SensorInputChanged())
    {
      //fprintf(stdout,"MUPP:OUTPUT=%f\n",SensorOutput(eSO_CURRENT_JOINT_VALUE));
      for (int i=0;i<SensorInputDimension();i++)
	SensorOutput(i)=SensorInput(i);
      // Es wurde was gendert, also
      SetSensorOutputChanged();
      // Eingang wurde damit abgearbeitet, also nderung zurcksetzen:
      ClearSensorInput();
    }
  else{
    //  ClearSensorOutput();
    //fprintf(stdout,"OUTPUT=%f\n",SensorOutput(eSO_CURRENT_JOINT_VALUE));
  }
}

//----------------------------------------------------------------------
// class JointControl Exception
//----------------------------------------------------------------------
void tJointControl::Exception(tExceptionType type)
{
  switch (type)
    {

    case eET_ParameterChanged:
      
      max_angular_velocity[0]=ParameterFloat(ePAR_MAX_ANG_VEL_WAIST);
      max_angular_velocity[1]=ParameterFloat(ePAR_MAX_ANG_VEL_SHOULDER);
      max_angular_velocity[2]=ParameterFloat(ePAR_MAX_ANG_VEL_ELBOW); 
      max_angular_velocity[3]=ParameterFloat(ePAR_MAX_ANG_VEL_WRIST_ROT);
      max_angular_velocity[4]=ParameterFloat(ePAR_MAX_ANG_VEL_WRIST_BEND);
      max_angular_velocity[5]=ParameterFloat(ePAR_MAX_ANG_VEL_FLANGE); 

      ClearParameter();
      break;

    case eET_PrepareForFirstSense:
      break;
    case eET_PrepareForFirstControl:
      break;
    case eET_PrepareForBreak:
      break;
    case eET_PrepareForRestart:
      break;
    }
}
