#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <math.h>

#include "visiontrackerlib.h"
#include <pthread.h>

#include <sys/types.h>
#include <sys/time.h>
#include <time.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "packet_typdef.h"

#include <cwiid.h>

#include "tracker.h"

/* 
    This program aims to test the capabilities of the accelerometers to determine 
    position.

    Parts of the code is stolen from wmdemo.c from libcwiid.

    The rest is written by Christian Smith, <ccs@kth.se>

    compile by issuing:
    
    gcc -Wall -lm -lcwiid robot_control.c tracker.c -o robot_control

    // local version with lib paths explicit:
     gcc -Wall -lm -lcwiid  -lsvs -lfltk -lpthread  -L../../../SmallVisionSystem/bin -lcwrap -I../../../SmallVisionSystem/samples/  robot_control.c tracker.c  ../../../SmallVisionSystem/src/visiontrackerlib.o ../../../SmallVisionSystem/samples/flwin.o ../../../SmallVisionSystem/src/rgb2hsv.o -o robot_control



*/


// Used for internet connection
int sock, n;
unsigned int length; 
struct sockaddr_in robot,from;
struct hostent *hp;
char buffer[256];
struct posvel_control_packet utdata;   
struct response_packet indata;


int robot_send(double x,
	       double y,
	       double z,
	       double pan,
	       double tilt,
	       double yaw,
	       double xv,
	       double yv,
	       double zv);


int robot_connect(void);


cwiid_mesg_callback_t cwiid_callback;
int pos_tracking = 0;
int robot_controlling = 0;
int reset = 0;

// 1 = free mode, 2 = virtual fixtures, 3 = task mode
int control_mode = 1;

struct acc_cal main_acc_cal;
struct acc_cal nunchuk_acc_cal;
FILE *logfile;
FILE *logfile_ang;

int vision_tracker_running = 0;

int X1 = -1000;
int X2 = -1000;
int Y1 = -1000;
int Y2 = -1000;

int have_ir=0;  // If we have an IR source or not.
                // used to determine if we can calibrate

unsigned char led_state = 0;


#define SIGN(X)  (((X)>=0)?(1):(-1))

#define GRAVITY 9.81

#ifndef PI
#define PI (3.1415926)
#endif

#define toggle_bit(bf,b)	\
	(bf) = ((bf) & b)		\
	       ? ((bf) & ~(b))	\
	       : ((bf) | (b))

#define unset_bit(bf,b)	 (bf) = ((bf) & ~(b))
#define set_bit(bf,b)	 (bf) = ((bf) | (b))


#define MENU \
	"1: Set free motion mode\n" \
	"2: Set virtual fixture mode\n" \
	"3: Set task selection mode\n" \
	"4: Set free mode with angle control\n" \
	"5: toggle rumble\n" \
	"a: toggle accelerometer reporting\n" \
	"b: toggle button reporting\n" \
	"c: toggle position tracking (Nunchuck)\n" \
	"e: toggle extension reporting\n" \
	"h: toggle robot control\n" \
	"i: toggle ir reporting\n" \
	"m: toggle messages\n" \
	"p: print this menu\n" \
	"r: request status message ((t) enables callback output)\n" \
	"s: print current state\n" \
	"t: toggle status reporting\n" \
	"x: exit\n"

void set_led_state(cwiid_wiimote_t *wiimote, unsigned char led_state);
void set_rpt_mode(cwiid_wiimote_t *wiimote, unsigned char rpt_mode);
void print_state(struct cwiid_state *state);

void postrack(int x_acc, int y_acc, int z_acc, int reset);
void robot_ctr(int x_acc, int y_acc, int z_acc,
	       int ir_x1,int ir_y1,
	       int ir_x2,int ir_y2,
	       int reset);

cwiid_err_t err;
void err(cwiid_wiimote_t *wiimote, const char *s, va_list ap)
{
	if (wiimote) printf("%d:", cwiid_get_id(wiimote));
	else printf("-1:");
	vprintf(s, ap);
	printf("\n");
}

int main(int argc, char *argv[])
{
        cwiid_wiimote_t *wiimote;	/* wiimote handle */
	struct cwiid_state state;	/* wiimote state */
	bdaddr_t bdaddr;	/* bluetooth device address */
	unsigned char mesg = 0;
	unsigned char rpt_mode = 0;
	unsigned char rumble = 0;
	int exit = 0;

	pthread_t pth;
	    
	//pth = tracker_init();

	//connect to robot
	robot_connect();

	cwiid_set_err(err);

	/* Connect to address given on command-line, if present */
	if (argc > 1) {
		str2ba(argv[1], &bdaddr);
	}
	else {
		bdaddr = *BDADDR_ANY;
	}

	/* Connect to the wiimote */
	printf("Put Wiimote in discoverable mode now (press 1+2)...\n");
	if (!(wiimote = cwiid_open(&bdaddr, 0))) {
		fprintf(stderr, "Unable to connect to wiimote\n");
		return -1;
	}
	if (cwiid_set_mesg_callback(wiimote, cwiid_callback)) {
		fprintf(stderr, "Unable to set message callback\n");
	}

	if (cwiid_get_acc_cal(wiimote, CWIID_EXT_NONE, &main_acc_cal)){
	  fprintf(stderr, "Error getting accelaration calibration for main\n");
	}

	if (cwiid_get_acc_cal(wiimote, CWIID_EXT_NUNCHUK, &nunchuk_acc_cal)){
	  fprintf(stderr, "Error getting accelaration calibration for nunchuk\n");
	}

	printf("Note: To demonstrate the new API interfaces, wmdemo no longer "
	       "enables messages by default.\n"
		   "Output can be gathered through the new state-based interface (s), "
	       "or by enabling the messages interface (c).\n");


	logfile = fopen("acc_logs.txt","w");
	logfile_ang = fopen("ang_logs.txt","w");




	/* Menu */
	printf("%s", MENU);

	while (!exit) {
		switch (getchar()) {
		case '1':
			set_bit(led_state, CWIID_LED1_ON);
			unset_bit(led_state, CWIID_LED2_ON);
			unset_bit(led_state, CWIID_LED3_ON);
			unset_bit(led_state, CWIID_LED4_ON);
			set_led_state(wiimote, led_state);
			control_mode = 1;
			break;
		case '2':
			unset_bit(led_state, CWIID_LED1_ON);
			set_bit(led_state, CWIID_LED2_ON);
			unset_bit(led_state, CWIID_LED3_ON);
			unset_bit(led_state, CWIID_LED4_ON);
			set_led_state(wiimote, led_state);
			control_mode = 2;
			break;
		case '3':
    		        unset_bit(led_state, CWIID_LED1_ON);
			unset_bit(led_state, CWIID_LED2_ON);
			set_bit(led_state, CWIID_LED3_ON);
			unset_bit(led_state, CWIID_LED4_ON);
			set_led_state(wiimote, led_state);
			control_mode = 3;
			break;
		case '4':
    		        unset_bit(led_state, CWIID_LED1_ON);
			unset_bit(led_state, CWIID_LED2_ON);
			unset_bit(led_state, CWIID_LED3_ON);
			set_bit(led_state, CWIID_LED4_ON);
			set_led_state(wiimote, led_state);
			control_mode = 4;
			break;
		case '5':
			toggle_bit(rumble, 1);
			if (cwiid_set_rumble(wiimote, rumble)) {
				fprintf(stderr, "Error setting rumble\n");
			}
			break;
		case 'a':
			toggle_bit(rpt_mode, CWIID_RPT_ACC);
			set_rpt_mode(wiimote, rpt_mode);
			break;
		case 'b':
			toggle_bit(rpt_mode, CWIID_RPT_BTN);
			set_rpt_mode(wiimote, rpt_mode);
			break;
		case 'c':
  		        if(!pos_tracking) pos_tracking = 1;
			else pos_tracking = 0;
			break;
		case 'e':
			/* CWIID_RPT_EXT is actually
			 * CWIID_RPT_NUNCHUK | CWIID_RPT_CLASSIC */
			toggle_bit(rpt_mode, CWIID_RPT_EXT);
			set_rpt_mode(wiimote, rpt_mode);
			break;
		case 'h':
		        robot_controlling = !robot_controlling;
			break;
		case 'i':
			/* libwiimote picks the highest quality IR mode available with the
			 * other options selected (not including as-yet-undeciphered
			 * interleaved mode */
			toggle_bit(rpt_mode, CWIID_RPT_IR);
			set_rpt_mode(wiimote, rpt_mode);
			break;
		case 'm':
			if (!mesg) {
				if (cwiid_enable(wiimote, CWIID_FLAG_MESG_IFC)) {
					fprintf(stderr, "Error enabling messages\n");
				}
				else {
					mesg = 1;
				}
			}
			else {
				if (cwiid_disable(wiimote, CWIID_FLAG_MESG_IFC)) {
					fprintf(stderr, "Error disabling message\n");
				}
				else {
					mesg = 0;
				}
			}
			break;
		case 'p':
			printf("%s", MENU);
			break;
		case 'r':
			if (cwiid_request_status(wiimote)) {
				fprintf(stderr, "Error requesting status message\n");
			}
			break;
		case 's':
			if (cwiid_get_state(wiimote, &state)) {
				fprintf(stderr, "Error getting state\n");
			}
			print_state(&state);
			break;
		case 't':
			toggle_bit(rpt_mode, CWIID_RPT_STATUS);
			set_rpt_mode(wiimote, rpt_mode);
			break;
		case 'x':
			exit = -1;
			break;
		case '\n':
			break;
		default:
			fprintf(stderr, "invalid option\n");
		}
	}

	if (cwiid_close(wiimote)) {
		fprintf(stderr, "Error on wiimote disconnect\n");
		return -1;
	}

	fclose(logfile);
	return 0;
}

void set_led_state(cwiid_wiimote_t *wiimote, unsigned char led_state)
{
	if (cwiid_set_led(wiimote, led_state)) {
		fprintf(stderr, "Error setting LEDs \n");
	}
}
	
void set_rpt_mode(cwiid_wiimote_t *wiimote, unsigned char rpt_mode)
{
	if (cwiid_set_rpt_mode(wiimote, rpt_mode)) {
		fprintf(stderr, "Error setting report mode\n");
	}
}

void print_state(struct cwiid_state *state)
{
	int i;
	int valid_source = 0;

	printf("Report Mode:");
	if (state->rpt_mode & CWIID_RPT_STATUS) printf(" STATUS");
	if (state->rpt_mode & CWIID_RPT_BTN) printf(" BTN");
	if (state->rpt_mode & CWIID_RPT_ACC) printf(" ACC");
	if (state->rpt_mode & CWIID_RPT_IR) printf(" IR");
	if (state->rpt_mode & CWIID_RPT_NUNCHUK) printf(" NUNCHUK");
	if (state->rpt_mode & CWIID_RPT_CLASSIC) printf(" CLASSIC");
	printf("\n");
	
	printf("Active LEDs:");
	if (state->led & CWIID_LED1_ON) printf(" 1");
	if (state->led & CWIID_LED2_ON) printf(" 2");
	if (state->led & CWIID_LED3_ON) printf(" 3");
	if (state->led & CWIID_LED4_ON) printf(" 4");
	printf("\n");

	printf("Rumble: %s\n", state->rumble ? "On" : "Off");

	printf("Battery: %d%%\n",
	       (int)(100.0 * state->battery / CWIID_BATTERY_MAX));

	printf("Buttons: %X\n", state->buttons);

	printf("Acc: x=%d y=%d z=%d\n", state->acc[CWIID_X], state->acc[CWIID_Y],
	       state->acc[CWIID_Z]);

	printf("IR: ");
	for (i = 0; i < CWIID_IR_SRC_COUNT; i++) {
		if (state->ir_src[i].valid) {
			valid_source++;
			printf("(%d,%d) ", state->ir_src[i].pos[CWIID_X],
			                   state->ir_src[i].pos[CWIID_Y]);
		}
	}
	if (!valid_source) {
		printf("no sources detected");
	}
	printf(" have IR(%d) ",valid_source);
	if(valid_source > 1){
	  have_ir = 1;
	}else{
	  have_ir = 0;
	}
	printf("\n");

	switch (state->ext_type) {
	case CWIID_EXT_NONE:
		printf("No extension\n");
		break;
	case CWIID_EXT_UNKNOWN:
		printf("Unknown extension attached\n");
		break;
	case CWIID_EXT_NUNCHUK:
		printf("Nunchuk: btns=%.2X stick=(%d,%d) acc.x=%d acc.y=%d "
		       "acc.z=%d\n", state->ext.nunchuk.buttons,
		       state->ext.nunchuk.stick[CWIID_X],
		       state->ext.nunchuk.stick[CWIID_Y],
		       state->ext.nunchuk.acc[CWIID_X],
		       state->ext.nunchuk.acc[CWIID_Y],
		       state->ext.nunchuk.acc[CWIID_Z]);
		break;
	case CWIID_EXT_CLASSIC:
		printf("Classic: btns=%.4X l_stick=(%d,%d) r_stick=(%d,%d) "
		       "l=%d r=%d\n", state->ext.classic.buttons,
		       state->ext.classic.l_stick[CWIID_X],
		       state->ext.classic.l_stick[CWIID_Y],
		       state->ext.classic.r_stick[CWIID_X],
		       state->ext.classic.r_stick[CWIID_Y],
		       state->ext.classic.l, state->ext.classic.r);
		break;
	}
}

/* Prototype cwiid_callback with cwiid_callback_t, define it with the actual
 * type - this will cause a compile error (rather than some undefined bizarre
 * behavior) if cwiid_callback_t changes */
/* cwiid_mesg_callback_t has undergone a few changes lately, hopefully this
 * will be the last.  Some programs need to know which messages were received
 * simultaneously (e.g. for correlating accelerometer and IR data), and the
 * sequence number mechanism used previously proved cumbersome, so we just
 * pass an array of messages, all of which were received at the same time.
 * The id is to distinguish between multiple wiimotes using the same callback.
 * */
void cwiid_callback(cwiid_wiimote_t *wiimote, int mesg_count,
                    union cwiid_mesg mesg[], struct timespec *timestamp)
{
	int i, j;
	int valid_source;

	for (i=0; i < mesg_count; i++)
	{
		switch (mesg[i].type) {
		case CWIID_MESG_STATUS:
			printf("Status Report: battery=%d extension=",
			       mesg[i].status_mesg.battery);
			switch (mesg[i].status_mesg.ext_type) {
			case CWIID_EXT_NONE:
				printf("none");
				break;
			case CWIID_EXT_NUNCHUK:
				printf("Nunchuk");
				break;
			case CWIID_EXT_CLASSIC:
				printf("Classic Controller");
				break;
			default:
				printf("Unknown Extension");
				break;
			}
			printf("\n");
			break;
		case CWIID_MESG_BTN:
			printf("Button Report: %.4X\n", mesg[i].btn_mesg.buttons);
			reset = (mesg[i].btn_mesg.buttons & CWIID_BTN_B);

			if(mesg[i].btn_mesg.buttons & CWIID_BTN_MINUS){
			  control_mode--;
			  if(control_mode<1) control_mode = 4;
			  printf("minus");
			}
			if(mesg[i].btn_mesg.buttons & CWIID_BTN_PLUS){
			  control_mode++;
			  if(control_mode>4) control_mode = 1;
			  printf("plus");
			}
			switch(control_mode){
			case 1:
			  set_bit(led_state, CWIID_LED1_ON);
			  unset_bit(led_state, CWIID_LED2_ON);
			  unset_bit(led_state, CWIID_LED3_ON);
			  unset_bit(led_state, CWIID_LED4_ON);
			  set_led_state(wiimote, led_state);
			  break;
			case 2:
			  unset_bit(led_state, CWIID_LED1_ON);
			  set_bit(led_state, CWIID_LED2_ON);
			  unset_bit(led_state, CWIID_LED3_ON);
			  unset_bit(led_state, CWIID_LED4_ON);
			  set_led_state(wiimote, led_state);
			  break;
			case 3:
			  unset_bit(led_state, CWIID_LED1_ON);
			  unset_bit(led_state, CWIID_LED2_ON);
			  set_bit(led_state, CWIID_LED3_ON);
			  unset_bit(led_state, CWIID_LED4_ON);
			  set_led_state(wiimote, led_state);
			  break;
			case 4:
			  unset_bit(led_state, CWIID_LED1_ON);
			  unset_bit(led_state, CWIID_LED3_ON);
			  unset_bit(led_state, CWIID_LED3_ON);
			  set_bit(led_state, CWIID_LED4_ON);
			  set_led_state(wiimote, led_state);
			  break;
			}
			break;
		case CWIID_MESG_ACC:
		        if(pos_tracking){ 
  		                postrack(mesg[i].acc_mesg.acc[CWIID_X],
					 mesg[i].acc_mesg.acc[CWIID_Y],
					 mesg[i].acc_mesg.acc[CWIID_Z],0);
				if(mesg[i].nunchuk_mesg.buttons){
				  postrack(mesg[i].acc_mesg.acc[CWIID_X],
					   mesg[i].acc_mesg.acc[CWIID_Y],
					   mesg[i].acc_mesg.acc[CWIID_Z],1);
				}
			}
			else
			printf("Acc Report: x=%d, y=%d, z=%d\n",
			       mesg[i].acc_mesg.acc[CWIID_X],
			       mesg[i].acc_mesg.acc[CWIID_Y],
			       mesg[i].acc_mesg.acc[CWIID_Z]);
			if(robot_controlling){
			  robot_ctr(mesg[i].acc_mesg.acc[CWIID_X],
				   mesg[i].acc_mesg.acc[CWIID_Y],
				   mesg[i].acc_mesg.acc[CWIID_Z],
				    X1,Y1,X2,Y2,reset);
			}
			break;
		case CWIID_MESG_IR:
			printf("IR Report: ");
			valid_source = 0;
			for (j = 0; j < CWIID_IR_SRC_COUNT; j++) {
				if (mesg[i].ir_mesg.src[j].valid) {
					valid_source ++;
					printf("(%d:%d,%d) ", j,mesg[i].ir_mesg.src[j].pos[CWIID_X],
					                   mesg[i].ir_mesg.src[j].pos[CWIID_Y]);
					if(j==0){
					  X1 = mesg[i].ir_mesg.src[j].pos[CWIID_X];
					  Y1 = mesg[i].ir_mesg.src[j].pos[CWIID_Y];
					}
					if(j==1){
					  X2 = mesg[i].ir_mesg.src[j].pos[CWIID_X];
					  Y2 = mesg[i].ir_mesg.src[j].pos[CWIID_Y];
					}
				}
			}
			// We assume that x1 is left of x2:
			if(X2<X1){
			  int tempx,tempy;
			  tempx = X2;
			  X2 = X1;
			  X1 = tempx;
			  tempy = Y2;
			  Y2 = Y1;
			  Y1 = tempy;
			}
			

			if (!valid_source) {
				printf("no sources detected");
				//X1 = Y1 = X2 = Y2 = -1000;
			}
			if(valid_source > 1){
			  have_ir = 1;
			}else{
			  have_ir = 0;
			}
			printf("\n");
			break;
		case CWIID_MESG_NUNCHUK:
		  
         		  printf("Nunchuk Report: btns=%.2X stick=(%d,%d) acc.x=%d acc.y=%d "
				 "acc.z=%d\n", mesg[i].nunchuk_mesg.buttons,
				 mesg[i].nunchuk_mesg.stick[CWIID_X],
				 mesg[i].nunchuk_mesg.stick[CWIID_Y],
				 mesg[i].nunchuk_mesg.acc[CWIID_X],
				 mesg[i].nunchuk_mesg.acc[CWIID_Y],
				 mesg[i].nunchuk_mesg.acc[CWIID_Z]);
			break;
		case CWIID_MESG_CLASSIC:
			printf("Classic Report: btns=%.4X l_stick=(%d,%d) r_stick=(%d,%d) "
			       "l=%d r=%d\n", mesg[i].classic_mesg.buttons,
			       mesg[i].classic_mesg.l_stick[CWIID_X],
			       mesg[i].classic_mesg.l_stick[CWIID_Y],
			       mesg[i].classic_mesg.r_stick[CWIID_X],
			       mesg[i].classic_mesg.r_stick[CWIID_Y],
			       mesg[i].classic_mesg.l, mesg[i].classic_mesg.r);
			break;
		case CWIID_MESG_ERROR:
			if (cwiid_close(wiimote)) {
				fprintf(stderr, "Error on wiimote disconnect\n");
				exit(-1);
			}
			exit(0);
			break;
		default:
			printf("Unknown Report");
			break;
		}
	}
}



void postrack(int x_acc, int y_acc, int z_acc, int reset){
  static int initialized = 0;
  static double xpos = 0;
  static double ypos = 0;
  static double zpos = 0;

  static double xvel = 0;
  static double yvel = 0;
  static double zvel = 0;

  //  static double xg = 0.0;
  //static double yg = 0.0;
  //static double zg = 0.0;

  static struct timeval t_old;
  //struct timeval t_new;
  //double diff_t;
  struct timezone tz;

  if(!initialized || reset){
    initialized = 1;
    gettimeofday(&t_old,&tz);
    if(reset){ printf("Resetting vel/pos data for nunchuk\n");
      xpos = 0;
      ypos = 0;
      zpos = 0;
      
      xvel = 0;
      yvel = 0;
      zvel = 0;
    }
  }else{

  }


}

void robot_ctr(int x_acc, int y_acc, int z_acc,
	       int ir_x1,int ir_y1,
	       int ir_x2,int ir_y2,
	       int reset){
  
  static int calibrated = 1;

  struct timeval time_now;
  

  // Raw acceleration in m/s^2
  double xa;
  double ya;
  double za;

  // Rotationally/gravitationally corrected values for acc
  double xa_c;
  double ya_c;
  double za_c;

  // values to use with parallell system not using camera:
  double xa_c2;
  double ya_c2;
  double za_c2;

  // positions
  double pos_x2;
  double pos_y2;
  double pos_z2;

  // velocities
  //double vel_x2;
  //double vel_y2;
  //double vel_z2;

  double yaw2, tilt2, pan2;


  // positions
  double pos_x;
  double pos_y;
  double pos_z;

  // velocities
  double vel_x;
  double vel_y;
  double vel_z;

  double vision_pos[3] = {0.0, 0.0, 0.0};

  static double pan;
  static double tilt;
  static double yaw;  

  // predicted endpoints of motion
  double end_x = 0;
  double end_y = 0;
  double end_z = 0;

  // estimated target task of motion
  double task_x = 0;
  double task_y = 0;
  double task_z = 0;

  double task_vel_x = 0;
  double task_vel_y = 0;
  double task_vel_z = 0;


  // motion along virtual fixtures
  double virt_x = 0;
  double virt_y = 0;
  double virt_z = 0;
  double virt_vel_x = 0;
  double virt_vel_y = 0;
  double virt_vel_z = 0;

  xa_c=xa = GRAVITY*((double)(x_acc - main_acc_cal.zero[0]))/
    ((double)(main_acc_cal.one[0] - main_acc_cal.zero[0]));
  ya_c=ya = GRAVITY*((double)(y_acc - main_acc_cal.zero[1]))/
    ((double)(main_acc_cal.one[1] - main_acc_cal.zero[1]));
  za_c=za = GRAVITY*((double)(z_acc - main_acc_cal.zero[2]))/
    ((double)(main_acc_cal.one[2] - main_acc_cal.zero[2]));

  za_c-=GRAVITY;

 
  if(!have_ir && calibrated){
    return;
  }
 

  if((calibrated = calibrate(xa,ya,za,ir_x1,ir_y1,ir_x2,ir_y2))){
    printf("Calibration return: %d\n",calibrated);
    return;
  }

  printf("Calibration return: %d\n",calibrated);

  /*
  printf("\n sending following IR pos to angle calculator (x1:%d y1:%d x2:%d y2:%d)\n",
	 ir_x1, // The x and y positions for
	 ir_y1, // IR marker 1 and 2. These should
	 ir_x2, // be in raw pixel values.
	 ir_y2);
  */

  if(have_ir){
  // correct for rotation/gravity
  correct_acc(xa, // raw acceleration data for
	      ya, // x, y, and z, corrected to
	      za, // m/s^2, but not for gravity
	      
	      ir_x1, // The x and y positions for
	      ir_y1, // IR marker 1 and 2. These should
	      ir_x2, // be in raw pixel values.
	      ir_y2,
	      
	      &xa_c, // corrected outputs
	      &ya_c,
	      &za_c,
	      
	      &pan,
	      &tilt,
	      &yaw);
  }else{
    
    // correct for rotation/gravity, not using cameras
    correct_acc_nocam(xa, // raw acceleration data for
		      ya, // x, y, and z, corrected to
		      za, // m/s^2, but not for gravity
		      
		      &xa_c, // corrected outputs
		      &ya_c,
		      &za_c,
		      
		      pan,
		      tilt,
		      yaw,
		      
		      &pan2,
		      &tilt2,
		      &yaw2);
  }
    acc2pos(xa_c, // inputs: acceleration in world frame
	    ya_c,
	    za_c,
	    
	    &pos_x, // outputs: position in world frame
	    &pos_y,
	    &pos_z,
	    
	    &vel_x, // outputs: velocity in world frame
	    &vel_y,
	    &vel_z,
	    
	    &end_x, // outputs: predicted end of motion
	    &end_y,
	    &end_z,

	    reset);

    /*
    acc2pos(xa_c2, // inputs: acceleration in world frame
	    ya_c2,
	    za_c2,
	    
	    &pos_x, // outputs: position in world frame
	    &pos_y,
	    &pos_z,
	    
	    &vel_x, // outputs: velocity in world frame
	    &vel_y,
	    &vel_z,
	    
	    reset);
    */

  /*
  acc2pos2(xa_c2, // inputs: acceleration in world frame
	  ya_c2,
	  za_c2,

	  &pos_x2, // outputs: position in world frame
	  &pos_y2,
	  &pos_z2,

	  &vel_x2, // outputs: velocity in world frame
	  &vel_y2,
	  &vel_z2,

	  reset);
  */


  //get_vision_trackerpos(vision_pos);
  
  gettimeofday(&time_now, NULL);

  // virtual fixtures:
  virt_y = 0.0;
  if(pos_z>0){
    virt_z = pos_z;
    virt_vel_z = vel_z;
  }else{
    virt_z = 0.0;
    virt_vel_z = 0.0;
  }
  if(fabs(pos_x/pos_z) > ((25.0/30.0)/2)){
    virt_x = virt_z*(25.0/30.0)*SIGN(pos_x);
    virt_vel_x = virt_vel_z*(25.0/30.0)*SIGN(pos_x);
  }else{
    virt_x = 0.0;
    virt_vel_x = 0.0;
  }



  // estimate target task
  if(end_z > 0.1){
    task_z = 0.3;
    if(fabs(end_x/end_z) > ((25.0/30.0)/2)){    
      task_x = 0.25*SIGN(end_x);
    }
    task_vel_x = 0.0;
    task_vel_z = 0.0;
  }else{
    task_vel_x = virt_vel_x;
    task_vel_z = virt_vel_x;

    task_z = virt_z;
    task_x = virt_x;
    task_y = 0.0;
  }

  

  printf("pos(%6.3f %6.3f %6.3f) vFix(%6.3f %6.3f %6.3f) \n end(%6.3f %6.3f %6.3f)  task(%6.3f %6.3f %6.3f) \n",
	 pos_x,pos_y,pos_z,
	 virt_x,virt_y,virt_z,
	 end_x,end_y,end_z,
	 task_x,task_y,task_z);


  // log angles:
  /*
  fprintf(logfile_ang,
	  "%6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f\n",
	  pan,tilt,yaw,
	  pan2,tilt2,yaw2,
	  xa,ya,za);
  */

  // log positions and other
  fprintf(logfile,"%d%06d %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f  %6.3f  %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %d %d %d %d %d %d %d\n",
	  (int)time_now.tv_sec,(int)time_now.tv_usec,
	  xa_c,ya_c,za_c,
	  pan,yaw,tilt,
	  //	  vision_pos[0],
	  // vision_pos[1],
	  //vision_pos[2],
	  xa, ya, za,
	  pos_x,pos_y,pos_z,
	  virt_x, virt_y, virt_z,
	  end_x, end_y, end_z,
	  ir_x1, ir_y1,
	  ir_x2, ir_y2,
	  control_mode,have_ir,reset);


  //printf("(a: %5.2f %5.2f %5.2f) (pan: %5.2f yaw:%5.2f, tilt:%5.2f)",xa_c,ya_c,za_c,pan,yaw,tilt);
  
  //printf("\n(pos: %5.2f %5.2f %5.2f) (vel: %5.2f %5.2f %5.2f)\n",
  //	 pos_x,pos_y,pos_z,vel_x,vel_y,vel_z);

  // send to robot:
  if(reset){
    robot_send(0,0,0,
	       0,0,0,
	       0,0,0);
  }else{
    switch(control_mode){
    case 0:  // Test mode
      robot_send(pos_x,pos_y,pos_z,
		 pan,tilt,yaw,
		 vel_x,vel_y,vel_z);
      break;
    case 1:  // Direct control
      robot_send(pos_x,pos_y,pos_z,
		 pan,tilt,yaw,
		 vel_x,vel_y,vel_z);
      break;
    case 2:  // Virtual fixtures
      robot_send(virt_x,virt_y,virt_z,
		 pan,tilt,yaw,
		 virt_vel_x,0,virt_vel_z);
      break;
    case 3:
      robot_send(task_x,task_y,task_z,
		 pan,tilt,yaw,
		 task_vel_x,0,task_vel_z);
      break;
    case 4:  // Test mode
      if(have_ir){
	robot_send(pos_x,pos_y,pos_z,
		   pan,tilt,yaw,
		   vel_x,vel_y,vel_z);
      }else{
	robot_send(pos_x,pos_y,pos_z,
		   pan2,tilt2,yaw2,
		   vel_x,vel_y,vel_z);
      }
      break;
    }
  }
}


int robot_connect(void){
  sock= socket(AF_INET, SOCK_DGRAM, 0);
  if (sock < 0) printf("socket error");
      
  robot.sin_family = AF_INET;
  hp = gethostbyname(ROBOT_IP_PROXY);
  
  bcopy((char *)hp->h_addr, 
	(char *)&robot.sin_addr,
	hp->h_length);
  robot.sin_port = htons(ROBOT_PROXY_UDP_PORT);
  length=sizeof(struct sockaddr_in);
  return 0;
}



int robot_send(double x,
	       double y,
	       double z,
	       double pan,
	       double tilt,
	       double yaw,
	       double xv,
	       double yv,
	       double zv){

  static double x_stat = 0.0;
  static double y_stat = 0.0;
  static double z_stat = 0.0;

  // limit commanded pos and vel:
  if(fabs(x)>0.3 || fabs(y)>0.12 || fabs(z)>0.3){
    x = x_stat;
    y = y_stat;
    z = z_stat;
    xv = 0.0;
    yv = 0.0;
    zv = 0.0;
  }else{
    x_stat = x;
    y_stat = y;
    z_stat = z;
  }

  /*
  if(fabs(x)>0.3) x = 0.3*SIGN(x);
  if(fabs(y)>0.1) x = 0.1*SIGN(y);
  if(fabs(z)>0.3) x = 0.3*SIGN(z);
  */
  if(fabs(xv)>3.0) xv = 3.0*SIGN(xv);
  if(fabs(yv)>3.0) yv = 3.0*SIGN(yv);
  if(fabs(zv)>3.0) zv = 3.0*SIGN(zv);
 
  if(z < -0.05) z = -0.05;
 
  // safety margin
  if((z>0.15) && (fabs(x)>0.15)){
    if(y<0){
      y = 0.0;
      yv = 0.0;
    }
  }

  // convert to milimeters and shift origo to center of workspace
  // also, we convert coordinate systems
  utdata.x = 420 + 1000*(-y);
  utdata.y = 0 + 1000*x;
  utdata.z = 300 + 1000*z;

  // conservative values for velocity
  // give more sluggish movements, but more stability
  utdata.xvel = 400*(-yv);
  utdata.yvel = 400*xv;
  utdata.zvel = 400*zv;

  if(control_mode==4){
    utdata.p=pan;
    utdata.t=((PI/2)+tilt);
    utdata.a=(yaw);
  }else{
    utdata.p=0;
    utdata.t=(PI/2);
    utdata.a=0;
  }  

  utdata.command = PKT_CMD_POSVEL;
  n=sendto(sock,(char*)&utdata,
	   sizeof(utdata),0,(struct sockaddr *)&robot,length);
  if (n < 0) printf("\n error sending data to server!\n");
  n = recvfrom(sock,(char*)&indata,sizeof(indata),0,(struct sockaddr *)&from, &length);
  return n;
}
