% Track a torso and right arm using particle filter.
% The human model is described in: Sidenbladh et al, ECCV 00
%                                  Sidenbladh and Black, ICCV 01  
% The Likelihood is described in: Sidenbladh and Black, ICCV 01
%                                 Sidenbladh and Black, IJCV to appear
% Parameters that can be changed:
%    SEQUENCE:      Name of image sequence
%    N:             Number of particles
%    LEN, DISPLACE: Model sizes
%    STATES_DEV:    Motion model noise
%    POS, ROT etc:  Start parameters
%
% Written by: Hedvig Sidenbladh, KTH, Sweden
% http://www.nada.kth.se/~hedvig/
% Date: March 2002

clear 

global SEQUENCE;
global CAMERA_CENTER;
global CAMERA_F;
global LEN;
global DISPLACE;
global EDGE_NORMOFFSET;
global EDGE_NORMSLOPE;
global EDGE_NBINS;
global EDGE_NUNITS;
global EDGE_RATIO;
global RIDGE_NORMOFFSET;
global RIDGE_NORMSLOPE;
global RIDGE_NBINS;
global RIDGE_NUNITS;
global RIDGE_RATIO;
global FLOW_NBINS;
global HIGH_LEVEL;
global FLOW_FG;
global FLOW_BG;
global STATES_DEV;
global N;

global im;
global oldIm;
global x_im;
global y_im;
global xx_im; 
global xy_im;
global yy_im;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%          CONSTANTS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Camera and sequence parameters

SEQUENCE = 'moving'; 

if (SEQUENCE == 'moving')
  CAMERA_CENTER = [120; 160];
  CAMERA_F = 380; 
  SEQUENCE_NAME = 'TestSequences/moving/';
  SEQUENCE_START  = 1;
  SEQUENCE_LENGTH = 51;
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Human model parameters

% A limb is a truncated cone with elliptical crossection. It has a length,
% an x and y radius at the proximal end, and an x and y radius at the distal 
% end.
%
% Length parameters are:
% [torso len, x proxrad, y proxrad, x distrad, y distrad
%  u arm       "
%  l arm       "                                        ]
LEN = [0.55 0.12 0.18 0.13 0.18
       0.36 0.06 0.06 0.05 0.05
       0.36 0.05 0.05 0.04 0.04];

% Displacement parameters (x,y,z) in parent limb coordinates for right upper 
% arm
DISPLACE = [0 -0.16 0.50];

% Motion model noise 
STATES_DEV = [0 0.002 0 ...
              0.003 0 0 ...
              0.005 0.005 0.005 0.01 ...
              0 0.006 0 ...
              0.009 0 0 ...
              0.015 0.015 0.015 0.03];

% Parameters are global pos and rot, shoulder angles, elbow angle, and their
% angular velocities.
% Start configuration set by hand = [POS, ROT, RA, DPOS, DROT, DRA]
if (SEQUENCE == 'moving')
  POS = [2.1 -0.29 -0.48];  
  ROT = [0 0 pi];
  RA = [-2.1 0.3 1.6 2.3]; 
  DPOS = [0 -0.01 0.002]; 
  DROT = [-0.001 0 0]; 
  DHEAD = [0 0]; 
  DLA = [0 -0.03 0 0]; 
  DRA = [0.026 0 0 -0.017]; 
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Image parameters

% Edge likelihood parameters
EDGE_NORMOFFSET = 45;              % offset of step
EDGE_NORMSLOPE = 0.05;             % slope of step
EDGE_NBINS = 40;                   % number of bins per unit edge contrast
EDGE_NUNITS = 2;

% Lookuptable RATIO for log(Pon/Poff) (grad, 1, limb)  
load LikelihoodDistributions/contrnormalizedEdge
EDGE_RATIO = RATIO;

% Ridge likelihood parameters
RIDGE_NORMOFFSET = 75;             % offset of step
RIDGE_NORMSLOPE = 0.1;             % slope of step
RIDGE_NBINS = 20;                  % number of bins per unit ridge contrast
RIDGE_NUNITS = 4;

% Lookuptable RATIO for log(Pon/Poff) (grad, 1, limb)  
load LikelihoodDistributions/contrnormalizedRidge
RIDGE_RATIO = RATIO;

% Flow likelihood parameters
FLOW_NBINS = 0.2;                  % number of bins per unit flow difference
HIGH_LEVEL = 3;                    % highest level in image pyramid

% Lookuptable FG for log(Pon) (grad, level, limb)  
%         och BG for log(Poff) (grad, level)
load LikelihoodDistributions/unnormalizedFlow
FLOW_FG = FG;
FLOW_BG = BG;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Condensation parameters

% Number of samples
N = 1000

% Diff between highest and lowest log likelihood
DIFF = 30;




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%           INIT   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

states = ones(N, 1)*[POS, ROT, RA, DPOS, DROT, DRA];

% Probabilities
Pstates = ones(N, 1)/N;             % Real distribution
smoothPstates = ones(N, 1)/N;       % Smoothed distribution (Isard ECCV 98)
logPstates = -ones(N, 1);           % Negative log distribution

% Movies
moviebest = moviein(SEQUENCE_LENGTH);
moviemean = moviein(SEQUENCE_LENGTH);

% Image data
imname = [SEQUENCE_NAME, 'frame', num2strPad(SEQUENCE_START, 4), '.tif'];
im = double(imread(imname, 'tiff'));
[height width depth] = size(im);




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%           TRACK
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

for image_no = 1:SEQUENCE_LENGTH

  % If we want to grab sequence images for a paper.
  % if ((image_no == 2) | (image_no == 12) | (image_no == 22) | ...
  %     (image_no == 32) | (image_no == 37) | (image_no == 42))
  %   disp(['Grab image as frame ', num2str(image_no - 2), ...
  %         ' and then press enter']);
  %   pause
  % end

  disp('**************************');
  image_no

  % Time
  tic
 

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % Get an image from the sequence

  disp('Get image')

  % Save old image
  if (image_no > 1)
    oldIm = im;
  end

  % Get new image
  frNum = image_no + SEQUENCE_START - 1;
  imname = [SEQUENCE_NAME, 'frame', num2strPad(frNum, 4), '.tif'];
  im = cell(HIGH_LEVEL, 1);
  im{1} = double(imread(imname, 'tiff'));
  if (depth == 3)
    im{1} = (im{1}(:, :, 1) + im{1}(:, :, 2) + im{1}(:, :, 3))/3;
  end

  % Create image pyramid 
  G_filter = [0.0030    0.0133    0.0219    0.0133    0.0030
              0.0133    0.0596    0.0983    0.0596    0.0133
              0.0219    0.0983    0.1621    0.0983    0.0219
              0.0133    0.0596    0.0983    0.0596    0.0133
              0.0030    0.0133    0.0219    0.0133    0.0030];

  for level = 2:HIGH_LEVEL
    % Pad
    temp_im = [im{level-1}(:, 1), im{level-1}(:, 1), im{level-1}, ...
               im{level-1}(:, width/2^(level-2)), im{level-1}(:, width/2^(level-2))];
    temp_im = [temp_im(1, :); temp_im(1, :); temp_im; ...
               temp_im(height/2^(level-2), :); temp_im(height/2^(level-2), :)];
    % Blur
    temp_im = conv2(temp_im, G_filter, 'valid');
    % Subsample
    temp_im = temp_im(1:2:height/2^(level-2), 1:2:width/2^(level-2));
 
    im{level} = temp_im;
  end


  % Compute image gradients (vertical(down) and horizontal(right))
  x_im = cell(HIGH_LEVEL, 1);
  y_im = cell(HIGH_LEVEL, 1);
  xx_im = cell(HIGH_LEVEL, 1);
  xy_im = cell(HIGH_LEVEL, 1);
  yy_im = cell(HIGH_LEVEL, 1);

  % First derivatives
  for level = 1:HIGH_LEVEL - 1
    h = floor(height/2^(level - 1));
    w = floor(width/2^(level - 1));

    % Pad
    pad_im = [im{level}(:, 1), im{level}, im{level}(:, w)];
    pad_im = [pad_im(1, :); pad_im; pad_im(h, :)];

    % Compute normalization constant
    max_win = max(max(max(max(max(max(max(max( ...
      pad_im(1:h, 1:w), ...
      pad_im(1:h, 2:(w + 1))), ...
      pad_im(1:h, 3:(w + 2))), ...
      pad_im(2:(h + 1), 1:w)), ...
      pad_im(2:(h + 1), 2:(w + 1))), ...
      pad_im(2:(h + 1), 3:(w + 2))), ...
      pad_im(3:(h + 2), 1:w)), ...
      pad_im(3:(h + 2), 2:(w + 1))), ...
      pad_im(3:(h + 2), 3:(w + 2)));
    min_win = min(min(min(min(min(min(min(min( ... 
      pad_im(1:h, 1:w), ... 
      pad_im(1:h, 2:(w + 1))), ...
      pad_im(1:h, 3:(w + 2))), ... 
      pad_im(2:(h + 1), 1:w)), ... 
      pad_im(2:(h + 1), 2:(w + 1))), ...
      pad_im(2:(h + 1), 3:(w + 2))), ...
      pad_im(3:(h + 2), 1:w)), ... 
      pad_im(3:(h + 2), 2:(w + 1))), ...
      pad_im(3:(h + 2), 3:(w + 2)));

    norm_const = (1 + tanh(EDGE_NORMSLOPE*((max_win - min_win) - ...
                           EDGE_NORMOFFSET)))./ ...
                 (2*max(0.01, (max_win - min_win)));	

    x_im{level} = norm_const.*(pad_im(3:(h + 2), 2:(w + 1)) - ...
                               pad_im(1:h, 2:(w + 1)));
    y_im{level} = norm_const.*(pad_im(2:(h + 1), 3:(w + 2)) - ...
                               pad_im(2:(h + 1), 1:w));
  end

  % Second derivatives
  for level = 1:HIGH_LEVEL 
    h = floor(height/2^(level - 1));
    w = floor(width/2^(level - 1));

    % Pad
    pad_im = [im{level}(:, 1), im{level}(:, 1), im{level}, ...
               im{level}(:, w), im{level}(:, w)];
    pad_im = [pad_im(1, :); pad_im(1, :); pad_im; ...
               pad_im(h, :); pad_im(h, :)];

    % Compute normalization constant
    max_win = max(max(max(max(max(max(max(max(max(max(max(max(max(max(max(max(max(max(max(max(max(max(max(max(  ...
      pad_im(1:h, 1:w), ... 
      pad_im(1:h, 2:(w + 1))), ...
      pad_im(1:h, 3:(w + 2))), ... 
      pad_im(1:h, 4:(w + 3))), ... 
      pad_im(1:h, 5:(w + 4))), ... 
      pad_im(2:(h + 1), 1:w)), ... 
      pad_im(2:(h + 1), 2:(w + 1))), ...
      pad_im(2:(h + 1), 3:(w + 2))), ... 
      pad_im(2:(h + 1), 4:(w + 3))), ... 
      pad_im(2:(h + 1), 5:(w + 4))), ... 
      pad_im(3:(h + 2), 1:w)), ... 
      pad_im(3:(h + 2), 2:(w + 1))), ...
      pad_im(3:(h + 2), 3:(w + 2))), ... 
      pad_im(3:(h + 2), 4:(w + 3))), ... 
      pad_im(3:(h + 2), 5:(w + 4))), ... 
      pad_im(4:(h + 3), 1:w)), ... 
      pad_im(4:(h + 3), 2:(w + 1))), ...
      pad_im(4:(h + 3), 3:(w + 2))), ... 
      pad_im(4:(h + 3), 4:(w + 3))), ... 
      pad_im(4:(h + 3), 5:(w + 4))), ... 
      pad_im(5:(h + 4), 1:w)), ... 
      pad_im(5:(h + 4), 2:(w + 1))), ...
      pad_im(5:(h + 4), 3:(w + 2))), ... 
      pad_im(5:(h + 4), 4:(w + 3))), ... 
      pad_im(5:(h + 4), 5:(w + 4)));
    min_win = min(min(min(min(min(min(min(min(min(min(min(min(min(min(min(min(min(min(min(min(min(min(min(min(  ...
      pad_im(1:h, 1:w), ... 
      pad_im(1:h, 2:(w + 1))), ...
      pad_im(1:h, 3:(w + 2))), ... 
      pad_im(1:h, 4:(w + 3))), ... 
      pad_im(1:h, 5:(w + 4))), ... 
      pad_im(2:(h + 1), 1:w)), ... 
      pad_im(2:(h + 1), 2:(w + 1))), ...
      pad_im(2:(h + 1), 3:(w + 2))), ... 
      pad_im(2:(h + 1), 4:(w + 3))), ... 
      pad_im(2:(h + 1), 5:(w + 4))), ... 
      pad_im(3:(h + 2), 1:w)), ... 
      pad_im(3:(h + 2), 2:(w + 1))), ...
      pad_im(3:(h + 2), 3:(w + 2))), ... 
      pad_im(3:(h + 2), 4:(w + 3))), ... 
      pad_im(3:(h + 2), 5:(w + 4))), ... 
      pad_im(4:(h + 3), 1:w)), ... 
      pad_im(4:(h + 3), 2:(w + 1))), ...
      pad_im(4:(h + 3), 3:(w + 2))), ... 
      pad_im(4:(h + 3), 4:(w + 3))), ... 
      pad_im(4:(h + 3), 5:(w + 4))), ... 
      pad_im(5:(h + 4), 1:w)), ... 
      pad_im(5:(h + 4), 2:(w + 1))), ...
      pad_im(5:(h + 4), 3:(w + 2))), ... 
      pad_im(5:(h + 4), 4:(w + 3))), ... 
      pad_im(5:(h + 4), 5:(w + 4)));

    norm_const = (1 + tanh(RIDGE_NORMSLOPE*((max_win - min_win) - ...
                                            RIDGE_NORMOFFSET)))./ ...
                   (2*max(0.01, (max_win - min_win)));

    xx_im{level} = norm_const.*(pad_im(1:h, 3:(w + 2)) + ...
                                pad_im(5:(h + 4), 3:(w + 2)) - ...
                                2*pad_im(3:(h + 2), 3:(w + 2)));
    xy_im{level} = norm_const.*(pad_im(2:(h + 1), 2:(w + 1)) + ...
                                pad_im(4:(h + 3), 4:(w + 3)) - ...
                                pad_im(2:(h + 1), 4:(w + 3)) - ...
                                pad_im(4:(h + 3), 2:(w + 1)));
    yy_im{level} = norm_const.*(pad_im(3:(h + 2), 1:w) + ...
                                pad_im(3:(h + 2), 5:(w + 4)) - ...
                                2*pad_im(3:(h + 2), 3:(w + 2)));
  end

  % Time
  toc


  figure(1)
  clf
  colormap(gray)
  subplot(2,2,1)
  imagesc(xx_im{2})
  title('xx derivative level 2')
  subplot(2,2,2)
  imagesc(xx_im{3})
  title('xx derirative level 3')
  subplot(2,2,3)
  imagesc(yy_im{2})
  title('yy derivative level 2')
  subplot(2,2,4)
  imagesc(yy_im{3})
  title('yy derivative level 3')

  % No tracking first iteration
  if (image_no > 1)

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Condensation with resampling (Isard ECCV 98)

    disp('Condensation step')

    % Time
    tic

    % Montecarlo sampling of N indices s_index from the old posterior distrbution 
    cum = cumsum(Pstates);
    cum = cum/cum(N);
    probes = rand(N, 1);
    sIndex = montecarlo(cum, probes, N);

    % Predict where the chosen samples should move in the new frame given 
    % the previous samples
    [states, priorStates] = propagate(states(sIndex, :, :), ...
                                      smoothPstates(sIndex, :, :), ...
                                      states, Pstates);

    % Time
    toc
    tic

    for sample = 1:N
      logPstates(sample) = getLikelihood(states(sample, :, :));
    end

    % Scale and normalize
    logPstates = logPstates - max(logPstates);
    Pstates = exp(logPstates);
    Pstates = Pstates/sum(Pstates);

    % Scale and normalize smoothed version
    diff = max(logPstates) - min(logPstates);
    smoothLogPstates = DIFF/diff*logPstates;
    smoothPstates = exp(smoothLogPstates);
    smoothPstates = priorStates.*smoothPstates/sum(priorStates.*smoothPstates);

    % Time
    toc
  end

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % Plot

  disp('Plot')


  if (N > 1)
  % Plot the Mbest best estimates
  Mbest = 10;

  figure(3)
  colormap(gray)
  clf
  imagesc(im{1})
  %title([num2str(Mbest), ' best states'])
  hold on
  s = [smoothPstates states(:, :, 1)];
  sort_s = sortrows(s);
  sort_s = sort_s(N:(-1):1, :);

  % Plot 10, 9, ..., 1 so that 1 goes on top
  for state = min(N, Mbest):(-1):1
    cylinder = state2cylinder(sort_s(state, 2:11), 21);

    if (state == 1)
      clr = 'r';
    else  if (state <= 10)
      clr = 'w';
    else
      clr = 'y';
    end, end

    for limb = 1:3
      plot(cylinder(2, 1:2, limb), cylinder(1, 1:2, limb), clr, 'linewidth', 3)
      plot(cylinder(2, 3:4, limb), cylinder(1, 3:4, limb), clr, 'linewidth', 3)
      plot(cylinder(2, 5:25, limb), cylinder(1, 5:25, limb), clr, 'linewidth', 3)
      plot(cylinder(2, 26:46, limb), cylinder(1, 26:46, limb), clr, 'linewidth', 3)
    end
  end 
  axis equal
  axis([1 320 1 240])
  axis off
  hold off

  moviebest(:, image_no) = getframe;
  end

  % Plot weighted mean
  figure(4)
  colormap(gray)
  clf
  imagesc(im{1})
  %title('Mean state')
  hold on
  if (N == 1)
    mean_s = (Pstates*ones(1, 20)).*states(:, :, 1);
  else
    mean_s = sum((Pstates*ones(1, 20)).*states(:, :, 1));
  end

  cylinder = state2cylinder(mean_s(1:10), 21);

  for limb = 1:3
    plot(cylinder(2, 1:2, limb), cylinder(1, 1:2, limb), 'w', 'linewidth', 3)
    plot(cylinder(2, 3:4, limb), cylinder(1, 3:4, limb), 'w', 'linewidth', 3)
    plot(cylinder(2, 5:25, limb), cylinder(1, 5:25, limb), 'w', 'linewidth', 3)
    plot(cylinder(2, 26:46, limb), cylinder(1, 26:46, limb), 'w', 'linewidth', 3)
  end

  axis equal
  axis([1 320 1 240])
  axis off
  hold off

  moviemean(:, image_no) = getframe;


  pause(0.0)

end

% Make mpeg movies
map = colormap;
mpgwrite(moviemean, gray, 'trackMean.mpg');
mpgwrite(moviebest, gray, 'trackBest.mpg');

% Save matlab movies
%save movies moviebest moviemean
%figure(5)
%clf
%movie(moviebest, 10, 10)






