function [logLikelihood] = getLikelihood(state);
% 
% [logLikelihood] = getLikelihood(state);
% Get likelihood that the two latest time steps in state explain the image change
%
% Written by: Hedvig Sidenbladh, KTH, Sweden
% http://www.nada.kth.se/~hedvig/
% Date: March 2002

global CAMERA_CENTER;
global CAMERA_F;
global LEN;
global DISPLACE;
global GLOB_POS;
global GLOB_ROT;
global EDGE_NORMOFFSET;
global EDGE_NORMSLOPE;
global EDGE_NBINS;
global EDGE_NUNITS;
global EDGE_RATIO;
global RIDGE_NORMOFFSET;
global RIDGE_NORMSLOPE;
global RIDGE_NBINS;
global RIDGE_NUNITS;
global RIDGE_RATIO;
global FLOW_NBINS;
global HIGH_LEVEL;
global FLOW_FG;
global FLOW_BG;

global im;
global oldIm;
global x_im;
global y_im;
global xx_im; 
global xy_im;
global yy_im;



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Approx with a flat surface:

% Compute four corner points of the surface in the image, on the form
%  |x x x x|     1 2
%  |y y y y|     4 3
corner = state2corner(state(1, :, 1));
oldCorner = state2corner(state(1, :, 2));

% Index of limbs, sorted according to increasing depth
depthOrder = limbDepth(state(1, :, 1));

% limbtype = torso, thigh, calf, upper arm, lower arm, head
limbType = [1, 4, 5];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LIKELIHOOD COMPUTATION
%

logLikelihood = 0;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Likelihood of FLOW: Sample a number of points on the limb. 

% Number of points on a limb
n = 15;

for limb = 1:3
  for level = 1:(HIGH_LEVEL - 1)

    i = im{level};
    oi = oldIm{level};
    [h w] = size(i); 

    for x = 1:n
      occluded = 1;
      while (occluded)
        [point, oldPoint] = getCorrespondingPoints(corner(:, :, limb), ...
     	                                             oldCorner(:, :, limb));

        % Check occlusion from nearer limbs
        l = 1;
        occluded = 0;
        while ((depthOrder(l) ~= limb) & (~(occluded)))
	       occluded = contains(corner(:, :, depthOrder(l)), point);
          l = l + 1;
        end
      end

      point = max([1; 1], min([h; w], (point - [1;1])/2^(level - 1) + [1;1]));
      oldPoint = max([1; 1], min([h; w], ...
                     (oldPoint - [1;1])/2^(level - 1) + [1;1]));

      % Linear interpolation of pixel values
      eps =  point - floor(point);
      newVal = (1-eps(1))* ...
                ((1-eps(2))*i(floor(point(1)), floor(point(2))) + ...
	              eps(2)*i(floor(point(1)), ceil(point(2)))) + ...
                eps(1)* ...
                ((1-eps(2))*i(ceil(point(1)), floor(point(2))) + ...
                 eps(2)*i(ceil(point(1)), ceil(point(2))));
      oldBgVal = (1-eps(1))* ...
                 ((1-eps(2))*oi(floor(point(1)), floor(point(2))) + ...
                  eps(2)*oi(floor(point(1)), ceil(point(2)))) + ...
                 eps(1)* ...
                 ((1-eps(2))*oi(ceil(point(1)), floor(point(2))) + ...
                  eps(2)*oi(ceil(point(1)), ceil(point(2))));

      eps = oldPoint - floor(oldPoint);
      oldFgVal = (1-eps(1))* ...
                 ((1-eps(2))*oi(floor(oldPoint(1)), floor(oldPoint(2))) + ...
                  eps(2)*oi(floor(oldPoint(1)), ceil(oldPoint(2)))) + ...
                 eps(1)* ...
                 ((1-eps(2))*oi(ceil(oldPoint(1)), floor(oldPoint(2))) + ...
                  eps(2)*oi(ceil(oldPoint(1)), ceil(oldPoint(2))));

      fgVal = newVal - oldFgVal;
      bgVal = newVal - oldBgVal;
      logLikelihood = logLikelihood + ...
                      (FLOW_FG(max(1, min(2*255*FLOW_NBINS, ...
                       255*FLOW_NBINS + floor(fgVal*FLOW_NBINS) + 1)), ...
                       level, limbType(limb)) - ...
                       FLOW_BG(max(1, min(2*255*FLOW_NBINS, ...
                       255*FLOW_NBINS + floor(bgVal*FLOW_NBINS) + 1)), ...
                       level));
  
    end
  end
end



%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Likelihood of EDGE: Sample a number of points on the edges of the limb

% Number of points on a limb
n = 50;

for limb = 2:3
  for level = 1:(HIGH_LEVEL - 1)

    xi = x_im{level};
    yi = y_im{level};
    [h w] = size(xi); 
    
    for x = 1:n
      occluded = 1;
      while (occluded)

        % Sample a point on the edge and compute expected angle
        rnd = -1 + 2*rand;
        if (rnd < 0)	
          point = -rnd*corner(:, 2, limb) + (1 + rnd)*corner(:, 3, limb);
          a = atan2(corner(2, 2, limb) - corner(2, 3, limb), ...
  	                 corner(1, 2, limb) - corner(1, 3, limb));
        else
          point = rnd*corner(:, 1, limb) + (1 - rnd)*corner(:, 4, limb);
          a = atan2(corner(2, 1, limb) - corner(2, 4, limb), ...
	                 corner(1, 1, limb) - corner(1, 4, limb));
        end

        % Check occlusion from nearer limbs
        l = 1;
        occluded = 0;
        while ((depthOrder(l) ~= limb) & (~(occluded)))
	       occluded = contains(corner(:, :, depthOrder(l)), point);
          l = l + 1;
        end
 
      end

      point = max([1; 1], min([h; w], (point - [1;1])/2^(level - 1) + [1;1]));

      % Linear interpolation of pixel values
      eps =  point - floor(point);
      xVal = (1-eps(1))* ...
             ((1-eps(2))*xi(floor(point(1)), floor(point(2))) + ...
	           eps(2)*xi(floor(point(1)), ceil(point(2)))) + ...
             eps(1)* ...
             ((1-eps(2))*xi(ceil(point(1)), floor(point(2))) + ...
	           eps(2)*xi(ceil(point(1)), ceil(point(2))));
    
      yVal = (1-eps(1))* ...
             ((1-eps(2))*yi(floor(point(1)), floor(point(2))) + ...
	           eps(2)*yi(floor(point(1)), ceil(point(2)))) + ...
             eps(1)* ...
             ((1-eps(2))*yi(ceil(point(1)), floor(point(2))) + ...
	           eps(2)*yi(ceil(point(1)), ceil(point(2))));

      val = yVal*cos(a) - xVal*sin(a);
      logLikelihood = logLikelihood + ...
                      EDGE_RATIO(max(1, min(2*EDGE_NUNITS*EDGE_NBINS, ...
                                            EDGE_NUNITS*EDGE_NBINS + ...
                                            floor(val*EDGE_NBINS) + 1)), ...
                                 1, limbType(limb));
    end
  end
end



%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Likelihood of RIDGE: Sample a number of points on the fg of the limb

% Number of points on a limb
n = 100;

% Compute angles and level
for limb = 2:3
  ends(:, :, limb) = [(corner(:, 1, limb) + corner(:, 2, limb))/2 ...
                   (corner(:, 4, limb) + corner(:, 3, limb))/2];
  a(limb) = atan2(ends(2, 1, limb) - ends(2, 2, limb), ...
	            ends(1, 1, limb) - ends(1, 2, limb));
  mindia(limb) = min(norm(corner(:, 1, limb) - corner(:, 2, limb)), ...
                  norm(corner(:, 4, limb) - corner(:, 3, limb)));
end
scale = (mindia - 6)/0.32;
level = min(HIGH_LEVEL*ones(1, 3), ...
            ones(1, 3)+(scale > 0.5) + (scale > 3) + (scale > 13) + (scale > 53));

for limb = 2:3

  xxi = xx_im{level(limb)};
  xyi = xy_im{level(limb)};
  yyi = yy_im{level(limb)};
  [h w] = size(xxi);

  for x = 1:n
    occluded = 1;
    while (occluded)
      % Sample a point on the limb
      rrand = rand;
      lrand = rand;
      p14 = lrand*corner(:, 1, limb) + (1 - lrand)*corner(:, 4, limb);
      p23 = lrand*corner(:, 2, limb) + (1 - lrand)*corner(:, 3, limb);
      point = rrand*p14 + (1 - rrand)*p23;

      % Check occlusion from nearer limbs
      l = 1;
      occluded = 0;
      while ((depthOrder(l) ~= limb) & (~(occluded)))
	     occluded = contains(corner(:, :, depthOrder(l)), point);
        l = l + 1;
      end
    end

    point = max([1; 1], min([h; w], (point - [1;1])/2^(level(limb) - 1) + [1;1]));
        
    % Linear interpolation of pixel values
    eps = point - floor(point);
    xxVal = (1-eps(1))* ...
            ((1-eps(2))*xxi(floor(point(1)), floor(point(2))) + ... 
    	      eps(2)*xxi(floor(point(1)), ceil(point(2)))) + ...
            eps(1)* ...
            ((1-eps(2))*xxi(ceil(point(1)), floor(point(2))) + ...
    	      eps(2)*xxi(ceil(point(1)), ceil(point(2))));
    xyVal = (1-eps(1))* ...
            ((1-eps(2))*xyi(floor(point(1)), floor(point(2))) + ...
    	      eps(2)*xyi(floor(point(1)), ceil(point(2)))) + ...
            eps(1)* ...
            ((1-eps(2))*xyi(ceil(point(1)), floor(point(2))) + ...
	          eps(2)*xyi(ceil(point(1)), ceil(point(2))));
    yyVal = (1-eps(1))* ...
            ((1-eps(2))*yyi(floor(point(1)), floor(point(2))) + ...
 	          eps(2)*yyi(floor(point(1)), ceil(point(2)))) + ...
            eps(1)* ...
            ((1-eps(2))*yyi(ceil(point(1)), floor(point(2))) + ...
             eps(2)*yyi(ceil(point(1)), ceil(point(2))));

    val = abs(xxVal*sin(a(limb))^2 + yyVal*cos(a(limb))^2 - ...
              2*xyVal*sin(a(limb))*cos(a(limb))) - ...
          abs(xxVal*cos(a(limb))^2 + yyVal*sin(a(limb))^2 + ...
              2*xyVal*sin(a(limb))*cos(a(limb)));

    logLikelihood = logLikelihood + ...
                    RIDGE_RATIO(max(1, min(2*RIDGE_NUNITS*RIDGE_NBINS, ...
                                           RIDGE_NUNITS*RIDGE_NBINS + ...
                                           floor(val*RIDGE_NBINS) + 1)), ...
                                1, limbType(limb));
  end
end

